/*
 * Copyright (c) 2010 ICM Uniwersytet Warszawski All rights reserved. See LICENCE file for licensing
 * information.
 * 
 * Author: T. Rękawek <newton@mat.uni.torun.pl>
 */

package de.fzj.unicore.uas.impl.dsms;

import java.util.Calendar;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.unigrids.x2006.x04.services.sms.ChangePermissionsDocument;
import org.unigrids.x2006.x04.services.sms.ChangePermissionsResponseDocument;
import org.unigrids.x2006.x04.services.sms.CopyDocument;
import org.unigrids.x2006.x04.services.sms.CopyResponseDocument;
import org.unigrids.x2006.x04.services.sms.CreateDirectoryDocument;
import org.unigrids.x2006.x04.services.sms.CreateDirectoryResponseDocument;
import org.unigrids.x2006.x04.services.sms.DeleteDocument;
import org.unigrids.x2006.x04.services.sms.DeleteResponseDocument;
import org.unigrids.x2006.x04.services.sms.ExportFileDocument;
import org.unigrids.x2006.x04.services.sms.ExportFileResponseDocument;
import org.unigrids.x2006.x04.services.sms.FindDocument;
import org.unigrids.x2006.x04.services.sms.FindResponseDocument;
import org.unigrids.x2006.x04.services.sms.ImportFileDocument;
import org.unigrids.x2006.x04.services.sms.ImportFileResponseDocument;
import org.unigrids.x2006.x04.services.sms.ListDirectoryDocument;
import org.unigrids.x2006.x04.services.sms.ListDirectoryResponseDocument;
import org.unigrids.x2006.x04.services.sms.ListPropertiesDocument;
import org.unigrids.x2006.x04.services.sms.ListPropertiesResponseDocument;
import org.unigrids.x2006.x04.services.sms.ReceiveFileDocument;
import org.unigrids.x2006.x04.services.sms.ReceiveFileResponseDocument;
import org.unigrids.x2006.x04.services.sms.RenameDocument;
import org.unigrids.x2006.x04.services.sms.RenameResponseDocument;
import org.unigrids.x2006.x04.services.sms.SendFileDocument;
import org.unigrids.x2006.x04.services.sms.SendFileResponseDocument;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import pl.edu.icm.dsms.catalogue.CatalogueClient;
import pl.edu.icm.dsms.catalogue.CatalogueClient.LdasAndEprs;
import pl.edu.icm.dsms.catalogue.CatalogueUtils;
import pl.edu.icm.dsms.catalogue.model.Lda;
import pl.edu.icm.x2010.x10.services.catalogue.SharedStoragePropertiesType;
import pl.edu.icm.x2010.x10.services.catalogue.CreateSmsResponseDocument.CreateSmsResponse;
import de.fzj.unicore.uas.UAS;
import de.fzj.unicore.uas.UASProperties;
import de.fzj.unicore.uas.client.StorageClient;
import de.fzj.unicore.uas.impl.dsms.meta.DSMSMetadataManagement;
import de.fzj.unicore.uas.impl.dsms.meta.DSMSMetadataManager;
import de.fzj.unicore.uas.impl.dsms.operation.ChangePermissions;
import de.fzj.unicore.uas.impl.dsms.operation.Copy;
import de.fzj.unicore.uas.impl.dsms.operation.CreateDirectory;
import de.fzj.unicore.uas.impl.dsms.operation.Delete;
import de.fzj.unicore.uas.impl.dsms.operation.ExportFile;
import de.fzj.unicore.uas.impl.dsms.operation.Find;
import de.fzj.unicore.uas.impl.dsms.operation.ImportFile;
import de.fzj.unicore.uas.impl.dsms.operation.ListDirectory;
import de.fzj.unicore.uas.impl.dsms.operation.ListProperties;
import de.fzj.unicore.uas.impl.dsms.operation.ReceiveFile;
import de.fzj.unicore.uas.impl.dsms.operation.Rename;
import de.fzj.unicore.uas.impl.dsms.operation.SendFile;
import de.fzj.unicore.uas.impl.dsms.operation.helper.CatalogueClientUtils;
import de.fzj.unicore.uas.impl.dsms.operation.helper.ClientHelper;
import de.fzj.unicore.uas.impl.sms.SMSBaseImpl;
import de.fzj.unicore.uas.impl.sms.StorageDescription;
import de.fzj.unicore.uas.metadata.BaseMetadataManagementImpl;
import de.fzj.unicore.uas.metadata.MetadataManager;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.wsrflite.Home;
import de.fzj.unicore.wsrflite.messaging.ResourceDeletedMessage;
import de.fzj.unicore.wsrflite.persistence.Persist;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import eu.unicore.util.Log;
import eu.unicore.util.httpclient.IClientConfiguration;

/**
 * Distributed Storage Management Service
 * 
 * @author newton
 * 
 */
public class DSMSImpl extends SMSBaseImpl {
	private static final Logger logger = Log.getLogger(LogUtil.SERVICES, DSMSImpl.class);

	public static final String DSMS_CONFIG = "dSmsConfig";

	private static DSMSCatalogueCache CACHE;

	static {
		initCache();
	}

	@Persist
	private String dSmsId;

	@Persist
	private String dSmsRoot;

	@Persist
	private StorageDescription configuration;

	@Persist
	private EndpointReferenceType catalogueEpr;

	@Override
	public void initialise(String name, Map<String, Object> initobjs) throws Exception {
		super.initialise(name, initobjs);
		if (storageFactoryID == null) {
			configuration = (StorageDescription) initobjs.get(DSMS_CONFIG);
		} else {
			configuration = storageDescription;
		}
		if (configuration == null) {
			throw new IllegalStateException("Can not create DSMS without specifying its properties.");
		}
		workdir = getStorageRoot();

		catalogueEpr = CatalogueClientUtils.createCatalogueEpr(getKernel(), configuration
				.getAdditionalProperties().get(DSMSProperties.DSMS_CATALOGUE));
		if (catalogueEpr == null) {
			throw new Exception("Can't get catalogue");
		}
		properties.put(RPFileSystem, new DSMSFileSystemRP(this, configuration.getName(), catalogueEpr));
		CatalogueClient cat = CatalogueClientUtils.getCatalogueClient(getKernel().getClientConfiguration(),
				catalogueEpr);
		if (storageFactoryID == null) {
			SharedStoragePropertiesType catProps = cat.getSharedStorageProperties();
			this.dSmsRoot = catProps.getGlobalDsmsRoot();
			this.dSmsId = catProps.getGlobalDsmsId();
		} else {
			CreateSmsResponse response = cat.createSms(getUniqueID(), getOwner());
			this.dSmsRoot = response.getDSmsRoot();
			this.dSmsId = response.getDSmsId();
		}
		logger.info("Initialised dSMS " + (dSmsId == null ? "" : dSmsId));
	}

	@Override
	public void destroy() {
		if (storageFactoryID != null) {
			try {
				ResourceDeletedMessage m = new ResourceDeletedMessage("deleted:" + getUniqueID());
				m.setDeletedResource(getUniqueID());
				m.setServiceName(getServiceName());
				getKernel().getMessaging().getChannel(storageFactoryID).publish(m);
			} catch (Exception e) {
				logger.error("Could not send message to StorageFactory", e);
			}
		}
		try {
			ClientHelper clientHelper = new ClientHelper(getContext());
			if (configuration.isCleanupOnDestroy()) {
				LdasAndEprs lae = clientHelper.getCatalogueClient().removeSms(dSmsId);
				for (EndpointReferenceType epr : lae.getEprs()) {
					IClientConfiguration sec = CatalogueUtils.getTrustDelegation(kernel, getClient());
					StorageClient storageClient = new StorageClient(epr, sec);
					storageClient.destroy();
				}
				for (Lda lda : lae.getLdas()) {
					StorageClient storageClient = clientHelper.getStorageClient(lda);
					storageClient.delete(lda.getPhysicalName());
				}
			}
		} catch (Exception e) {
			logger.error("Could not destroy dSMS", e);
		}
		super.destroy();
		logger.info("Destroyed dSMS " + (dSmsId == null ? "" : dSmsId));
	}

	public ChangePermissionsResponseDocument ChangePermissions(ChangePermissionsDocument in) throws BaseFault {
		return (ChangePermissionsResponseDocument) new ChangePermissions(getContext()).perform(in);
	}

	public CopyResponseDocument Copy(CopyDocument in) throws BaseFault {
		return (CopyResponseDocument) new Copy(getContext(), getProtocol()).perform(in);
	}

	public CreateDirectoryResponseDocument CreateDirectory(CreateDirectoryDocument in) throws BaseFault {
		return (CreateDirectoryResponseDocument) new CreateDirectory(getContext()).perform(in);
	}

	public DeleteResponseDocument Delete(DeleteDocument in) throws BaseFault {
		return (DeleteResponseDocument) new Delete(getContext()).perform(in);
	}

	public ExportFileResponseDocument ExportFile(ExportFileDocument in) throws BaseFault {
		return (ExportFileResponseDocument) new ExportFile(getContext()).perform(in);
	}

	public FindResponseDocument Find(FindDocument in) throws BaseFault {
		return (FindResponseDocument) new Find(getContext()).perform(in);
	}

	public ImportFileResponseDocument ImportFile(ImportFileDocument in) throws BaseFault {
		return (ImportFileResponseDocument) new ImportFile(getContext()).perform(in);
	}

	public ListDirectoryResponseDocument ListDirectory(ListDirectoryDocument in) throws BaseFault {
		return (ListDirectoryResponseDocument) new ListDirectory(getContext()).perform(in);
	}

	public ListPropertiesResponseDocument ListProperties(ListPropertiesDocument in) throws BaseFault {
		return (ListPropertiesResponseDocument) new ListProperties(getContext()).perform(in);
	}

	public ReceiveFileResponseDocument ReceiveFile(ReceiveFileDocument in) throws BaseFault {
		return (ReceiveFileResponseDocument) new ReceiveFile(getContext()).perform(in);
	}

	public RenameResponseDocument Rename(RenameDocument in) throws BaseFault {
		return (RenameResponseDocument) new Rename(getContext()).perform(in);
	}

	public SendFileResponseDocument SendFile(SendFileDocument in) throws BaseFault {
		return (SendFileResponseDocument) new SendFile(getContext()).perform(in);
	}

	@Override
	protected String getStorageRoot() {
		return "/tmp";
	}

	/**
	 * get the {@link MetadataManager}<br/>
	 * This implementation will return a dSMS metadata manager
	 * 
	 * @return {@link MetadataManager} or <code>null</code> if disabled
	 */
	@Override
	protected MetadataManager getMetadataManager() throws Exception {
		if (disableMetadata) {
			return null;
		}
		return new DSMSMetadataManager(getContext());
	}

	@Override
	protected String getMetadataManagementImplClassName() {
		return DSMSMetadataManagement.class.getName();
	}

	@Override
	protected String createMetadataServiceInstance() {
		Home mdHome = kernel.getHome(UAS.META);
		if (mdHome != null) {
			try {
				Map<String, Object> opts = new HashMap<String, Object>();
				opts.put(BaseMetadataManagementImpl.INIT_SMS_ID, getUniqueID());
				Calendar tt = Calendar.getInstance();
				tt.add(Calendar.YEAR, 10);
				opts.put(INIT_INITIAL_TERMINATION_TIME, tt);
				// set a "guessable" UUID for the metadata service
				opts.put(INIT_UNIQUE_ID, getUniqueID() + "_metadata");
				opts.put(INIT_PARENT_NODE, getNode());
				opts.put(BaseMetadataManagementImpl.INIT_CLASS_NAME, DSMSMetadataManagement.class.getName());
				return mdHome.createWSRFServiceInstance(opts);
			} catch (Exception ex) {
				Log.logException("", ex, logger);
			}
		}
		return null;
	}

	public static void initCache() {
		CACHE = new DSMSCatalogueCache();
	}

	public OperationContext getContext() throws BaseFault {
		OperationContext context = new OperationContext();
		context.setCache(CACHE);
		context.setClient(getClient());
		context.setCreateRoot(this.storageFactoryID != null);
		context.setdSmsId(dSmsId);
		context.setdSmsRoot(dSmsRoot);
		context.setKernel(getKernel());

		String sync1 = configuration.getAdditionalProperties().get(DSMSProperties.DSMS_ENABLE_SYNC_1);
		context.setSync1(sync1 == null ? false : Boolean.parseBoolean(sync1));
		String sync2 = configuration.getAdditionalProperties().get(DSMSProperties.DSMS_ENABLE_SYNC_2);
		context.setSync2(sync2 == null ? false : Boolean.parseBoolean(sync2));
		context.setCatalogueEpr(catalogueEpr);
		return context;
	}

	private String getProtocol() {
		String protocols = configuration.getProtocols();
		if (protocols == null) {
			protocols = uasProperties.getValue(UASProperties.SMS_PROTOCOLS);
		}
		return StringUtils.split(protocols, ' ')[0];
	}
}