/*
 * Copyright (c) 2012 ICM Uniwersytet Warszawski All rights reserved. See LICENCE.txt file for licensing
 * information.
 */
package de.fzj.unicore.uas.impl.dsms;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.log4j.Logger;

import de.fzj.unicore.uas.UASProperties;
import de.fzj.unicore.uas.impl.sms.StorageDescription;
import de.fzj.unicore.uas.impl.sms.StorageManagementHomeImpl.StorageTypes;
import eu.unicore.util.Log;
import eu.unicore.util.configuration.ConfigurationException;
import eu.unicore.util.configuration.DocumentationReferenceMeta;
import eu.unicore.util.configuration.DocumentationReferencePrefix;
import eu.unicore.util.configuration.PropertiesHelper;
import eu.unicore.util.configuration.PropertyChangeListener;
import eu.unicore.util.configuration.PropertyMD;

/**
 * DSMS configuration
 * 
 * @author K. Benedyczak
 */
public class DSMSGeneralProperties extends PropertiesHelper {
	private static final Logger log = Log.getLogger(Log.SERVICES, DSMSGeneralProperties.class);

	public static final String DSMS_NAMESPACE = "dsms.";

	@DocumentationReferencePrefix
	public static final String PREFIX = UASProperties.PREFIX + DSMS_NAMESPACE;

	public static final String DSMS_ENABLED = "enabled";

	public static final String DSMS_STORAGE_PREFIX = "storage.";

	private Collection<StorageDescription> dsmsStorages;

	@DocumentationReferenceMeta
	public static final Map<String, PropertyMD> META = new HashMap<String, PropertyMD>();
	static {
		META.put(
				DSMS_ENABLED,
				new PropertyMD((String) null)
						.setDescription("Space separated list of names of enabled DistributedSMS storages. If this property is left undefined then all defined DSMSes are enabled. If this property value is empty then all are disabled."));
		META.put(
				DSMS_STORAGE_PREFIX,
				new PropertyMD()
						.setCanHaveSubkeys()
						.setDescription(
								"Properties with this prefix are used to configure DistributedSMS storages. See documentation of DistributedSMS for details."));
	}

	public DSMSGeneralProperties(Properties properties) throws ConfigurationException {
		super(PREFIX, properties, META, log);
		dsmsStorages = parseDSMSStorages();
		addPropertyChangeListener(new PropertyChangeListener() {
			private final String[] PROPS = { DSMS_STORAGE_PREFIX };

			@Override
			public void propertyChanged(String propertyKey) {
				updateStorages();
			}

			@Override
			public String[] getInterestingProperties() {
				return PROPS;
			}
		});
	}

	@Override
	protected void checkConstraints() throws ConfigurationException {
		super.checkConstraints();
		// this could be optimized little bit as creation of StorageDescriptions is not required for
		// parsing...
		parseDSMSStorages();
	}

	public Collection<StorageDescription> getDSMSStorages() {
		return dsmsStorages;
	}

	private void updateStorages() {
		for (StorageDescription storage : dsmsStorages) {
			DSMSProperties smsProps = new DSMSProperties(PREFIX + DSMS_STORAGE_PREFIX + storage.getName()
					+ ".", properties);
			Map<String, String> extra = new HashMap<String, String>();
			extra.put(DSMSProperties.DSMS_ENABLE_SYNC_1, smsProps.getValue(DSMSProperties.DSMS_ENABLE_SYNC_1));
			extra.put(DSMSProperties.DSMS_ENABLE_SYNC_2, smsProps.getValue(DSMSProperties.DSMS_ENABLE_SYNC_2));
			storage.update(smsProps.getValue(DSMSProperties.PROTOCOLS), false, false, null,
					smsProps.getValue(DSMSProperties.DESCRIPTION), extra);
		}
	}

	/**
	 * @return a list of DSMSes configurations
	 */
	private Collection<StorageDescription> parseDSMSStorages() {
		String enabledV = getValue(DSMS_ENABLED);
		Set<String> storageIds = UASProperties.getStorageIds(enabledV, DSMS_NAMESPACE + DSMS_STORAGE_PREFIX,
				properties);

		List<StorageDescription> ret = new ArrayList<StorageDescription>();
		for (String id : storageIds) {
			DSMSProperties smsProps = new DSMSProperties(PREFIX + DSMS_STORAGE_PREFIX + id + ".", properties);
			Map<String, String> extra = new HashMap<String, String>();
			// we take care to have all custom options in extra properties, so it is also possible
			// to configure DSMS via normal add-on storage facility and storage factories.
			extra.put(DSMSProperties.DSMS_ENABLE_SYNC_1, smsProps.getValue(DSMSProperties.DSMS_ENABLE_SYNC_1));
			extra.put(DSMSProperties.DSMS_ENABLE_SYNC_2, smsProps.getValue(DSMSProperties.DSMS_ENABLE_SYNC_2));
			StorageDescription sd = new StorageDescription(id, smsProps.getValue(DSMSProperties.NAME), null,
					StorageTypes.CUSTOM, DSMSImpl.class, smsProps.getValue(DSMSProperties.PROTOCOLS), false,
					false, false, null, smsProps.getValue(DSMSProperties.DESCRIPTION), extra);
			ret.add(sd);
		}
		return ret;
	}
}
