package de.fzj.unicore.uas.testsuite;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.List;

import org.junit.Test;
import org.oasisOpen.docs.wsrf.sg2.AddDocument;
import org.oasisOpen.docs.wsrf.sg2.EntryType;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.Base;
import de.fzj.unicore.uas.client.StorageClient;
import de.fzj.unicore.wsrflite.ContainerProperties;
import de.fzj.unicore.wsrflite.Home;
import de.fzj.unicore.wsrflite.impl.DefaultHome;
import de.fzj.unicore.wsrflite.xmlbeans.WSUtilities;
import de.fzj.unicore.wsrflite.xmlbeans.client.BaseWSRFClient;
import de.fzj.unicore.wsrflite.xmlbeans.client.ExternalRegistryClient;
import de.fzj.unicore.wsrflite.xmlbeans.client.RegistryClient;
import de.fzj.unicore.wsrflite.xmlbeans.registry.RegistryEntryUpdater;
import de.fzj.unicore.wsrflite.xmlbeans.sg.Registry;
import de.fzj.unicore.wsrflite.xmlbeans.sg.ServiceGroupEntry;
import eu.unicore.bugsreporter.annotation.FunctionalTest;
import eu.unicore.bugsreporter.annotation.RegressionTest;

/**
 * tests for the registry entry update/removal mechanism
 */
public class TestRegistryEntryUpdater extends Base {

	String url;
	EndpointReferenceType registryEpr;

	@FunctionalTest(id="TestEntryUpdater", description="Tests the registry entry updater mechanism")
	@Test
	public void testEntryUpdater()throws Exception{
		url=kernel.getContainerProperties().getValue(ContainerProperties.WSRF_BASEURL);
		registryEpr=EndpointReferenceType.Factory.newInstance();
		registryEpr.addNewAddress().setStringValue(url+"/"+Registry.REGISTRY_SERVICE+"?res=default_registry");
		RegistryClient reg=new RegistryClient(registryEpr,kernel.getClientConfiguration());
		//list entries and figure out storage entry EPR and registry entry EPR
		List<EntryType>entries=reg.listEntries();
		assertNotNull(entries);
		assertTrue(entries.size()>0);
		EndpointReferenceType storageEPR=null;
		EndpointReferenceType entryEPR=null;
		String entryID=null;
		for(EntryType e: entries){
			EndpointReferenceType member=e.getMemberServiceEPR();
			String memberURL=member.getAddress().getStringValue();
			if(memberURL.endsWith("default_storage")){
				storageEPR=member;
				entryEPR=e.getServiceGroupEntryEPR();
				entryID=WSUtilities.extractResourceID(entryEPR);
				break;
			}
		}
		assertNotNull(storageEPR);
		assertNotNull(entryEPR);
		assertNotNull(entryID);
		StorageClient sms=new StorageClient(storageEPR, kernel.getClientConfiguration());
		BaseWSRFClient entry=new BaseWSRFClient(entryEPR, kernel.getClientConfiguration());
		Home registryHome=kernel.getHome(ServiceGroupEntry.SERVICENAME);
		assertNotNull(registryHome);
		RegistryEntryUpdater upd=new RegistryEntryUpdater();
		boolean expired=upd.check(registryHome, entryID);
		assertFalse(expired);
	
		//check if process() returns true (since the service is up)
		boolean up=upd.process(registryHome, entryID);
		assertTrue(up);
		
		//remove the service
		sms.destroy();
		
		//and process() should be false now
		up=upd.process(registryHome, entryID);
		assertFalse(up);
		
		//and the registry entry should be gone
		try{
			entry.getCurrentTime();
			fail("Entry should be gone");
		}catch(Exception ex){
			/* OK */
		}
	
	}
	
	@RegressionTest(url="http://sourceforge.net/tracker/?func=detail&aid=3522980&group_id=102081&atid=633902", 
			        description="Tests removal of multiple entries for a given server")
	@Test
	public void testRemoveMultipleEntries()throws Exception{
		url=kernel.getContainerProperties().getValue(ContainerProperties.WSRF_BASEURL);
		registryEpr=EndpointReferenceType.Factory.newInstance();
		registryEpr.addNewAddress().setStringValue(url+"/"+Registry.REGISTRY_SERVICE+"?res=default_registry");
		RegistryClient reg=new RegistryClient(registryEpr,kernel.getClientConfiguration());
		reg.setUpdateInterval(-1);
		
		ExternalRegistryClient registry=new ExternalRegistryClient();
		registry.addClient(reg);
		
		int N=10;
		
		for(int i=0;i<N;i++){		
			AddDocument in=AddDocument.Factory.newInstance();
			EndpointReferenceType member1=EndpointReferenceType.Factory.newInstance();
			member1.addNewAddress().setStringValue("http://foo1/"+i);
			in.addNewAdd().setMemberEPR(member1);
			registry.addRegistryEntry(in);
		}
		
		int c=countEntries(reg, "foo1");
		assertEquals(N, c);
		
		DefaultHome registryHome=(DefaultHome)kernel.getHome(ServiceGroupEntry.SERVICENAME);
		assertNotNull(registryHome);
		registryHome.runExpiryCheckNow();
		
		c=countEntries(reg, "foo1");
		assertEquals(0, c);
		
		
	}
	
	private int countEntries(RegistryClient reg, String memberUrlContains)throws Exception{
		int c=0;
		for(EntryType e: reg.listEntries()){
			if(e.getMemberServiceEPR().getAddress().getStringValue().contains(memberUrlContains)){
				c++;
			}
		}
		return c;
	}
	
	
	@Test
	public void testChangeBaseURL()throws Exception{
		url=kernel.getContainerProperties().getValue(ContainerProperties.WSRF_BASEURL);
		registryEpr=EndpointReferenceType.Factory.newInstance();
		registryEpr.addNewAddress().setStringValue(url+"/"+Registry.REGISTRY_SERVICE+"?res=default_registry");
		RegistryClient reg=new RegistryClient(registryEpr,kernel.getClientConfiguration());
		//add a fake plain ws entry
		EndpointReferenceType memberEpr=EndpointReferenceType.Factory.newInstance();
		memberEpr.addNewAddress().setStringValue(url+"Test");
		reg.addRegistryEntry(memberEpr);
		
		//list entries and figure out correct registry entry EPR
		List<EntryType>entries=reg.listEntries();
		assertNotNull(entries);
		assertTrue(entries.size()>0);
		EndpointReferenceType entryEPR=null;
		String entryID=null;
		for(EntryType e: entries){
			EndpointReferenceType member=e.getMemberServiceEPR();
			String memberURL=member.getAddress().getStringValue();
			if(memberURL.contains("Test")){
				entryEPR=e.getServiceGroupEntryEPR();
				entryID=WSUtilities.extractResourceID(entryEPR);
				break;
			}
		}
		assertNotNull(entryEPR);
		assertNotNull(entryID);
		
		//change base url
		
		kernel.getContainerProperties().setProperty(ContainerProperties.WSRF_BASEURL, "x"+url);

		Home registryHome=kernel.getHome(ServiceGroupEntry.SERVICENAME);
		assertNotNull(registryHome);

		RegistryEntryUpdater upd=new RegistryEntryUpdater();
		boolean valid=upd.process(registryHome, entryID);
		assertFalse(valid);
		
		//and the registry entry should be gone
		try{
			registryHome.get(entryID);
			fail("Entry should be gone");
		}catch(Exception ex){
			/* OK */
		}
	
	}
	

}
