package de.fzj.unicore.uas.testsuite;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.ByteArrayOutputStream;

import org.junit.Test;
import org.unigrids.services.atomic.types.ProtocolType;
import org.unigrids.x2006.x04.services.fts.SummaryType;
import org.w3.x2005.x08.addressing.EndpointReferenceDocument;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.Base;
import de.fzj.unicore.uas.UAS;
import de.fzj.unicore.uas.UASProperties;
import de.fzj.unicore.uas.client.EnumerationClient;
import de.fzj.unicore.uas.client.StorageClient;
import de.fzj.unicore.uas.client.StorageFactoryClient;
import de.fzj.unicore.uas.client.TransferControllerClient;
import de.fzj.unicore.wsrflite.utils.WSServerUtilities;
import eu.unicore.bugsreporter.annotation.FunctionalTest;

/**
 * tests sendFile() and receiveFile()
 * 
 * @author schuller
 */
public class RunServerServerTransfer extends Base {

	EndpointReferenceType factory;
	StorageFactoryClient sfc;
	StorageClient source, target;
	String sourceURL,targetURL;
	
	
	@FunctionalTest(id="ServerServerFTTest", 
					description="Tests server-to-server filetransfer")
	@Test
	public void testTransfer()throws Exception{
		factory= WSServerUtilities.makeEPR(UAS.SMF, "default_storage_factory",kernel);
		sfc=new StorageFactoryClient(factory,kernel.getClientConfiguration());
		initSource();
		UASProperties cfg = kernel.getAttribute(UASProperties.class);
		for(String noOpt: new String[]{"true", "false"}){
			System.out.println("Testing with disabled local copy optimization: "+noOpt);
			cfg.setProperty(UASProperties.SMS_TRANSFER_FORCEREMOTE, noOpt);
			dataTransfer();
			checkSpacesInNames();
		}
	}
	
	protected void initSource() throws Exception {
		source=sfc.createSMS();
		source.createDirectory("folder1/folder11");
		source.createDirectory("folder2");
		source.getImport("folder1/test11").append("test11".getBytes());
		source.getImport("folder1/test12").append("test12".getBytes());
		source.getImport("folder1/zeros");
		source.getImport("folder1/folder11/test111").append("test111".getBytes());
		source.getImport("folder2/test21").append("test12".getBytes());
		source.getImport("test.txt").append("this is a test".getBytes());
		source.getImport("test1.txt").append("this is a test".getBytes());
		sourceURL=source.getEPR().getAddress().getStringValue();
	}
	
	protected void verifyTargetFolder1Content()throws Exception{
		ByteArrayOutputStream bos=new ByteArrayOutputStream();
		target.getExport("folder1/test11", ProtocolType.BFT).readAllData(bos);
		assertTrue("test11".equals(bos.toString()));
		bos=new ByteArrayOutputStream();
		target.getExport("folder1/folder11/test111", ProtocolType.BFT).readAllData(bos);
		assertTrue("test111".equals(bos.toString()));
	}
	

	protected void reInitTarget() throws Exception {
		target=sfc.createSMS();
		targetURL=target.getEPR().getAddress().getStringValue();
	}

	protected void dataTransfer()throws Exception{
		String [] protocols=new String[]{"BFT"};//new String[]{"BFT", "RBYTEIO", "SBYTEIO"};
		for(String protocol: protocols){
			System.out.println("\n\nTesting server-server transfer using <"+protocol+">");
			System.out.println(" ... Fetch single file");
			fetchSingleFile(protocol);
			System.out.println(" ... Send single file");
			sendSingleFile(protocol);
			System.out.println(" ... Fetch folder");
			fetchFolder(protocol);
			System.out.println(" ... Send folder");
			sendFolder(protocol);
		}
	}

	protected void fetchSingleFile(String protocol)throws Exception{
		reInitTarget();
		TransferControllerClient c=target.fetchFile(protocol+":"+sourceURL+"#test.txt",
				"/test.txt");
		assertTrue(c!=null);
		int cnt=0;
		while(!c.isComplete()&& !c.hasFailed()){
			if(cnt>0)System.out.print(".");
			Thread.sleep(1000);
			cnt++;
			if(cnt>100)throw new Exception("Filetransfer took too long, aborting test...");
		}
		assertTrue(SummaryType.DONE.equals(c.getStatusSummary()));
		//check rate property
		System.out.println("Transfer rate (according to server) " + c.getRate()+" Bytes/sec.");
		assertTrue(c.getRate()>-1);
	}

	@FunctionalTest(id="ServerServerFTReferencesTest", 
			description="Tests the references to server-to-server filetransfers")
	@Test
	public void testTransferReferences()throws Exception{
		factory= WSServerUtilities.makeEPR(UAS.SMF, "default_storage_factory",kernel);
		sfc=new StorageFactoryClient(factory,kernel.getClientConfiguration());
		source=sfc.createSMS();
		source.getImport("test.txt").append("test123".getBytes());
		reInitTarget();
		TransferControllerClient c=sendSingleFile("BFT");
		EnumerationClient<EndpointReferenceDocument> ftReferences=source.getFiletransferEnumeration();
		assertEquals(1, ftReferences.getNumberOfResults());
		EndpointReferenceDocument ft1=ftReferences.getResults(0, 1).get(0);
		System.out.println(ft1);
		System.out.println("File transfer url = "+
				ft1.getEndpointReference().getAddress().getStringValue());
		
		System.out.println(source.getResourcePropertyDocument());
		
		c.destroy();
		ftReferences.setUpdateInterval(-1);
		assertEquals(0, ftReferences.getNumberOfResults());
		
	}
	
	
	protected void fetchFolder(String protocol)throws Exception{
		reInitTarget();
		TransferControllerClient c=target.fetchFile(protocol+":"+sourceURL+"#folder1",
				"/folder1");
		assertTrue(c!=null);

		int cnt=0;
		while(!c.isComplete()&& !c.hasFailed()){
			System.out.print(".");
			Thread.sleep(1000);
			cnt++;
			if(cnt>1000)throw new Exception("Filetransfer took too long, aborting test...");
		}
		assertTrue(SummaryType.DONE.equals(c.getStatusSummary()));
		verifyTargetFolder1Content();
	}

	
	protected TransferControllerClient sendSingleFile(String protocol)throws Exception{
		reInitTarget();
		TransferControllerClient c=source.sendFile("/test.txt",
				protocol+":"+targetURL+"#testfile2");
		assertTrue(c!=null);

		int cnt=0;
		do{
			if(cnt>0)System.out.print(".");
			Thread.sleep(1000);
			cnt++;
			if(cnt>100)throw new Exception("Filetransfer took too long, aborting test...");
		}while(!c.isComplete() && !c.hasFailed());
		assertTrue(SummaryType.DONE.equals(c.getStatusSummary()));
		//check rate property
		System.out.println("Transfer rate (according to server) " + c.getRate()+" Bytes/sec.");
		assertTrue(c.getRate()>-1);
		return c;
	}

	protected void sendFolder(String protocol)throws Exception{
		reInitTarget();
		TransferControllerClient c=source.sendFile("folder1",protocol+":"+targetURL+"#folder1");
		assertTrue(c!=null);

		int cnt=0;
		while(!c.isComplete()&& !c.hasFailed()){
			System.out.print(".");
			Thread.sleep(1000);
			cnt++;
			if(cnt>12000)throw new Exception("Filetransfer took too long, aborting test...");
		}
		assertTrue(SummaryType.DONE.equals(c.getStatusSummary()));
		verifyTargetFolder1Content();
	}


	protected void checkSpacesInNames()throws Exception{
		//import a file that includes a space character
		String testdata="testdata";
		source.getImport("test%20file").write(testdata.getBytes());
		ByteArrayOutputStream bos=new ByteArrayOutputStream();
		source.getExport("test file").readAllData(bos);
		assertTrue(testdata.equals(bos.toString()));

		//now check we can send this to our default storage
		//tell uspace sms to send a file to the default_storage sms
		TransferControllerClient c1=source.sendFile("/test file","BFT:"+targetURL+"#test file");
		assertTrue(c1!=null);
		int cnt=0;
		do{
			if(cnt>0)System.out.println("Not complete...");
			Thread.sleep(1000);
			cnt++;
			if(cnt>100)throw new Exception("Filetransfer took too long, aborting test...");
		}while(!c1.isComplete() && !c1.hasFailed());
		
		//get data from sms and check correctness
		bos.reset();
		target.getExport("test file").readAllData(bos);
		assertTrue(testdata.equals(bos.toString()));


		//same story with receiveFile
		c1=source.fetchFile("BFT:"+targetURL+"#test file","/another test file");
		assertTrue(c1!=null);
		cnt=0;
		do{
			if(cnt>0)System.out.println("Not complete...");
			Thread.sleep(1000);
			cnt++;
		}while(!c1.isComplete() && !c1.hasFailed());
		System.out.println("Size of remote file = "+c1.getSize());
		System.out.println("Transferred         = "+c1.getTransferredBytes());
		System.out.println("Final status        = "+c1.getStatus());

		//get data from sms and check correctness
		bos.reset();
		source.getExport("another test file").readAllData(bos);
		assertTrue(testdata.equals(bos.toString()));
	}

}
