package de.fzj.unicore.uas.security;

import java.io.File;
import java.util.Map;

import junit.framework.Assert;

import org.apache.commons.io.FileUtils;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;
import org.unigrids.services.atomic.types.SecurityType;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.TestSecConfigs;
import de.fzj.unicore.uas.UAS;
import de.fzj.unicore.uas.client.StorageClient;
import de.fzj.unicore.uas.client.TSFClient;
import de.fzj.unicore.uas.client.TSSClient;
import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.wsrflite.security.IAttributeSource;
import de.fzj.unicore.wsrflite.utils.WSServerUtilities;
import eu.unicore.util.httpclient.IClientConfiguration;

public class TestSecurityInfo {

	static Kernel kernel;
	
	@BeforeClass
	public static void setUp()throws Exception{
		FileUtils.deleteQuietly(new File("target","data"));
		UAS uas=new UAS("src/test/resources/secure/uas.config");
		uas.startSynchronous();
		kernel=uas.getKernel();
	}
	
	@AfterClass
	public static void tearDown()throws Exception{
		kernel.shutdown();
		FileUtils.deleteQuietly(new File("target","data"));
	}
	
	@Test
	public void testSecurityInfo()throws Exception{
		IClientConfiguration sec = TestSecConfigs.getClientSecurityCfg(true);
		EndpointReferenceType epr=WSServerUtilities.makeEPR(UAS.TSF, "default_target_system_factory", kernel);
		TSFClient tsf=new TSFClient(epr, sec);
		Assert.assertNotNull(tsf.getSecurityInfo());
		Assert.assertEquals(2, tsf.getXlogins().length);
		Assert.assertEquals(2, tsf.getXgroups().length);
		Assert.assertEquals(1, tsf.getAcceptedCAs().length);
	}

	@Test
	public void testSecurityPreferencesPersistence()throws Exception{
		IClientConfiguration sec = TestSecConfigs.getClientSecurityCfg(true);
		EndpointReferenceType epr=WSServerUtilities.makeEPR(UAS.TSF, "default_target_system_factory", kernel);
		
		

		//no preferences - check if the defaults are ok
		TSFClient tsf=new TSFClient(epr, sec);
		TSSClient tss0 = tsf.createTSS();
		SecurityType secInfo0 = tss0.getSecurityInfo();
		Assert.assertEquals("somebody", secInfo0.getClientSelectedXlogin());
		Assert.assertEquals("users", secInfo0.getClientSelectedXgroup().getPrimaryGroup());

		//preferences used - check if were applied (here persistence is not yet used - just a normal situation)
		Map<String, String[]> preferences = sec.getETDSettings().getRequestedUserAttributes2();
		preferences.put(IAttributeSource.ATTRIBUTE_XLOGIN, new String[]{"nobody"});
		tsf=new TSFClient(epr, sec);
		TSSClient tss = tsf.createTSS();
		EndpointReferenceType smsEpr = tss.getStorages().get(0);
		StorageClient sms = new StorageClient(smsEpr, sec);
		SecurityType secInfo = sms.getSecurityInfo();
		Assert.assertEquals("nobody", secInfo.getClientSelectedXlogin());
		Assert.assertEquals("users", secInfo.getClientSelectedXgroup().getPrimaryGroup());

		//access the resource but with different preferences - explicitly set should overwrite any persisted
		Map<String, String[]> preferences2 = sec.getETDSettings().getRequestedUserAttributes2();
		preferences2.put(IAttributeSource.ATTRIBUTE_XLOGIN, new String[]{"somebody"});
		preferences2.put(IAttributeSource.ATTRIBUTE_GROUP, new String[]{"system"});
		sms = new StorageClient(smsEpr, sec);
		SecurityType secInfo2 = sms.getSecurityInfo();
		Assert.assertEquals("somebody", secInfo2.getClientSelectedXlogin());
		Assert.assertEquals("system", secInfo2.getClientSelectedXgroup().getPrimaryGroup());
		
		//check of persistence - not using any preferences
		Map<String, String[]> preferences3 = sec.getETDSettings().getRequestedUserAttributes2();
		preferences3.remove(IAttributeSource.ATTRIBUTE_XLOGIN);
		preferences3.remove(IAttributeSource.ATTRIBUTE_GROUP);
		sms = new StorageClient(smsEpr, sec);
		SecurityType secInfo3 = sms.getSecurityInfo();
		Assert.assertEquals("nobody", secInfo3.getClientSelectedXlogin());
		Assert.assertEquals("users", secInfo3.getClientSelectedXgroup().getPrimaryGroup());

	
		//and using only one preference
		Map<String, String[]> preferences4 = sec.getETDSettings().getRequestedUserAttributes2();
		preferences4.put(IAttributeSource.ATTRIBUTE_GROUP, new String[]{"system"});
		sms = new StorageClient(smsEpr, sec);
		SecurityType secInfo4 = sms.getSecurityInfo();
		Assert.assertEquals("nobody", secInfo4.getClientSelectedXlogin());
		Assert.assertEquals("system", secInfo4.getClientSelectedXgroup().getPrimaryGroup());

	}
}
