/*********************************************************************************
 * Copyright (c) 2006 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/

package de.fzj.unicore.uas.xnjs;

import java.io.FileInputStream;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlObject;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.ApplicationType;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.JobDefinitionDocument;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.ResourcesDocument;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.ResourcesType;
import org.unigrids.services.atomic.types.PermissionsType;
import org.unigrids.services.atomic.types.TextInfoType;
import org.unigrids.x2006.x04.services.sms.FilterType;

import de.fzj.unicore.uas.UASProperties;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.wsrflite.utils.WSServerUtilities;
import de.fzj.unicore.xnjs.Configuration;
import de.fzj.unicore.xnjs.XNJS;
import de.fzj.unicore.xnjs.ems.Action;
import de.fzj.unicore.xnjs.ems.Manager;
import de.fzj.unicore.xnjs.io.CompositeFindOptions;
import de.fzj.unicore.xnjs.io.Permissions;
import de.fzj.unicore.xnjs.io.SimpleFindOptions;
import de.fzj.unicore.xnjs.io.http.IConnectionFactory;
import de.fzj.unicore.xnjs.jsdl.Incarnation;
import de.fzj.unicore.xnjs.jsdl.IncarnationDataBase;
import de.fzj.unicore.xnjs.simple.SimpleIDB;
import de.fzj.unicore.xnjs.tsi.ApplicationInfo;
import de.fzj.unicore.xnjs.tsi.IExecutionSystemInformation;
import de.fzj.unicore.xnjs.tsi.IReservation;
import de.fzj.unicore.xnjs.tsi.TSI;
import eu.unicore.security.Client;

/**
 * This facade class wrap some XNJS specifics to reduce 
 * clutter in the UAS implementation. Various helper methods
 * provide convenient access to XNJS functionality.<br/>
 * 
 * TODO document how to deal with multiple XNJS instances and
 * how the "default" XNJS instance works
 * 
 * @author schuller
 */
public class XNJSFacade {

	private static final Logger logger=LogUtil.getLogger(LogUtil.UNICORE,XNJSFacade.class);

	private static final String DEFAULT_INSTANCE=XNJSFacade.class.getName()+"_DEFAULT_XNJS";

	private XNJS xnjs;

	private String id;

	private static synchronized  XNJSInstancesMap getXNJSInstanceMap(Kernel kernel){
		XNJSInstancesMap map=kernel.getAttribute(XNJSInstancesMap.class);
		if(map==null){
			map=new XNJSInstancesMap();
			kernel.setAttribute(XNJSInstancesMap.class, map);
		}
		return map;
	}
	
	/**
	 * get an {@link XNJSFacade} instance
	 * 
	 * @param xnjsReference - the ID of the XNJS to use. Can be null, 
	 * then the default XNJS instance defined in the Kernel config is used
	 * @param the {@link Kernel}
	 * @return {@link XNJSFacade}
	 */
	public static synchronized XNJSFacade get(String xnjsReference, Kernel kernel){
		String ref=xnjsReference!=null?xnjsReference:DEFAULT_INSTANCE;
		
		XNJSFacade r=getXNJSInstanceMap(kernel).get(ref);
		if (r==null){
			r=new XNJSFacade(kernel);
			if(xnjsReference==null){
				r.doDefaultInit(kernel);
			}
			else{
				
			}
			getXNJSInstanceMap(kernel).put(ref, r);
			r.setID(ref);
		}
		return r;
	}
	
	private final Kernel kernel;
	
	private XNJSFacade(Kernel kernel){
		this.kernel=kernel;
	}

	private void setID(String id){
		this.id=id;
	}
	/**
	 * create an XNJS instance, configured from the named file
	 * 
	 * @param configFileName -  the name of the config file
	 * @throws Exception
	 */
	public void configure(String configFileName)throws Exception{
		configure(new FileInputStream(configFileName));
	}

	/**
	 * create an XNJS instance, configured from the named file
	 * 
	 * @param config -  the configuration
	 * @throws Exception
	 */
	public void configure(InputStream config)throws Exception{
		xnjs=new XNJS(config);
		//register kernel in XNJS config now
		xnjs.getConfig().registerComponent(kernel);
		xnjs.getConfig().registerComponent(kernel.getClientConfiguration());
		xnjs.start();
		Thread.sleep(500);
		configure(xnjs);
	}

	private void doDefaultInit(Kernel kernel){
		UASProperties uasConfig = kernel.getAttribute(UASProperties.class);
		String config=uasConfig.getValue(UASProperties.TSF_XNJS_CONFIGFILE);
		if(config==null){
			logger.info("No XNJS configured.");
		}
		else{
			try{
				logger.info("Configuring XNJS from file "+config);
				FileInputStream fis=new FileInputStream(config);
				try{
					configure(fis);
				}
				finally{
					fis.close();
				}
			}
			catch(Exception e){
				throw new RuntimeException("Error configuring XNJS.",e);
			}
		}
	}
	/**
	 * do some UAS specific configuration
	 * 
	 * @param xnjs - the XNJS instance to configure
	 */
	protected void configure(XNJS xnjs){
		//set the https connection factory, if not defined in config already
		IConnectionFactory cf = xnjs.getConfig().getComponentInstanceOfType(IConnectionFactory.class);
		if(cf==null){
			xnjs.getConfig().registerComponent(U6HttpConnectionFactory.class);
		}
	}

	/**
	 * helper method to get the workdir for a given action
	 */
	public String getWorkdir(String actionID){
		try{
			return xnjs.getConfig().getInternalManager().getAction(actionID).getExecutionContext().getWorkingDirectory();
		}catch(Exception e){
			return null;
		}
	}
	/**
	 * helper method to make an action from an JSDL doc
	 */
	public Action makeAction(JobDefinitionDocument doc){
		try{
			return xnjs.getConfig().makeAction(doc);
		}catch(Exception e){
			return null;
		}
	}

	/**
	 * helper method to retrieve an action from the XNJS
	 */
	public final Action getAction(String id){
		try{
			return xnjs.getConfig().getInternalManager().getAction(id);
		}catch(Exception e){
			LogUtil.logException("Error retrieving action <"+id+">", e);
			return null;
		}
	}

	/**
	 * Retrieve the status of an action
	 */
	public final Integer getStatus(String id, Client client){
		try{
			return xnjs.getConfig().getEMSManager().getStatus(id,client);
		}catch(Exception e){
			LogUtil.logException("Error retrieving action status for <"+id+">", e);
			return null;
		}
	}

	/**
	 * Retrieve the exit code of an action
	 */
	public final Integer getExitCode(String id, Client client){
		try{
			Action a=xnjs.getConfig().getInternalManager().getAction(id);
			if(a!=null)return a.getExecutionContext().getExitCode();
			else return null;
		}catch(Exception e){
			LogUtil.logException("Error retrieving exit code for <"+id+">", e);
			return null;
		}
	}

	/**
	 * Retrieve the progress of an action
	 */
	public final Float getProgress(String id, Client client){
		try{
			Action a=xnjs.getConfig().getInternalManager().getAction(id);
			if(a!=null)return a.getExecutionContext().getProgress();
			else {
				logger.info("Can't get progress for action "+id+", not found on XNJS.");
				return null;
			}
		}catch(Exception e){
			LogUtil.logException("Error retrieving progress for <"+id+">", e);
			return null;
		}
	}


	/**
	 * Destroy an action on the XNJS, removing its Uspace
	 * 
	 * @param id - the ID of the action to destroy 
	 */
	public void destroyAction(String id, Client client){
		try{
			xnjs.getConfig().getEMSManager().destroy(id,client);
		}catch(Exception e){
			LogUtil.logException("Error destroying job <"+id+">", e);
		}
	}

	/**
	 * get the XNJS manager object
	 * @return manager
	 */
	public final Manager getManager(){
		return xnjs.getConfig().getEMSManager();
	}

	/**
	 * get the XNJS incarnation interface 
	 * @return
	 */
	public final Incarnation getGrounder(){
		return xnjs.getConfig().getGrounder();
	}

	/**
	 * get the XNJS incarnation database for accessing resource info
	 * @return
	 */
	public final IncarnationDataBase getIDB(){
		return xnjs.getConfig().getIDB();
	}

	/**
	 *Returns the XNJS config object
	 */
	public final Configuration getConfiguration(){
		return xnjs.getConfig();
	}

	/**
	 * Returns the number of active jobs (RUNNING/QUEUED) in the various queues. 
	 * The result may be <code>null</code>, if the system does not have queues 
	 */
	public Map<String,Integer>getQueueFill(){
		return getConfiguration().getComponentInstanceOfType(IExecutionSystemInformation.class).getQueueFill();
	}

	/**
	 * Returns the number of active jobs on the XNJS 
	 */
	public int getNumberOfJobs(){
		return getConfiguration().getInternalManager().getAllJobs()-getConfiguration().getInternalManager().getDoneJobs();
	}

	/**
	 * Returns the applications known to the XNJS
	 */
	public final ApplicationInfo[] getDefinedApplications(){
		SimpleIDB gr=(SimpleIDB)getIDB();
		return gr.getDefinedApplicationTypes();
	}

	/**
	 * Returns the applications known to the XNJS
	 * @deprecated consider using getDefinedApplications() instead
	 */
	public final ApplicationType[] getDefinedApplicationTypes(){
		ApplicationInfo[]apps=getDefinedApplications();
		ApplicationType[]at=new ApplicationType[apps.length];
		int i=0;
		for(ApplicationInfo app: apps){
			at[i]=ApplicationType.Factory.newInstance();
			at[i].setApplicationName(app.getApplicationName());
			at[i].setApplicationVersion(app.getApplicationVersion());
			i++;
		}
		return at;
	}

	/**
	 * Returns the TextInfo resources configured in the XNJS IDB
	 */
	public final TextInfoType[] getDefinedTextInfo(){
		IncarnationDataBase gr=getIDB();
		Map<String,String> infos=gr.getTextInfoProperties();
		TextInfoType[] res=new TextInfoType[infos.keySet().size()];
		int i=0;
		for(String name: infos.keySet()){
			res[i]=TextInfoType.Factory.newInstance();
			res[i].setName(name);
			res[i].setValue(infos.get(name));
			i++;
		}
		return res;
	}

	/**
	 * Returns the system resources	
	 */
	public final ResourcesType getResources(Client c){
		try {
			IncarnationDataBase gr=getIDB();
			return (ResourcesType)gr.listResources(c);
		} catch (Exception e) {
			LogUtil.logException("Could not obtain resource information from XNJS.",e);
			return null;
		}
	}


	/**
	 * get a TSI for accessing files
	 * @param storageRoot -  the directory the TSI initally points to
	 * @param client -  the client object with authN/ authZ information
	 * @return TSI
	 */
	public final TSI getStorageTSI(String storageRoot, Client client){
		TSI tsi=xnjs.getConfig().getTargetSystemInterface(client);
		tsi.setStorageRoot(storageRoot);
		return tsi;
	}

	/**
	 * get a TSI
	 * @param client -  the client object with authN/ authZ information
	 * @return TSI
	 */
	public final TSI getTSI(Client client){
		return xnjs.getConfig().getTargetSystemInterface(client);
	}

	/**
	 * check if the XNJS is configured to support reservation
	 * @return true if the XNJS supports reservation, false otherwise
	 */
	public boolean supportsReservation(){
		return xnjs.getConfig().getReservationInterface()!=null;
	}

	/**
	 * get the reservation interface
	 * @return an {@link IReservation} for making reservations
	 */
	public final IReservation getReservation(){
		return xnjs.getConfig().getReservationInterface();
	}


	//TODO check xnjs.stop()
	public void shutdown()throws Exception{
		//do not shutdown the default instance
		if(DEFAULT_INSTANCE.equals(id)){
			logger.warn("Tried to shutdown default XNJS, ignoring...");
			return;
		}
		xnjs.stop();
		//allow instance to be GC'ed
		getXNJSInstanceMap(kernel).put(id, null);
	}

	/**
	 * insert a "site-specific" resource request into the JSDL Resources document
	 *  
	 */
	public static void insertXNJSResourceSpec(String name, double value, ResourcesDocument resources){
		String s="<xnjs:ResourceSetting xmlns:xnjs=\"http://www.fz-juelich.de/unicore/xnjs/idb\">"
			+"<xnjs:Name>"+name+"</xnjs:Name>"
			+"<xnjs:Value xmlns:jsdl=\"http://schemas.ggf.org/jsdl/2005/11/jsdl\"><jsdl:Exact>"+value+"</jsdl:Exact></xnjs:Value>"
			+"</xnjs:ResourceSetting>";
		try{
			XmlObject o=XmlObject.Factory.parse(s);
			//and append to resources doc...
			WSServerUtilities.append(o, resources);
		}catch(Exception e){
			LogUtil.logException("Error building site-specific resource",e);
		}
	}

	/**
	 * get the last instant that the IDB was updated
	 *  
	 * @return long - the last update time (in millis)
	 */
	public final long getLastIDBUpdate(){
		return getIDB().getLastUpdateTime();
	}

	/**
	 * recursively build the filter options for the find command
	 * @param filter
	 * @return {@link CompositeFindOptions}
	 */
	public final CompositeFindOptions getFindOptions(final FilterType filter){
		CompositeFindOptions cfo=new CompositeFindOptions();
		if(filter!=null){
			String name=filter.getNameMatch();
			if(name!=null){
				cfo.and(SimpleFindOptions.stringMatch(name, false));
			}
			String regExp=filter.getNameMatchRegExp();
			if(regExp!=null){
				cfo.and(SimpleFindOptions.regExpMatch(regExp, false));
			}
			Calendar before=filter.getBefore();
			if(before!=null){
				cfo.and(SimpleFindOptions.lastAccessBefore(before,false));
			}
			Calendar after=filter.getAfter();
			if(after!=null){
				cfo.and(SimpleFindOptions.lastAccessAfter(after,false));
			}
			//check sub filters...
			if(filter.getAndFilter()!=null){
				CompositeFindOptions andCFO=getFindOptions(filter.getAndFilter());
				cfo.and(andCFO);
			}
			if(filter.getOrFilter()!=null){
				CompositeFindOptions orCFO=getFindOptions(filter.getOrFilter());
				cfo.or(orCFO);
			}
		}
		return cfo;
	}

	public static Permissions getXNJSPermissions(PermissionsType uasPermissions){
		Permissions perm=new Permissions();
		perm.setExecutable(uasPermissions.getExecutable());
		perm.setReadable(true);//all else would be complete nonsense
		perm.setWritable(uasPermissions.getWritable());
		return perm;
	}

	public static PermissionsType getUASPermissions(Permissions xnjsPermissions){
		PermissionsType perm=PermissionsType.Factory.newInstance();
		perm.setExecutable(xnjsPermissions.isExecutable());
		perm.setReadable(true);//all else would be complete nonsense
		perm.setWritable(xnjsPermissions.isWritable());
		return perm;
	}

	/**
	 * list all jobs accessible by the particular client (may be more than actually wanted, so 
	 * a second, more detailed filtering step should be made...)
	 * @param client
	 * @return
	 */
	public List<String>listJobIDs(Client client)throws Exception{
		return Arrays.asList(xnjs.getConfig().getEMSManager().list(client));
	}

	public static class XNJSInstancesMap extends HashMap<String, XNJSFacade>{
		private static final long serialVersionUID = 1L;
	}
	
}
