/*********************************************************************************
 * Copyright (c) 2006 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
 
package de.fzj.unicore.uas.xnjs;

import java.net.URI;
import java.util.Arrays;

import org.apache.log4j.Logger;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.fts.FileTransferCapabilities;
import de.fzj.unicore.uas.fts.FileTransferCapability;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.xnjs.Configuration;
import de.fzj.unicore.xnjs.io.DataStagingCredentials;
import de.fzj.unicore.xnjs.io.IFileTransfer;
import de.fzj.unicore.xnjs.io.IFileTransferCreator;
import eu.unicore.security.Client;

/**
 * creates {@link IFileTransfer} instances that use a UNICORE 6 protocol such as BFT 
 *
 * @author schuller
 */
public class U6FileTransferCreator implements IFileTransferCreator{
	
	protected static final Logger logger=LogUtil.getLogger(LogUtil.SERVICES,U6FileTransferCreator.class);
	private final Configuration configuration;
	
	public U6FileTransferCreator(Configuration config) {
		super();
		this.configuration=config;
	}

	public IFileTransfer createFileExport(Client client, String workdir,String source,
			URI target,DataStagingCredentials creds) {
		String scheme=target.getScheme();
		FileTransferCapability fc=FileTransferCapabilities.getCapability(scheme);
		if(fc!=null){
			try{
				return createExport(fc.getExporter(),client,workdir,source,target,creds);
			}catch(Exception ex){
				LogUtil.logException("Error creating importer for protocol <"+scheme+">",ex);
			}
		}
		return null;
	}

	public IFileTransfer createFileImport(Client client, String workdir, URI source, String target, DataStagingCredentials creds){
		
		String scheme=source.getScheme();
		FileTransferCapability fc=FileTransferCapabilities.getCapability(scheme);
		if(fc!=null){
			try{
				return createImport(fc.getImporter(),client,workdir,source,target,creds);
			}catch(Exception ex){
				LogUtil.logException("Error creating importer for protocol <"+scheme+">",ex);
			}
		}
		return null;
	}

	public String getProtocol() {
		return Arrays.toString(FileTransferCapabilities.getProtocols());
	}
	
	/**
	 * create a transfer FROM a SMS to a local file
	 * 
	 * the assumed URI format is
	 *   
	 *   u6protocol:protocol://host:port/servicespec?res=resourceID#filespec
	 * 
	 * @param parent
	 * @param workdir
	 * @param source
	 * @param target
	 * @param credentials - ignored
	 * @return
	 */
	public IFileTransfer createImport(Class<? extends U6FileTransferBase> clazz,Client client, String workdir, URI source, String targetFile, DataStagingCredentials creds){
		EndpointReferenceType epr=createStorageEPR(source);
		
		String sourceFile=urlDecode(source.getFragment());
		try{
			U6FileTransferBase ft=clazz.getConstructor(Configuration.class).newInstance(configuration);
			ft.setClient(client);
			ft.setWorkdir(workdir);
			ft.setSource(sourceFile);
			ft.setTarget(targetFile);
			ft.setSmsEPR(epr);
			ft.setExport(false);
			return ft;
		}catch(Exception e){
			logger.warn("Unable to instantiate File transfer", e);
			return null;
		}
	} 
	
	
	/**
	 * create a transfer TO an SMS from a local file
	 * 
	 * the assumed URL encoding is
	 *   
	 *   u6protocol:protocol://host:port/servicespec?res=resourceID#filespec
	 * 
	 * @param client
	 * @param workdir
	 * @param source
	 * @param target
	 * @param credentials - ignored
	 * @return
	 */
	public IFileTransfer createExport(Class<? extends U6FileTransferBase> clazz, Client client, String workdir, String sourceFile, URI target, DataStagingCredentials credentials){
		
		EndpointReferenceType epr=createStorageEPR(target);
		String targetFile=urlDecode(target.getFragment());
		try{
			U6FileTransferBase ft=clazz.getConstructor(Configuration.class).newInstance(configuration);
			ft.setClient(client);
			ft.setSource(sourceFile);
			ft.setTarget(targetFile);
			ft.setWorkdir(workdir);
			ft.setSmsEPR(epr);
			ft.setExport(true);
			return ft;
		}catch(Exception e){
			//can't happen
			e.printStackTrace();
			return null;
		}
	} 
		
	/**
	 * replace URI-encoded characters by their unencoded counterparts
	 * @param orig
	 * @return
	 */
	protected String urlDecode(String orig){
		try{
			return orig.replaceAll("%20", " ");
		}catch(Exception e){
			return orig;
		}
	}
	
	protected EndpointReferenceType createStorageEPR(URI uri)
	{
		EndpointReferenceType epr=EndpointReferenceType.Factory.newInstance();
		String withoutScheme=uri.getSchemeSpecificPart();
		String upToFragment = withoutScheme.split("#")[0];
		epr.addNewAddress().setStringValue(upToFragment);
		return epr;
	}


}