package de.fzj.unicore.uas.xnjs;

import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.unigrids.services.atomic.types.GridFileType;
import org.unigrids.services.atomic.types.PermissionsType;
import org.unigrids.services.atomic.types.ProtocolType;
import org.unigrids.x2006.x04.services.sms.ExportFileDocument;
import org.unigrids.x2006.x04.services.sms.ExportFileResponseDocument;

import de.fzj.unicore.uas.UASProperties;
import de.fzj.unicore.uas.client.FileTransferClient;
import de.fzj.unicore.uas.client.ReliableFileTransferClient;
import de.fzj.unicore.uas.client.StorageClient;
import de.fzj.unicore.uas.fts.FiletransferOptions.IMonitorable;
import de.fzj.unicore.uas.fts.FiletransferOptions.SupportsPartialRead;
import de.fzj.unicore.uas.fts.ProgressListener;
import de.fzj.unicore.uas.fts.rft.StoreImpl;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.uas.util.Pair;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceUnavailableFault;
import de.fzj.unicore.xnjs.Configuration;
import de.fzj.unicore.xnjs.ems.ExecutionException;
import de.fzj.unicore.xnjs.io.IStorageAdapter;
import de.fzj.unicore.xnjs.io.Permissions;
import de.fzj.unicore.xnjs.io.XnjsFile;
import de.fzj.unicore.xnjs.io.XnjsFileWithACL;
import de.fzj.unicore.xnjs.tsi.TSI;
import eu.unicore.util.Log;

/**
 * Base class for UNICORE file imports<br/>
 * Sub classes need only provide an appropriate client class and 
 * the protocol that is used.
 *  
 * @author demuth
 * @author schuller
 */
public abstract class U6FileImportBase extends U6FileTransferBase {

	private GridFileType remote;
	private StorageClient sms;
	
	public U6FileImportBase(Configuration configuration){
		super(configuration);
	}

	public final void run(){
		try{
			checkCancelled();

			initSecurityProperties();

			sms=new StorageClient(smsEPR,sec);

			localFile=target;

			//if overwrite policy is DONT_OVERWRITE: check if file exists
			if(OverwritePolicy.DONT_OVERWRITE.equals(overwrite)){
				XnjsFileWithACL f=null;
				try{
					f=getStorageAdapter().getProperties(localFile);
				}catch(ExecutionException ex){/*ignored*/}
				if(f!=null){
					throw new IOException("Target file <"+localFile+"> exists, and we have been asked to not overwrite.");
				}
			}

			remote = sms.listProperties(source);

			status=Status.RUNNING;
			startTime=System.currentTimeMillis();

			if(remote.getIsDirectory())
			{
				transferFolder(remote, sms, target);
			}
			else
			{
				dataSize=remote.getSize();
				createParentDirectories(target);
				transferFile(remote, sms, target);
			}
			status=Status.DONE;

			computeMetrics();
			destroyFileTransferResource();
		}
		catch(ProgressListener.CancelledException ce){
			String msg="File transfer cancelled.";
			status=Status.DONE; //TODO: CANCELLED does not exist
			statusMessage=msg;
		}
		catch(Exception ex){
			String msg="Error executing filetransfer: "+ex.getMessage();
			LogUtil.logException(msg,ex,logger);
			status=Status.FAILED;
			statusMessage=msg;
		}

		onFinishCleanup();
	}

	@Override
	protected void onFinishCleanup(){
		super.onFinishCleanup();
		remote=null;
	}



	/**
	 * import the file. Will check if the remote file system is in fact the local one, and
	 * will invoke the transferFileByCopy() in that case
	 * @param source
	 * @param sms
	 * @param target
	 * @throws Exception
	 */
	protected void transferFile(GridFileType source, StorageClient sms, String target) throws Exception
	{
		checkCancelled();
		//handle directory as target
		localFile = target;
		IStorageAdapter adapter=getStorageAdapter();
		XnjsFile xFile=adapter.getProperties(localFile);

		if(xFile!=null){
			if(xFile.isDirectory()){
				String name=getFileName(source.getPath());
				if(!target.endsWith(adapter.getFileSeparator())){
					localFile=target+adapter.getFileSeparator()+name;	
				}
				else localFile=target+name;
			}
			if(!xFile.getPermissions().isWritable()){
				throw new IOException("Local target file is not writeable");
			}
		}

		String remoteFSID=sms.getFileSystem().getDescription();
		String localFSID=getStorageAdapter().getFileSystemIdentifier();
		UASProperties conf = kernel.getAttribute(UASProperties.class);
		if(remoteFSID!=null && localFSID!=null && remoteFSID.equals(localFSID) && 
				!conf.getBooleanValue(UASProperties.SMS_TRANSFER_FORCEREMOTE)){
			transferFileLocal(source, sms, target);
		}
		else{
			transferFileFromRemote(source, sms);
		}
		copyPermissions();
	}


	protected void transferFileLocal(GridFileType source, StorageClient sms, String target)throws Exception{
		String base=sms.getFileSystem().getMountPoint();
		if(base==null || OverwritePolicy.APPEND==overwrite){
			transferFileFromRemote(source, sms);
		}
		else{
			String src=base+source.getPath();
			logger.info("Optimization enabled: importing file by local copy of <"+src+"> to <"+localFile+">");
			IStorageAdapter s=getStorageAdapter();
			String sRoot=s.getStorageRoot();
			try{
				s.setStorageRoot("/");
				String tgt=workdir+"/"+localFile;
				checkOverwriteAllowed(s, tgt);
				s.cp(src, tgt);
				transferredBytes+=source.getSize();
			}
			finally{
				s.setStorageRoot(sRoot);
			}
		}
	}


	protected void transferFileFromRemote(GridFileType source, StorageClient sms)throws Exception{
		checkCancelled();
		createNewExport(source, sms);
		if(ftc instanceof IMonitorable){
			((IMonitorable)ftc).setProgressListener(this);
		}
		doRun();
	}

	protected void createNewExport(GridFileType source, StorageClient sms)throws Exception{
		ExportFileDocument req=ExportFileDocument.Factory.newInstance();
		req.addNewExportFile().setSource(source.getPath());
		req.getExportFile().setProtocol(ProtocolType.Enum.forString(getProtocol()));
		Map<String,String>ep=getExtraParameters();
		if(ep!=null && ep.size()>0){
			req.getExportFile().setExtraParameters(convert(ep));
		}
		ExportFileResponseDocument res=sms.ExportFile(req);
		fileTransferInstanceEpr=res.getExportFileResponse().getExportEPR();
		ftc=getFTClient();
	}

	protected void transferFolder(GridFileType source, StorageClient sms, String target) throws Exception
	{
		List<Pair<GridFileType,String>> collection = new ArrayList<Pair<GridFileType,String>>();
		dataSize = collectRemoteFiles(collection, source, sms, target);

		for (Pair<GridFileType,String> pair : collection) {

			GridFileType currentSource = pair.getM1();
			String currentTarget = pair.getM2();
			if(currentSource.getIsDirectory())
			{
				createParentDirectories(currentTarget+"/file");
				transferredBytes++;
			}
			else transferFile(currentSource, sms, currentTarget);
		}
	}



	/**
	 * Recursively gather all files and directories that need to be copied
	 */
	protected long collectRemoteFiles(List<Pair<GridFileType,String>> collection, GridFileType sourceFolder, StorageClient sms, String targetFolder) throws Exception
	{
		long result = 1;
		collection.add(new Pair<GridFileType,String>(sourceFolder, targetFolder));
		for (GridFileType child : sms.listDirectory(sourceFolder.getPath())) {
			String relative = child.getPath().substring(sourceFolder.getPath().length()+1);
			String target = targetFolder+getStorageAdapter().getFileSeparator()+relative;

			if(child.getIsDirectory()) 
			{
				result += collectRemoteFiles(collection, child, sms, target);
			}
			else 
			{
				collection.add(new Pair<GridFileType,String>(child,target));
				result+= child.getSize();
			}
		}
		return result;
	}

	public long getDataSize()
	{
		return dataSize;
	}

	@Override
	protected void doRun()throws Exception{
		if(reliableMode){
			try{
				doRunReliably();
			}catch(IllegalArgumentException iae){
				Log.logException("", iae, logger);
				doRunUnreliably();
			}
		}
		else{
			doRunUnreliably();
		}
	}

	protected void doRunUnreliably()throws Exception{
		OutputStream os=null;
		boolean append=OverwritePolicy.APPEND.equals(overwrite);
		os=getStorageAdapter().getOutputStream(localFile, append);
		try{
			checkCancelled();
			if(ftc instanceof IMonitorable){
				((IMonitorable)ftc).setProgressListener(this);
			}
			ftc.readAllData(os);
		}finally{
			if(os!=null)os.close();
		}
	}	

	protected void doRunReliably()throws Exception{
		TSI tsi=null;
		if(getStorageAdapter() instanceof TSI){
			tsi=(TSI)getStorageAdapter();
		}
		else{
			throw new IllegalArgumentException("Reliable mode not supported for <"+getStorageAdapter().getClass()+">");
		}
		boolean ftcOK=true;
		do{
			try{
				StoreImpl store=new StoreImpl(client,tsi,configuration,target,ftc.getSourceFileSize());
				singleRFTRun(store);
			}catch(CancelledException ce){
				throw ce;
			}catch(Exception ex){
				//check if it is some sort of ftc related issue.
				//for example "resource unknown"
				ftcOK=checkOK(ftc);
				if(!ftcOK){
					//create new transfer resource and continue the download
					createNewExport(remote, sms);
				}
				throw ex;	
			}
		}while(!ftcOK);
	}

	protected void singleRFTRun(StoreImpl store)throws Exception{
		ReliableFileTransferClient rft=new ReliableFileTransferClient((SupportsPartialRead)ftc, store);
		rft.setProgressListener(this);
		checkCancelled();
		rft.run();
	}



	protected boolean checkOK(FileTransferClient ftc){
		Exception ex=null;
		do{
			try{
				ftc.getCurrentTime();
			}
			catch(ResourceUnavailableFault ruf){
				ex=ruf;
				try{
					Thread.sleep(5000);
				}catch(InterruptedException ie){}
			}
			catch(Exception ex1){
				return false;
			}
		}while(ex!=null && ex instanceof ResourceUnavailableFault);
		
		return true;
	}

	protected void copyPermissions()throws Exception{
		PermissionsType p=remote.getPermissions();
		Permissions perm=XNJSFacade.getXNJSPermissions(p);
		getStorageAdapter().chmod(localFile, perm);
		if(logger.isDebugEnabled()){
			logger.debug("Copied remote permissions to :"+perm);
		}
	}


}
