package de.fzj.unicore.uas.xnjs;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.unigrids.services.atomic.types.GridFileType;
import org.unigrids.services.atomic.types.ProtocolType;
import org.unigrids.x2006.x04.services.sms.ImportFileDocument;
import org.unigrids.x2006.x04.services.sms.ImportFileResponseDocument;

import de.fzj.unicore.uas.UASProperties;
import de.fzj.unicore.uas.client.StorageClient;
import de.fzj.unicore.uas.fts.FiletransferOptions.IMonitorable;
import de.fzj.unicore.uas.fts.ProgressListener;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.uas.util.Pair;
import de.fzj.unicore.xnjs.Configuration;
import de.fzj.unicore.xnjs.ems.ExecutionException;
import de.fzj.unicore.xnjs.io.IStorageAdapter;
import de.fzj.unicore.xnjs.io.Permissions;
import de.fzj.unicore.xnjs.io.XnjsFile;

/**
 * Base class for UNICORE file exports<br/>
 * Sub classes need only provide an appropriate client class and 
 * the protocol that is used.
 *  
 * @author demuth
 * @author schuller
 */
public abstract class U6FileExportBase extends U6FileTransferBase {

	private StorageClient sms;

	private GridFileType gridFile;
	
	public U6FileExportBase(Configuration configuration){
		super(configuration);
	}
	
	public final void run(){
		try{
			checkCancelled();
			
			initSecurityProperties();

			sms=new StorageClient(smsEPR,sec);
			source=cleanLocalFilePath(source);
			localFile=source;

			status=Status.RUNNING;
			startTime=System.currentTimeMillis();
			
			if(isDirectory(source))
			{
				transferFolder(source, sms, target);
			}
			else
			{
				dataSize=computeFileSize(source);
				transferFile(source, sms, target);
			}
			status=Status.DONE;
			computeMetrics();
			destroyFileTransferResource();
		}catch(ProgressListener.CancelledException ce){
			String msg="File transfer cancelled.";
			status=Status.DONE; //TODO: CANCELLED does not exist
			statusMessage=msg;
		}
		catch(Exception ex){
			String msg="Error executing filetransfer: "+ex.getMessage();
			LogUtil.logException(msg,ex,logger);
			status=Status.FAILED;
			statusMessage=msg;
		}
		onFinishCleanup();
	}

	@Override
	protected void onFinishCleanup(){
		super.onFinishCleanup();
		sms=null;
		gridFile=null;
	}

	protected void transferFile(String source, StorageClient sms, String target) throws Exception
	{
		checkCancelled();
		target=checkIfTargetIsADirectory(source, sms, target);
		String remoteFSID=sms.getFileSystem().getDescription();
		String localFSID=getStorageAdapter().getFileSystemIdentifier();
		UASProperties conf = kernel.getAttribute(UASProperties.class);
		if(remoteFSID!=null && localFSID!=null && remoteFSID.equals(localFSID) && 
				!conf.getBooleanValue(UASProperties.SMS_TRANSFER_FORCEREMOTE)){
			transferFileLocal(source, sms, target);
		}
		else{
			transferFileFromRemote(source, sms, target);
		}
		copyPermissions(source, sms, target);
	}

	protected void transferFileLocal(String source, StorageClient sms, String target) throws Exception
	{
		checkCancelled();
		String base=sms.getFileSystem().getMountPoint();
		String targetUmask = sms.getResourcePropertiesDocument().getStorageProperties().getUmask();
		//if targetUmask is null now then fine - default will be used.
		
		if(base==null || OverwritePolicy.APPEND==overwrite){
			transferFileFromRemote(source, sms, target);
		}
		else{
			String tgt=base+"/"+target;
			String src=workdir+"/"+source;
			IStorageAdapter s=getStorageAdapter();
			String sRoot=s.getStorageRoot();
			String sUmask=s.getUmask();
			try{
				s.setStorageRoot("/");
				s.setUmask(targetUmask);
				checkOverwriteAllowed(s,tgt);
				createParentDirectories(tgt);
				logger.info("Optimization enabled: exporting file by local copy of <"+src+"> to <"+tgt+">");
				s.cp(src, tgt);
				transferredBytes+=s.getProperties(src).getSize();
			}finally{
				s.setStorageRoot(sRoot);
				s.setUmask(sUmask);
			}
		}
	}

	protected void transferFileFromRemote(String source, StorageClient sms, String target) throws Exception
	{
		checkCancelled();
		checkOverwriteAllowed(sms,target);
		ImportFileDocument req=ImportFileDocument.Factory.newInstance();
		req.addNewImportFile().setDestination(target);
		req.getImportFile().setProtocol(ProtocolType.Enum.forString(getProtocol()));
		boolean append=OverwritePolicy.APPEND.equals(overwrite);
		Map<String,String>ep=getExtraParameters();
		if(ep!=null && ep.size()>0){
			req.getImportFile().setExtraParameters(convert(ep));
		}
		req.getImportFile().setOverwrite(!append);
		ImportFileResponseDocument res=sms.ImportFile(req);
		fileTransferInstanceEpr=res.getImportFileResponse().getImportEPR();	
		localFile = source;
		ftc=getFTClient();
		ftc.setAppend(append);
		if(ftc instanceof IMonitorable){
			((IMonitorable)ftc).setProgressListener(this);
		}
		doRun();
	}

	/**
	 * checks if we may overwrite the given file (in case it exists)
	 * 
	 * @param fileName -  the file to check
	 * @throws IOException - if the file exists and DONT_OVERWRITE is requested
	 */
	protected void checkOverwriteAllowed(StorageClient sms, String fileName)throws Exception{
		if(OverwritePolicy.DONT_OVERWRITE.equals(overwrite)){
			GridFileType g=null;
			try{
				//will throw IOexception if file does not exist
				g=sms.listProperties(fileName);
			}catch(IOException ioe){}
			if(g!=null){
				throw new IOException("File <"+g.getPath()+"> exists, and we have been asked to not overwrite.");
			}
		}
	}

	/**
	 * checks if the export target is a directory, and returns a new target name
	 * that has the source file name appended
	 */
	protected String checkIfTargetIsADirectory(String source,StorageClient sms, String fileName)throws Exception{
		GridFileType g=null;
		try{
			//will throw IOexception if file does not exist
			g=sms.listProperties(fileName);
			if(g.getIsDirectory()){
				String name=getFileName(gridFile.getPath());
				return fileName+name;
			}
		}catch(IOException ioe){}
		return fileName;
	}

	protected void transferFolder(String source, StorageClient sms, String target) throws Exception
	{
		checkCancelled();
		List<Pair<String,String>> collection = new ArrayList<Pair<String,String>>();
		dataSize = collectFiles(collection, source, sms, target);

		for (Pair<String,String> pair : collection) {

			String currentSource = pair.getM1();
			String currentTarget = pair.getM2();
			if(isDirectory(currentSource))
			{
				checkOverwriteAllowed(sms,currentTarget);
				sms.createDirectory(currentTarget);
				transferredBytes++;
			}
			else transferFile(currentSource, sms, currentTarget);
		}
	}


	/**
	 * Recursively gather all files and directories that need to be copied
	 */
	protected long collectFiles(List<Pair<String,String>> collection, String sourceFolder, StorageClient sms, String targetFolder) throws Exception
	{
		long result = 1;
		collection.add(new Pair<String,String>(sourceFolder, targetFolder));
		for (XnjsFile child : getStorageAdapter().ls(sourceFolder)) {
			String relative = child.getPath().substring(sourceFolder.length()+1);
			String target = targetFolder+"/"+relative;

			if(child.isDirectory())
			{
				result += collectFiles(collection, child.getPath(), sms, target);
			}
			else 
			{
				collection.add(new Pair<String,String>(child.getPath(),target));
				result+= child.getSize();
			}
		}
		return result;
	}

	/**
	 * @param file - the file on the target system to check
	 * @throws IOException if the file does not exist
	 * @throws ExecutionException if the file properties can not be checked
	 */
	protected synchronized long computeFileSize(String file)throws ExecutionException,IOException {
		XnjsFile f=getStorageAdapter().getProperties(file);
		if(f!=null){
			return f.getSize();
		}
		else throw new IOException("The file <"+file+"> does not exist or can not be accessed.");
	}

	protected boolean isDirectory(String file) throws ExecutionException
	{
		XnjsFile f=getStorageAdapter().getProperties(file);
		if(f==null)throw new ExecutionException("The file <"+file+"> does not exist or can not be accessed.");
		return f.isDirectory();
	}


	@Override
	protected void doRun() throws Exception {
		InputStream is=getInputStream();
		try{
			checkCancelled();
			ftc.writeAllData(is);
		}
		finally{
			is.close();
		}
	}


	protected InputStream getInputStream()throws ExecutionException, IOException{
		return getStorageAdapter().getInputStream(localFile);
	}
	
	/**
	 * copies the permissions of the local file to the remote file
	 * @throws Exception
	 */
	protected void copyPermissions(String source, StorageClient sms, String target)throws Exception{
		Permissions p=getStorageAdapter().getProperties(source).getPermissions();
		sms.changePermissions(target, p.isReadable(), p.isWritable(), p.isExecutable());
	}

	protected synchronized GridFileType getGridFile(){
		if(gridFile==null){
			try{
				gridFile=sms.listProperties(target);
			}catch(Exception ex){}
		}
		return gridFile;
	}
}
