/*********************************************************************************
 * Copyright (c) 2006 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/

package de.fzj.unicore.uas.util;

import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.locks.Lock;

import org.apache.log4j.Logger;

import de.fzj.unicore.persist.impl.LockSupport;
import de.fzj.unicore.uas.SMSProperties;
import de.fzj.unicore.uas.StorageManagement;
import de.fzj.unicore.uas.UAS;
import de.fzj.unicore.uas.UASProperties;
import de.fzj.unicore.uas.impl.UASWSResourceImpl;
import de.fzj.unicore.uas.impl.sms.StorageDescription;
import de.fzj.unicore.uas.impl.sms.PathedStorageImpl;
import de.fzj.unicore.uas.impl.sms.SMSBaseImpl;
import de.fzj.unicore.uas.impl.sms.StorageManagementHomeImpl.StorageTypes;
import de.fzj.unicore.wsrflite.Home;
import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.wsrflite.exceptions.ResourceNotCreatedException;
import de.fzj.unicore.wsrflite.exceptions.ResourceUnknownException;
import de.fzj.unicore.wsrflite.xmlbeans.impl.WSResourceImpl;

/**
 * create a storage server "default_storage" pointing to a 
 * configurable work directory <br/>
 * Parameters:
 * <ul>
 * <li><code>defaultsms.workdir</code> : the base directory (on the TSI machine)</li>
 * <li><code>defaultsms.name</code>    : the name to use for this storage (default: "SHARE")</li>
 * <li><code>defaultsms.class</code>   : (optional) the custom class to use for this storage</li>
 * </ul>
 * 
 * Run on startup by setting or adding to the property "uas.onstartup"
 * 
 * TODO want full {@link SMSProperties} config on this?
 * 
 * @author schuller
 */
public class CreateSMSOnStartup implements Runnable{

	private static final Logger logger=LogUtil.getLogger(LogUtil.UNICORE,CreateSMSOnStartup.class);

	public static final String INSTANCE_ID="default_storage";

	private final Kernel kernel;
	
	public CreateSMSOnStartup(Kernel kernel){
		this.kernel=kernel;
	}

	public void run() {
		try { 
			Home smsHome=kernel.getHome(UAS.SMS);
			if(smsHome==null){
				logger.info("No SMS configured for this site.");
				return;
			}
			//get "global" lock
			LockSupport ls=kernel.getPersistenceManager().getLockSupport();
			Lock smsLock=ls.getOrCreateLock(CreateSMSOnStartup.class.getName());
			if(smsLock.tryLock()){
				try{
					UASWSResourceImpl sms=(UASWSResourceImpl)smsHome.get(INSTANCE_ID);
					//it exists, so force re-publish
					sms.publish();
					return;
				}catch(ResourceUnknownException e){
					//ok, does not exist
				}
				try{
					doCreateSMS(smsHome);
				}finally{
					smsLock.unlock();
				}
			}
		}catch(Exception e){
			logger.error("Could not add "+INSTANCE_ID,e);

		}
	}
	
	private void doCreateSMS(Home smsHome)throws ResourceNotCreatedException{
		Map<String,Object>map=new HashMap<String,Object>();
		map.put(WSResourceImpl.INIT_UNIQUE_ID,INSTANCE_ID);
		//set lifetime to 72 months
		Calendar c1=new GregorianCalendar();
		c1.add(Calendar.MONTH,72);
		map.put(WSResourceImpl.INIT_INITIAL_TERMINATION_TIME,c1);
		map.put(UASWSResourceImpl.INITPARAM_PUBLISH_TO_REGISTRY,Boolean.TRUE);

		
		UASProperties config = kernel.getAttribute(UASProperties.class);
		String name=config.getValue(UASProperties.DEFSMS_NAME);
		String dir=config.getValue(UASProperties.DEFSMS_WORKDIR);
                String protocols=config.getValue(UASProperties.DEFSMS_CONF_PROTOCOLS);
		if(dir==null)
			dir=System.getProperty("java.io.tmpdir");
		Class<? extends StorageManagement> clazz = config.getClassValue(UASProperties.DEFSMS_CLASS, StorageManagement.class);
		StorageTypes type = clazz == null ? StorageTypes.VARIABLE : StorageTypes.CUSTOM;
		
		StorageDescription description = new StorageDescription(INSTANCE_ID, name, dir, type, clazz, 
				protocols, true, false, false, null, "Default site storage", 
				null);
		map.put(SMSBaseImpl.INIT_STORAGE_DESCRIPTION, description);

		//skip initial path resolution (avoids problems due to missing client context)
		map.put(PathedStorageImpl.INIT_SKIP_RESOLVE, Boolean.TRUE);
		
		smsHome.createWSRFServiceInstance(map);
		StringBuilder msg=new StringBuilder();
		msg.append("Added "+INSTANCE_ID+" resource to service "+UAS.SMS+"\nUsing work dir: "+dir);
		if(clazz!=null)msg.append("\nUsing custom class <"+clazz+">");
		logger.info(msg.toString());
	}

}
