/*********************************************************************************
 * Copyright (c) 2008 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DEVELOPED IN THE CONTEXT OF THE OMII-EUROPE PROJECT.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/

package de.fzj.unicore.uas.security;

import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.codehaus.xfire.MessageContext;
import org.codehaus.xfire.handler.AbstractHandler;
import org.codehaus.xfire.handler.Phase;
import org.jdom.Element;
import org.jdom.Namespace;

import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.wsrflite.KernelInjectable;
import de.fzj.unicore.wsrflite.security.util.AttributeHandlingCallback;
import de.fzj.unicore.wsrflite.xfire.security.AccessControlHandler;
import eu.unicore.security.SecurityTokens;
import eu.unicore.security.xfireutil.AuthInHandler;

/**
 * reads a proxy cert (incl. private key) in PEM format from
 * a SOAP header and stores it in the security context
 * 
 * @author schuller
 */
public class ProxyCertInHandler extends AbstractHandler implements KernelInjectable
{
	private static final Logger logger = LogUtil.getLogger(LogUtil.SECURITY,ProxyCertInHandler.class);
	
	//proxy header has this namespace
	public static final Namespace PROXY_NS=Namespace.getNamespace("http://www.unicore.eu/unicore6");

	//Element name
	public static final String PROXY="Proxy";

	private Kernel kernel;
	
	public void setKernel(Kernel kernel){
		this.kernel=kernel;
	}
	
	public ProxyCertInHandler(Kernel kernel){
		this.kernel=kernel;	
		setPhase(Phase.POLICY);
		after(AuthInHandler.class.getName());
		before(AccessControlHandler.class.getName());
		addAttributeCallback();
	}

	//add callback that extracts attributes when creating a client
	private void addAttributeCallback(){
		kernel.getSecurityManager().addCallback(getCallback());
	}
	
	
	/**
	 * Retrieves the proxy cert from the SOAP Header and puts it
	 * into the security context for later evaluations
	 */
	@SuppressWarnings("rawtypes")
	public void invoke(MessageContext ctx) throws Exception
	{
		// get the SOAP header
		Element header = ctx.getInMessage().getHeader();
		if (header == null){
			return ;	
		} 

		// get proxy header extensions
		List extensions = header.getChildren(PROXY, PROXY_NS);
		if(extensions.size()==0){
			logger.debug("No Proxy cert header found.");
			return;
		}
		Element proxyElement = (Element) extensions.get(0);		
		String proxy=proxyElement.getText();
		if(logger.isDebugEnabled()){
			logger.debug("Extracted Proxy header:\n"+proxy);		
		}

		// get the security context
		SecurityTokens tokens=((SecurityTokens)ctx.getProperty(SecurityTokens.KEY));		
		if (tokens==null){
			//should never happen in production (always have AuthInHandler)
			logger.warn("No security tokens found, new ones created...");			
			tokens=new SecurityTokens();
			ctx.setProperty(SecurityTokens.KEY, tokens);
		}
		tokens.getContext().put(PROXY, proxy);			
	}
	
	private static AttributeHandlingCallback aac;
	
	private synchronized AttributeHandlingCallback getCallback(){
		if(aac==null){
			aac=new ProxyCertAttributeCallback();
		}
		return aac;
	}
	
	/**
	 * callback functions
	 */
	private static class ProxyCertAttributeCallback implements AttributeHandlingCallback{
		/**
		 * gets the proxy cert from the sec tokens and returns it as an attribute
		 * which will be stored in the client
		 */
		public Map<String,Serializable> extractAttributes(SecurityTokens tokens) {
			Map<String,Serializable> result = new HashMap<String,Serializable>();
			String proxy= (String)tokens.getContext().get(PROXY);
			if(proxy!=null){
				result.put(PROXY , proxy);
			}
			return result;
		}
	}
	
}



