package de.fzj.unicore.uas.metadata;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Future;

/**
 * basic metadata interface, used to perform metadata operations on file resources
 * 
 * @author Waqas Noor
 * @author schuller
 */
public interface MetadataManager {

    public static final String META_PROPERTIES = "metaproperties";
    public static final String MD_FILE_PREFIX = ".unicore_metadata";

    /**
     * Create Metadata on given resource.
     * @param resourceName	
     * @param metadata	metadata properties
     * 
     * @throws IOException
     */
    public void createMetadata(String resourceName, Map<String, String> metadata) throws IOException;

    /**
     * This method updates the Metadata. If UniMetadata contains any existing property then it will take the newer one.
     * @param resourceName The resource who has the metadata
     * @param metadata map of properties
     * 
     * @throws Exception
     */
    public void updateMetadata(String resourceName, Map<String, String> metadata) throws Exception;

    /**
     * This method search the Index against the provided search string.
     * @param searchString search string space is used as delimeter.
     * @param isAdvancedSearch If advance search is true then query is translated to boolean/range query.
     * @return
     */
    public List<SearchResult> searchMetadataByContent(String searchString, boolean isAdvancedSearch) throws Exception;

    /**
     * This method retrieves the metadata
     * @param resourceName
     * @return
     */
    public Map<String, String> getMetadataByName(String resourceName) throws Exception;

    /**
     * This method removes the metadata from both the index and the storage.
     * @param resourceName
     */
    public void removeMetadata(String resourceName) throws Exception;

    /**
     * rename a resource
     * @param source
     * @param target
     * @param storage
     */
    public void renameResource(String source, String target) throws Exception;

    /**
     * copy metadata
     * @param source
     * @param target
     * @param storage
     * @param pathToResource
     */
    public void copyResourceMetadata(String source, String target) throws Exception;

    /**
     * This method asynchronously crawls the file system and create the metadata. 
     * @param base the path where crawler starts it job.
     * @param depthLimit The number of level, crawler do its job. It should be greater than 0.
     * @return a {@link Future} for monitoring execution and potentially canceling, or <code>null</code> if
     * this functionality is not provided by the underlying code
     */
    public Future<ExtractionStatistics> startAutoMetadataExtraction(String base, int depthLimit) throws Exception;
}
