/*********************************************************************************
 * Copyright (c) 2006 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
 

package de.fzj.unicore.uas.impl.tss.rp;

import java.util.ArrayList;
import java.util.List;

import org.unigrids.services.atomic.types.TextInfoDocument;
import org.unigrids.services.atomic.types.TextInfoType;

import de.fzj.unicore.uas.xnjs.XNJSFacade;
import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.wsrflite.Resource;

/**
 * exposes the TextInfo fields defined in the XNJS IDB<br>
 * 
 * @author schuller
*/
public class TextInfoResourceProperty extends IDBBasedResourceProperty<TextInfoType[]> {
	
	private static final long serialVersionUID=20348L;
	
	private List<TextInfoType> otherInfos;
	
	private TextInfoDocument[] xdoc;
	
	public TextInfoResourceProperty(Resource parent,String xnjsReference) {
		super(parent,xnjsReference);
		otherInfos=new ArrayList<TextInfoType>();
		xdoc = new TextInfoDocument[0];
	}

	/**
	 * updates text info from NJS
	 */
	@Override
	public TextInfoResourceProperty update(){
		try{
			Kernel kernel=parentWSResource.getKernel();
			setProperty(XNJSFacade.get(xnjsReference,kernel).getDefinedTextInfo());
			return this;
		}catch(Exception e){
			logger.fatal("Could not update textinfo resourceproperty, information not available.",e);
			return null;
		}
	}
	
	/**
	 * @returns TextInfoType[] text info
	 */
	public TextInfoType[] getProperty() {
		TextInfoType[] infos=new TextInfoType[xdoc.length];
		for(int i=0;i<xdoc.length;i++){
			infos[i]=xdoc[i].getTextInfo();
		}
		return infos;
	}
	
	public TextInfoDocument[] getXml() {
		return xdoc;
	}

	/**
	 * sets the array of TextInfoType
	 * 
	 * @param TextInfoType[]
	 */
	public void setProperty(TextInfoType[] textInfo) {
		xdoc=new TextInfoDocument[textInfo.length+otherInfos.size()];
		for(int i=0;i<textInfo.length;i++){
			xdoc[i]=TextInfoDocument.Factory.newInstance();
			xdoc[i].setTextInfo(textInfo[i]);
		}
		for(int i=0;i<otherInfos.size();i++){
			xdoc[i+textInfo.length]=TextInfoDocument.Factory.newInstance();
			xdoc[i+textInfo.length].setTextInfo(otherInfos.get(i));
		}
	}


	/**
	 * adds a textinfo
	 *  
	 * @param textInfo
	 */
	public void add(TextInfoType textInfo){
		otherInfos.add(textInfo);
		update();
	}

	/**
	 * checks if a textinfo of the given name exists already
	 * @param name
	 * @return true if exists, false otherwise
	 */
	public boolean contains(String name){
		for(TextInfoDocument t: xdoc){
			if(name.equals(t.getTextInfo().getName()))return true;
		}
		return false;
	}
	
	/**
	 * gets the value of the named textinfo 
	 * @param name - the name of the textinfo
	 * @return the value, or null if the requested textinfo does not exist
	 */
	public String getInfo(String name){
		for(TextInfoDocument t: xdoc){
			if(name.equals(t.getTextInfo().getName()))return t.getTextInfo().getValue();
		}
		return null;
	}

	
}
