/*********************************************************************************
 * Copyright (c) 2006-2009 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
 

package de.fzj.unicore.uas.impl.tss.rp;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlObject;
import org.unigrids.x2006.x04.services.tss.JobReferenceDocument;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.UAS;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.wsrflite.Resource;
import de.fzj.unicore.wsrflite.utils.WSServerUtilities;
import de.fzj.unicore.wsrflite.xmlbeans.ResourceProperty;

/**
 * Holds a set of references to Jobs on a TargetSystemService<br/>
 * 
 * @author schuller
 */
public class JobReferenceResourceProperty extends ResourceProperty<EndpointReferenceType[]>{
	
	private static final long serialVersionUID=2L;

	private static final Logger logger=LogUtil.getLogger(LogUtil.SERVICES,JobReferenceResourceProperty.class);
	
	private final ArrayList<String> uuids;
	
	private String JMS_BASE;
	
	public JobReferenceResourceProperty(Resource parent){
		super(parent);
		uuids=new ArrayList<String>();
	}
		
	/**
	 * @param EndpointReferenceType[]
	 */
	@Override
	public void setProperty(EndpointReferenceType[] types) {
		synchronized(uuids){
			uuids.clear();
			for(int i=0;i<types.length;i++){
				uuids.add(WSServerUtilities.extractResourceID(types[i]));
			}
		}
	}

	/**
	 * size limit for job list returned by {@link #getXml()}. New client code should 
	 * use the chunked version {@link #getXml(int, int)} 
	 */
	public static final int LIMIT=100;
	
	/**
	 * returns list of job references, (since 6.3.0) LIMITED 
	 * to {@link #LIMIT} entries
	 */
	@Override
	public JobReferenceDocument[] getXml() {
		synchronized (uuids) {
			JMS_BASE=WSServerUtilities.makeAddress(UAS.JMS, "", 
					parentWSResource.getKernel().getContainerProperties());
			JobReferenceDocument[] xdoc=new JobReferenceDocument[Math.min(LIMIT,uuids.size())];
			for(int i=0;i<uuids.size()&&i<LIMIT;i++){
				xdoc[i]=JobReferenceDocument.Factory.newInstance();
				xdoc[i].setJobReference(makeEPR(uuids.get(i)));
			}
			return xdoc;
		}
		
	}
	
	@Override
	public List<XmlObject> getXml(int offset, int length)
			throws IndexOutOfBoundsException {
		List<XmlObject>res=new ArrayList<XmlObject>();
		synchronized (uuids) {
			JMS_BASE=WSServerUtilities.makeAddress(UAS.JMS, "", 
					parentWSResource.getKernel().getContainerProperties());
			if(offset>=uuids.size())throw new IndexOutOfBoundsException("Requested offset is too large, only have <"+uuids.size()+"> results.");
			for(int i=offset;i<offset+length&&i<uuids.size();i++){
				JobReferenceDocument r=JobReferenceDocument.Factory.newInstance();
				r.setJobReference(makeEPR(uuids.get(i)));
				res.add(r);
			}
			return res;
		}
	}

	@Override
	public int getNumberOfElements(){
		return uuids.size();
	}
	
	public void add(EndpointReferenceType epr){
		synchronized(uuids){
			uuids.add(WSServerUtilities.extractResourceID(epr));
		}
	}
	
	/**
	 * remove an entry by ID
	 * @param id - the id of the TSS to remove
	 */
	public void remove(String id){
		synchronized (uuids) {
			Iterator<String> it=uuids.iterator();
			while(it.hasNext()){
				try{
					if(it.next().equals(id)){
						it.remove();
						logger.debug("Removed "+id);
					}
				}catch(Exception e){
					logger.warn(e.getMessage());
				}
			}
		}
	}
	
	@Override
	public EndpointReferenceType[] getProperty() {
		EndpointReferenceType[] res=new EndpointReferenceType[uuids.size()];
		int i=0;
		synchronized(uuids){
			JMS_BASE=WSServerUtilities.makeAddress(UAS.JMS, "", 
					parentWSResource.getKernel().getContainerProperties());
			for(String uuid: uuids){
				res[i]=makeEPR(uuid);
			}
		}
		return res;
	}
	
	protected EndpointReferenceType makeEPR(String id){
		EndpointReferenceType epr=EndpointReferenceType.Factory.newInstance();
		epr.addNewAddress().setStringValue(JMS_BASE+id);
		return epr;
	}
	
	public List<String>getUUIDs(){
		return uuids;
	}
	
	public void addAll(List<String>ids){
		synchronized(uuids){
			uuids.addAll(ids);
		}
	}
	
}
