/*********************************************************************************
 * Copyright (c) 2006 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
 

package de.fzj.unicore.uas.impl.tss.rp;

import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlObject;
import org.unigrids.services.atomic.types.MetadataDocument;
import org.unigrids.x2006.x04.services.tss.ApplicationResourceDocument;
import org.unigrids.x2006.x04.services.tss.ApplicationResourceType;

import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.uas.xnjs.XNJSFacade;
import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.wsrflite.Resource;
import de.fzj.unicore.wsrflite.utils.WSServerUtilities;
import de.fzj.unicore.xnjs.tsi.ApplicationInfo;

/**
 * represents the applications on a TargetSystem resource<br>
 * 
 * @author schuller
 */
public class ApplicationsResourceProperty extends IDBBasedResourceProperty<ApplicationResourceType[]> {

	private static final long serialVersionUID=20348L;

	private static final Logger logger=LogUtil.getLogger(LogUtil.SERVICES,ApplicationsResourceProperty.class);
	   
	private ApplicationResourceType[] apps;

	private ApplicationResourceDocument[] xdoc;
	
	public ApplicationsResourceProperty(Resource parent,String xnjsReference) {
		super(parent,xnjsReference);
		apps = new ApplicationResourceType[0];
		xdoc = new ApplicationResourceDocument[0];
	}

	/**
	 * updates application information from NJS
	 */
	@Override
	public synchronized ApplicationsResourceProperty update(){
		try{
			if(mustUpdate()){
				Kernel kernel=parentWSResource.getKernel();
				ApplicationInfo[] xnjsapps=XNJSFacade.get(xnjsReference, kernel).getDefinedApplications();
				ApplicationResourceType[] apps=new ApplicationResourceType[xnjsapps.length];
				for(int i=0;i<apps.length;i++){
					apps[i]=convertXNJSApp(xnjsapps[i]);
				}
				logger.debug("Have "+apps.length+" applications on XNJS");
				setProperty(apps);
			}
			return this;
		}catch(Exception e){
			LogUtil.logException("Could not update applications resourceproperty, information not available.",e,logger);
			return null;
		}
	}
	
	/**
	 * @returns ApplicationResourceType[] apps
	 */
	public ApplicationResourceType[] getProperty() {
		return apps;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.fzj.unicore.wsrflite.ResourceProperty#getXml()
	 */
	public ApplicationResourceDocument[] getXml() {
		return xdoc;
	}

	/**
	 * sets the array of applications
	 * 
	 * @param ApplicationResourceType[]
	 */
	public void setProperty(ApplicationResourceType[] apps) {
		xdoc=new ApplicationResourceDocument[apps.length];
		for(int i=0;i<apps.length;i++){
			xdoc[i]=ApplicationResourceDocument.Factory.newInstance();
			xdoc[i].setApplicationResource(apps[i]);
		}
	}

	
	public ApplicationResourceType convertXNJSApp(ApplicationInfo xnjsApp){
		ApplicationResourceType result=ApplicationResourceType.Factory.newInstance();
		result=ApplicationResourceType.Factory.newInstance();
		result.setApplicationName(xnjsApp.getApplicationName());
		result.setApplicationVersion(xnjsApp.getApplicationVersion());
		result.setDescription(xnjsApp.getApplicationDescription());
		XmlObject meta=xnjsApp.getMetadata();
		if(meta!=null){
			MetadataDocument mdd=MetadataDocument.Factory.newInstance();
			mdd.addNewMetadata();
			WSServerUtilities.append(meta, mdd);
			result.setMetadata(mdd.getMetadata());
		}
		return result;
	}

}
