/*********************************************************************************
 * Copyright (c) 2006 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/


package de.fzj.unicore.uas.impl.tss;

import java.util.Calendar;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import javax.xml.namespace.QName;

import org.apache.log4j.Logger;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.JobDefinitionDocument;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.JobDefinitionType;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.ResourcesDocument;
import org.oasisOpen.docs.wsrf.rl2.TerminationTimeDocument.TerminationTime;
import org.unigrids.services.atomic.types.ServiceStatusDocument;
import org.unigrids.services.atomic.types.StorageReferenceType;
import org.unigrids.services.atomic.types.StorageTypeEnumeration;
import org.unigrids.x2006.x04.services.reservation.ResourceReservationRequestDocument;
import org.unigrids.x2006.x04.services.reservation.ResourceReservationResponseDocument;
import org.unigrids.x2006.x04.services.tss.JobReferenceEnumerationDocument;
import org.unigrids.x2006.x04.services.tss.SubmitDocument;
import org.unigrids.x2006.x04.services.tss.SubmitResponseDocument;
import org.unigrids.x2006.x04.services.tss.SupportsReservationDocument;
import org.unigrids.x2006.x04.services.tss.TargetSystemPropertiesDocument;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.Enumeration;
import de.fzj.unicore.uas.JobManagement;
import de.fzj.unicore.uas.ReservationManagement;
import de.fzj.unicore.uas.ResourceReservation;
import de.fzj.unicore.uas.StorageManagement;
import de.fzj.unicore.uas.TargetSystem;
import de.fzj.unicore.uas.UAS;
import de.fzj.unicore.uas.UASProperties;
import de.fzj.unicore.uas.client.BaseUASClient;
import de.fzj.unicore.uas.impl.ServiceStateResourceProperty;
import de.fzj.unicore.uas.impl.UASWSResourceImpl;
import de.fzj.unicore.uas.impl.UmaskResourceProperty;
import de.fzj.unicore.uas.impl.UmaskResourceProperty.UmaskChangedListener;
import de.fzj.unicore.uas.impl.bp.BPSupportImpl;
import de.fzj.unicore.uas.impl.enumeration.EnumerationImpl;
import de.fzj.unicore.uas.impl.job.JobManagementImpl;
import de.fzj.unicore.uas.impl.reservation.ReservationManagementImpl;
import de.fzj.unicore.uas.impl.sms.StorageDescription;
import de.fzj.unicore.uas.impl.sms.SMSBaseImpl;
import de.fzj.unicore.uas.impl.sms.StorageManagementHomeImpl.StorageTypes;
import de.fzj.unicore.uas.impl.tss.rp.ApplicationsResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.AvailableResourcesRP;
import de.fzj.unicore.uas.impl.tss.rp.CPUCountResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.CPUTimeResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.ExecutionEnvironmentRP;
import de.fzj.unicore.uas.impl.tss.rp.ExtendedSecurityInfoResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.JobReferenceResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.MemoryPerNodeResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.NameResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.NodesResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.NumberOfJobsProperty;
import de.fzj.unicore.uas.impl.tss.rp.OperatingSystemResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.ProcessorResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.ReservationReferenceResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.SiteSpecificResourcesRP;
import de.fzj.unicore.uas.impl.tss.rp.StorageReferenceResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.TextInfoResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.TotalCPUsResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.UpSinceResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.XLoginResourceProperty;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.uas.xnjs.XNJSFacade;
import de.fzj.unicore.wsrflite.Home;
import de.fzj.unicore.wsrflite.exceptions.InvalidModificationException;
import de.fzj.unicore.wsrflite.exceptions.ResourceNotCreatedException;
import de.fzj.unicore.wsrflite.exceptions.ResourceUnknownException;
import de.fzj.unicore.wsrflite.exceptions.UnableToSetTerminationTimeException;
import de.fzj.unicore.wsrflite.messaging.Message;
import de.fzj.unicore.wsrflite.messaging.PullPoint;
import de.fzj.unicore.wsrflite.messaging.ResourceDeletedMessage;
import de.fzj.unicore.wsrflite.persistence.Persist;
import de.fzj.unicore.wsrflite.utils.Utilities;
import de.fzj.unicore.wsrflite.utils.WSServerUtilities;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import de.fzj.unicore.wsrflite.xmlbeans.ResourceProperty;
import de.fzj.unicore.wsrflite.xmlbeans.rp.ImmutableResourceProperty;
import de.fzj.unicore.xnjs.ems.Action;
import de.fzj.unicore.xnjs.io.IStorageAdapter;
import de.fzj.unicore.xnjs.tsi.TSI;
import eu.unicore.security.Client;
import eu.unicore.util.Log;

/**
 * The implementation of the TargetSystem service.<br/>
 * 
 * It uses an XNJS instance as back end. The XNJS is configured 
 * through a config file, which is referenced in the global 
 * UNICORE/X config file<br/>
 * 
 * Each target system instance has a HOME storage attached, which will expose
 * the current user's home directory.<br/>
 * 
 * Further configuration options:
 * <ul>
 * 	<li>attach more storages, @see {@link AddOnStorageDescriptionFactory}</li>
 * </ul>
 * 
 * @author schuller
 */
public class TargetSystemImpl extends UASWSResourceImpl implements TargetSystem {

	private static final Logger logger=LogUtil.getLogger(LogUtil.SERVICES, TargetSystemImpl.class);

	/**
	 * initialisation parameter giving the unique ID of the TSF 
	 * that created this TSS
	 */
	public static final String INIT_TSF_ID=TargetSystemImpl.class.getName()+".tsfid";

	/**
	 * initialisation parameter giving the name of this TSS
	 * if not given, the default name (defined in config file) is used
	 */
	public static final String INIT_TSS_NAME="TSS_NAME";

	/**
	 * initialisation parameter giving the id of the XNJS to be used
	 * if not given, the default one is used
	 */
	public static final String INIT_XNJS_REFERENCE="XNJS_REFERENCE";

	@Persist
	private String tsfID;

	@Override
	protected void addWSResourceInterfaces(BPSupportImpl baseProfile) {
		super.addWSResourceInterfaces(baseProfile);
		baseProfile.addWSResourceInterface(TSS_PORT);
		baseProfile.addWSResourceInterface(ResourceReservation.PORTTYPE);
	}

	@Override
	public QName getResourcePropertyDocumentQName() {
		return TargetSystemPropertiesDocument.type.getDocumentElementName();
	}

	@Override
	public void customPostActivate(){
		//check for deleted jobs/reservations and remove them...
		try{
			logger.trace("Getting messages from queue "+getUniqueID());
			PullPoint p=kernel.getMessaging().getPullPoint(getUniqueID());
			JobReferenceResourceProperty jrp=(JobReferenceResourceProperty)properties.get(RPJobReference);
			ReservationReferenceResourceProperty rrp=(ReservationReferenceResourceProperty)properties.get(RPReservationReference);
			while(p.hasNext()){
				Message message=p.next();
				if(message instanceof ResourceDeletedMessage){
					ResourceDeletedMessage rdm=(ResourceDeletedMessage)message;
					String id=rdm.getDeletedResource();
					String service=rdm.getServiceName();
					if(UAS.JMS.equals(service)){
						jrp.remove(id);
						setDirty();	
					}
					else if(UAS.RESERVATIONS.equals(service)){
						rrp.remove(id);
						setDirty();
					}
				}
			}
		}catch(Exception e){
			LogUtil.logException(e.getMessage(),e,logger);
		}
		UmaskResourceProperty umaskRP = (UmaskResourceProperty) properties.get(StorageManagement.RPUmask);
		if(umaskRP==null){//can happen after server update
			umaskRP=addUmaskRP(Integer.toOctalString(IStorageAdapter.DEFAULT_UMASK));
		}
		umaskRP.setListener(new UmaskListenerImpl());
	}



	public SubmitResponseDocument Submit(SubmitDocument in) throws BaseFault {
		if( !((TargetSystemHomeImpl)home).isJobSubmissionEnabled()){
			throw BaseFault.createFault(((TargetSystemHomeImpl)home).getHighMessage());
		}
		if(logger.isTraceEnabled()){
			logger.trace("submit: "+in.toString());
		}
		try {
			JobDefinitionType jsdl = in.getSubmit().getJobDefinition();
			JobDefinitionDocument doc = JobDefinitionDocument.Factory
					.newInstance();
			doc.setJobDefinition(jsdl);

			Map<String,Object> map=new HashMap<String,Object>();

			TerminationTime itt=in.getSubmit().getTerminationTime();
			if(itt!=null && itt.getCalendarValue()!=null){
				Calendar jobTT=itt.getCalendarValue();
				map.put(INIT_INITIAL_TERMINATION_TIME, jobTT);
				checkAndExtendLT(jobTT);
			}

			Action action = XNJSFacade.get(xnjsReference, kernel).makeAction(doc);
			action.setUmask(getUmask());
			if(in.getSubmit().getAutoStartWhenReady()){
				action.getProcessingContext().put(Action.AUTO_SUBMIT, Boolean.TRUE);
			}
			map.put(JobManagementImpl.INITPARAM_ACTION,action);
			map.put(JobManagementImpl.INITPARAM_TSS_UNIQUEID, getUniqueID());
			map.put(INITPARAM_XNJS_REFERENCE, xnjsReference);
			map.put(INIT_PARENT_NODE, nodeHelper);

			EndpointReferenceType epr = createJob(map);

			SubmitResponseDocument response = SubmitResponseDocument.Factory.newInstance();
			response.addNewSubmitResponse().setJobReference(epr);

			setDirty();
			return response;
		} catch (Exception e) {
			LogUtil.logException("Error submitting.",e,logger);
			throw BaseFault.createFault("Error submitting: "+e.getMessage(),e);
		}
	}

	/*
	 * if user submits a job with a tt longer than the TSS lifetime
	 * automatically extend the TSS lifetime
	 */
	protected void checkAndExtendLT(Calendar newTT)throws UnableToSetTerminationTimeException{
		if(getTerminationTime().compareTo(newTT)<0){
			logger.debug("Job termination time exceeds TSS termination time, extending TSS lifetime...");
			Calendar tt=(Calendar)newTT.clone();
			tt.add(Calendar.DATE, 1);
			setTerminationTime(tt);
		}
	}


	/*
	 * (non-Javadoc)
	 * 
	 * @see de.fzj.unicore.wsrflite.impl.WSResourceImpl#initialise(de.fzj.unicore.wsrflite.Home,
	 *      java.lang.Object[])
	 */
	@Override
	public void initialise(String name, Map<String,Object>initobjs) throws Exception{
		super.initialise(name, initobjs);

		tsfID=(String)initobjs.get(INIT_TSF_ID);
		// add our properties
		properties.put(RPNumberOfJobs, new NumberOfJobsProperty(this,xnjsReference));
		String tssName=(String)initobjs.get(INIT_TSS_NAME);
		if (tssName != null)
			properties.put(RPName, new NameResourceProperty(this,tssName));
		else
			properties.put(RPName, new NameResourceProperty(this));
		properties.put(RPApplication, createApplicationResourceProperty());
		properties.put(RPProcessor, createProcessorResourceProperty());
		properties.put(RPTextInfo, new TextInfoResourceProperty(this,xnjsReference));

		//reservation support?
		SupportsReservationDocument supportsReservation=SupportsReservationDocument.Factory.newInstance();
		supportsReservation.setSupportsReservation(XNJSFacade.get(xnjsReference, kernel).supportsReservation());
		properties.put(ResourceReservation.RP_SUPPORTS_RESERVATION,new ImmutableResourceProperty(supportsReservation));

		if(!uasProperties.getBooleanValue(UASProperties.TSS_DISABLE_HOME_PROPERTY)){
			createHomeStorage();
		}
		createAdditionalStorages();

		properties.put(RPJobReference, createJobReferenceProperty());
		properties.put(RPReservationReference, createReservationReferenceProperty());

		properties.put(RPOperatingSystem, createOSResourceProperty());
		properties.put(RPCPUTime, new CPUTimeResourceProperty(this,xnjsReference));
		properties.put(RPNodeCount, new NodesResourceProperty(this,xnjsReference));
		properties.put(RPMemoryPerNode, new MemoryPerNodeResourceProperty(this,xnjsReference));
		properties.put(RPCPUCount, new CPUCountResourceProperty(this,xnjsReference));
		properties.put(RPTotalCPUCount, new TotalCPUsResourceProperty(this,xnjsReference));
		properties.put(RPUpSince, new UpSinceResourceProperty());
		properties.put(RPSiteSpecificResources, new SiteSpecificResourcesRP(this,xnjsReference));
		properties.put(RPAvailableResources, new AvailableResourcesRP(this,xnjsReference));
		properties.put(RPExecutionEnvironments, new ExecutionEnvironmentRP(this,xnjsReference));
		properties.put(ServiceStatusDocument.type.getDocumentElementName(), new ServiceStateResourceProperty(this));
		properties.put(RPXlogin, new XLoginResourceProperty(this));
		addUmaskRP(Integer.toOctalString(IStorageAdapter.DEFAULT_UMASK));

		try{
			JobReferenceEnumerationDocument jred=JobReferenceEnumerationDocument.Factory.newInstance();
			jred.setJobReferenceEnumeration(createJobListEnumeration());
			properties.put(RPJobReferenceEnumeration, new ImmutableResourceProperty(jred));
		}
		catch(Exception ex){
			LogUtil.logException("Error creating job reference enumeration",ex,logger);
		}

		//publish this in registry?
		Boolean publishFlag=(Boolean)initobjs.get(INITPARAM_PUBLISH_TO_REGISTRY);
		if(Boolean.TRUE.equals(publishFlag)){
			logger.info("Publishing TargetSystem <"+getUniqueID()+"> to registry.");
			publish();
		}
		setResourceStatus(ResourceStatus.INITIALIZING);
		setStatusMessage("OK");

		Client c=getClient();
		//re-connect any old JMS instances
		RecreateJMSReferenceList rj=new RecreateJMSReferenceList(kernel, getUniqueID(),c);
		//re-generate JMS instances from XNJS actions
		GenerateJMSInstances rx=new GenerateJMSInstances(kernel, getUniqueID(),c,xnjsReference);
		//re-generate Reservation list
		RecreateReservationReferenceList rr=new RecreateReservationReferenceList(kernel, getUniqueID(),c);
		kernel.getContainerProperties().getThreadingServices().getScheduledExecutorService().schedule(
				new TSSAsynchInitialisation(kernel, getUniqueID(), rj, rx, rr), 200, TimeUnit.MILLISECONDS);
	}

	@Override
	protected ResourceProperty<?> getSecurityInfoResourceProperty(boolean addServerCert) {
		return new ExtendedSecurityInfoResourceProperty(this, addServerCert);
	}

	//create an umask rp and add it to the properties map
	protected UmaskResourceProperty addUmaskRP(String umask){
		UmaskResourceProperty umaskRP = new UmaskResourceProperty(this,umask);
		umaskRP.setListener(new UmaskListenerImpl());
		properties.put(StorageManagement.RPUmask, umaskRP);
		return umaskRP;
	}

	//create the Home SMS
	//TODO need configuration of this storage?
	protected void createHomeStorage(){
		Map<String,Object>initMap=new HashMap<String,Object>();
		String uuid=createStorageID("Home");
		initMap.put(SMSBaseImpl.INIT_UNIQUE_ID,uuid);
		StorageDescription storageDescription = new StorageDescription("HomeOf-"+getUniqueID(), "Home", 
				null, StorageTypes.HOME, null, null, false, false, false, 
				null, "Home storage", null);
		initMap.put(SMSBaseImpl.INIT_STORAGE_DESCRIPTION, storageDescription);
		//set "infinite" tt, because the SMS will be destroyed together with this TSS
		Calendar tt=Calendar.getInstance();
		tt.add(Calendar.YEAR,10);
		initMap.put(SMSBaseImpl.INIT_INITIAL_TERMINATION_TIME,tt);
		initMap.put(INITPARAM_XNJS_REFERENCE, xnjsReference);
		initMap.put(SMSBaseImpl.INIT_PARENT_NODE, nodeHelper);
		
		EndpointReferenceType homeEPR = createStorageManagement("HOME", initMap);
		if (homeEPR == null)
			return;
		StorageReferenceType srt=StorageReferenceType.Factory.newInstance();
		srt.setStorageEndpointReference(homeEPR);
		srt.setType(StorageTypeEnumeration.HOME);
		getOrCreateStorageReferenceResourceProperty().add(srt);
	}


	protected String createStorageID(String smsName){
		try{
			if(!uasProperties.getBooleanValue(UASProperties.TSS_FORCE_UNIQUE_STORAGE_IDS)){
				String xlogin=getClient().getXlogin().getUserName();
				return xlogin+"-"+smsName;	
			}
		}catch(Exception ex){}
		return Utilities.newUniqueID();
	}

	//create additional storages defined in the config file...
	protected void createAdditionalStorages(){
		Collection<StorageDescription> storages = uasProperties.getAddonStorages();
		for(StorageDescription a: storages){
			createStorageResource(a);
			logger.debug("Added " + a);
		}
	} 

	protected void createStorageResource(StorageDescription desc){
		Map<String,Object>initMap=new HashMap<String,Object>();
		String uuid=createStorageID(desc.getName());
		initMap.put(SMSBaseImpl.INIT_UNIQUE_ID, uuid);
		initMap.put(SMSBaseImpl.INIT_STORAGE_DESCRIPTION, desc);
		initMap.put(SMSBaseImpl.INIT_PARENT_NODE, nodeHelper);

		//set "infinite" tt
		Calendar tt=Calendar.getInstance();
		tt.add(Calendar.YEAR,10);
		initMap.put(SMSBaseImpl.INIT_INITIAL_TERMINATION_TIME,tt);
		initMap.put(INITPARAM_XNJS_REFERENCE, xnjsReference);

		EndpointReferenceType smsEpr=createStorageManagement(desc.getStorageTypeAsString(), initMap);
		if (smsEpr == null)
			return;
		
		StorageReferenceResourceProperty srp=getOrCreateStorageReferenceResourceProperty();
		srp.add(smsEpr);
	}

	/**
	 * create new Job, store the EPR in the relevant lists and return the EPR
	 * 
	 * @param initialisation params
	 * @return EPR
	 */
	protected EndpointReferenceType createJob(Map<String,Object> initParam)throws Exception{
		Home jmsHome = kernel.getHome(UAS.JMS); 
		String id=jmsHome.createWSRFServiceInstance(initParam);
		EndpointReferenceType epr=WSServerUtilities.makeEPR(UAS.JMS,id,JobManagement.JMS_PORT,kernel);
		//internal stuff to do with the new job...
		try{
			//add job ref to the relevant resource property
			JobReferenceResourceProperty j=(JobReferenceResourceProperty)properties.get(RPJobReference);
			j.add(epr);
		}catch(Exception e){
			logger.error("Could not store job reference.",e);
		}
		return epr;
	}

	/**
	 * create new storage management service and return its epr
	 * @param initialisation params
	 * @return EPR or null if SMS could not be created
	 */
	protected EndpointReferenceType createStorageManagement(String type, Map<String,Object> initParam){
		try {
			Home home=kernel.getHome(UAS.SMS);
			if(home==null){
				logger.warn("Storage management service is not deployed.");
				return null;
			}

			String id=(String)initParam.get(INIT_UNIQUE_ID);
			boolean mustCreate=true;
			if(id!=null){
				try{
					SMSBaseImpl sms=(SMSBaseImpl)home.get(id);
					//exists, now check if it is accessible for the current user
					String smsOwner=sms.getOwner()!=null?sms.getOwner().getName():null;
					if(smsOwner!=null && !kernel.getSecurityManager().isAccessible(getClient(), 
							sms.getServiceName(), id, smsOwner, sms.getVOMembership())){
						mustCreate=true;
						logger.info("Existing storage <"+id+"> is not accessible to current user (certificate change?), re-creating it.");
					}
					else{
						mustCreate=false;
					}
				}catch(ResourceUnknownException ignored){
					mustCreate=true;
				}
			}
			if(mustCreate){
				id=home.createWSRFServiceInstance(initParam);
			}
			
			return WSServerUtilities.makeEPR(UAS.SMS,id,StorageManagement.SMS_PORT,kernel);
		} catch (ResourceNotCreatedException rnc){
			String msg=Log.createFaultMessage("Storage of type <"+type+"> was NOT created.", rnc);
			logger.info(msg);
		} catch (Exception e) {
			LogUtil.logException("Could not create storage management service.",e,logger);
		}
		return null;
	}


	/**
	 * create and initialise the resourceproperty describing applications
	 * 
	 * @return ApplicationsResourceProperty
	 */
	protected ApplicationsResourceProperty createApplicationResourceProperty(){
		return new ApplicationsResourceProperty(this,xnjsReference);
	}


	/**
	 * create and initialise the resourceproperty describing the system processor
	 * 
	 * @return ProcessorResourceProperty
	 */
	protected ProcessorResourceProperty createProcessorResourceProperty(){
		return new ProcessorResourceProperty(this,xnjsReference);
	}


	/**
	 * create storage reference property
	 * @return
	 */
	protected StorageReferenceResourceProperty createStorageReferenceResourceProperty(EndpointReferenceType epr, StorageTypeEnumeration.Enum storageType){
		StorageReferenceType type=StorageReferenceType.Factory.newInstance();
		type.setStorageEndpointReference(epr);
		type.setType(storageType);
		StorageReferenceResourceProperty srp=new StorageReferenceResourceProperty(new StorageReferenceType[]{type});
		return srp;
	}

	protected StorageReferenceResourceProperty getOrCreateStorageReferenceResourceProperty(){
		StorageReferenceResourceProperty srp=(StorageReferenceResourceProperty)properties.get(RPStorageReference);
		if(srp==null){
			srp=new StorageReferenceResourceProperty(new StorageReferenceType[0]);
			properties.put(RPStorageReference, srp);
		}
		return srp;
	}

	protected void addStorageReference(EndpointReferenceType epr, StorageTypeEnumeration.Enum storageType){
		StorageReferenceResourceProperty srp=getOrCreateStorageReferenceResourceProperty();
		StorageReferenceType type=StorageReferenceType.Factory.newInstance();
		type.setStorageEndpointReference(epr);
		type.setType(storageType);
		srp.add(type);
	}

	/**
	 * retrieve all the client's job references on the target system
	 * 
	 * @return JobReferenceResourceProperty
	 */
	protected JobReferenceResourceProperty createJobReferenceProperty(){
		JobReferenceResourceProperty rp=new JobReferenceResourceProperty(this);
		return rp;
	}

	protected ReservationReferenceResourceProperty createReservationReferenceProperty(){
		return new ReservationReferenceResourceProperty(new EndpointReferenceType[0]);
	}

	/**
	 * create the resource property representing the operating system(s) of this TSS
	 */
	protected ResourceProperty<?> createOSResourceProperty(){
		return new OperatingSystemResourceProperty(this, xnjsReference);
	}

	@Override
	public QName getPortType() {
		return TSS_PORT;
	}

	/**
	 * resource-specific destruction
	 */
	@Override
	public void destroy() {
		try{
			ResourceDeletedMessage m=new ResourceDeletedMessage("deleted:"+getUniqueID());
			m.setDeletedResource(getUniqueID());
			m.setServiceName(getServiceName());
			kernel.getMessaging().getChannel(tsfID).publish(m);
		}
		catch(Exception e){
			LogUtil.logException("Could not send internal message.",e,logger);
		}
		if(uasProperties.getBooleanValue(UASProperties.TSS_FORCE_UNIQUE_STORAGE_IDS)){
			try{
				//destroy our SMS instances
				StorageReferenceResourceProperty storages=(StorageReferenceResourceProperty)properties.get(RPStorageReference);
				if(storages!=null){
					StorageReferenceType[] smsList=storages.getProperty();
					for(StorageReferenceType sms: smsList){
						EndpointReferenceType epr=sms.getStorageEndpointReference();
						BaseUASClient c=new BaseUASClient(epr, kernel.getClientConfiguration());
						c.destroy();
					}
				}
			}catch(Exception e){
				LogUtil.logException("Could not destroy storages for TSS <"+getUniqueID()+">",e,logger);
			}
		}
		//destroy the enumeration instance
		try{
			ImmutableResourceProperty p=(ImmutableResourceProperty)properties.get(RPJobReferenceEnumeration);
			JobReferenceEnumerationDocument jred=JobReferenceEnumerationDocument.Factory.parse(p.getXml()[0].newInputStream());
			BaseUASClient c=new BaseUASClient(jred.getJobReferenceEnumeration(), kernel.getClientConfiguration());
			c.destroy();
		}catch(Exception e){
			LogUtil.logException("Could not destroy enumeration for TSS <"+getUniqueID()+">",e,logger);
		}

		try{
			//shutdown the xnjs instance
			if(xnjsReference!=null){
				XNJSFacade.get(xnjsReference,kernel).shutdown();
			}
		}catch(Exception e){
			LogUtil.logException("Could not shutdown the XNJS instance.",e,logger);
		}
		super.destroy();

		String ownerName=(getOwner()!=null?getOwner().toString():"<no client>");
		logger.info("Removed TargetSystem resource <"+getUniqueID()+"> owned by "+ownerName);
	}

	/**
	 * resource reservation
	 */
	public ResourceReservationResponseDocument ReserveResources(ResourceReservationRequestDocument in) throws BaseFault {
		try{
			ResourcesDocument rd=ResourcesDocument.Factory.newInstance();
			rd.setResources(in.getResourceReservationRequest().getResources());
			Calendar startTime=in.getResourceReservationRequest().getStartTime();
			
			EndpointReferenceType epr=createReservationResource(rd,startTime);
			ResourceReservationResponseDocument res=ResourceReservationResponseDocument.Factory.newInstance();
			res.addNewResourceReservationResponse().setReservationReference(epr);
			try{
				//add reservation epr to the relevant resource property
				ReservationReferenceResourceProperty j=(ReservationReferenceResourceProperty)properties.get(RPReservationReference);
				j.add(epr);
			}catch(Exception e){
				LogUtil.logException("Could not store reservation reference.",e,logger);
			}
			setDirty();
			return res;
		}catch(Exception e){
			LogUtil.logException("Reservation not created.",e,logger);
			throw BaseFault.createFault("Reservation not created. Reason: "+e.getMessage());
		}
	}

	//create a WS Resource for the reservation and return its EPR
	protected EndpointReferenceType createReservationResource(ResourcesDocument resources, Calendar startTime)throws Exception{
		Map<String,Object>initMap=new HashMap<String,Object>();
		initMap.put(INITPARAM_XNJS_REFERENCE, xnjsReference);
		
		initMap.put(ReservationManagementImpl.INITPARAM_TSS_REFERENCE, getEPR());
		initMap.put(ReservationManagementImpl.INITPARAM_RESOURCES, resources);
		initMap.put(ReservationManagementImpl.INITPARAM_STARTTIME, startTime);
		initMap.put(INIT_PARENT_NODE, getNode());
		String id=kernel.getHome(UAS.RESERVATIONS).createWSRFServiceInstance(initMap);
		return WSServerUtilities.makeEPR(UAS.RESERVATIONS,id,ReservationManagement.PORT,kernel);
	}

	protected EndpointReferenceType createJobListEnumeration()throws Exception{
		Map<String,Object>init=new HashMap<String, Object>();
		init.put(EnumerationImpl.INIT_TARGETSERVICE_EPR, this.getEPR());
		init.put(EnumerationImpl.INIT_TARGETSERVICE_RP, RPJobReference);
		Calendar c=Calendar.getInstance();
		c.add(Calendar.MONTH, 24);
		init.put(EnumerationImpl.INIT_INITIAL_TERMINATION_TIME, c);
		Home h=kernel.getHome(UAS.ENUMERATION);
		if(h==null)throw new Exception("Enumeration service is not deployed!");
		String id=h.createWSRFServiceInstance(init);
		return WSServerUtilities.makeEPR(UAS.ENUMERATION,id,Enumeration.ENUM_PORT,kernel);
	}

	public String getUmask() {
		UmaskResourceProperty rp = (UmaskResourceProperty) properties.get(StorageManagement.RPUmask);
		if (rp == null)
			return Integer.toOctalString(IStorageAdapter.DEFAULT_UMASK);
		return rp.getXml()[0].getUmask();
	}

	private class UmaskListenerImpl implements UmaskChangedListener {
		/**
		 * only dry-run to check if the current TSI also accepts the new umask. Otherwise the
		 * value set to this instance of the TSI will be discarded.
		 */
		@Override
		public void umaskChanged(String newUmask) throws InvalidModificationException {
			try {
				Client client=getClient();
				TSI ret = XNJSFacade.get(xnjsReference, kernel).getConfiguration().getTargetSystemInterface(client);
				ret.setUmask(newUmask);
			} catch (IllegalArgumentException e) {
				throw new InvalidModificationException(e.getMessage());
			}
		}

	}
}
