/*********************************************************************************
 * Copyright (c) 2006 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
 

package de.fzj.unicore.uas.impl.tss;

import java.io.InputStream;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Map;

import javax.xml.namespace.QName;

import org.apache.log4j.Logger;
import org.oasisOpen.docs.wsrf.rl2.DestroyDocument;
import org.oasisOpen.docs.wsrf.rl2.DestroyResponseDocument;
import org.oasisOpen.docs.wsrf.rl2.SetTerminationTimeDocument;
import org.oasisOpen.docs.wsrf.rl2.SetTerminationTimeResponseDocument;
import org.unigrids.x2006.x04.services.tsf.CreateTSRDocument;
import org.unigrids.x2006.x04.services.tsf.CreateTSRResponseDocument;
import org.unigrids.x2006.x04.services.tsf.NameDocument;
import org.unigrids.x2006.x04.services.tsf.TargetSystemFactoryPropertiesDocument;
import org.unigrids.x2006.x04.services.tss.SupportsReservationDocument;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.ResourceReservation;
import de.fzj.unicore.uas.TargetSystem;
import de.fzj.unicore.uas.TargetSystemFactory;
import de.fzj.unicore.uas.UAS;
import de.fzj.unicore.uas.impl.UASWSResourceImpl;
import de.fzj.unicore.uas.impl.bp.BPSupportImpl;
import de.fzj.unicore.uas.impl.tss.rp.AccessibleTSSReferenceRP;
import de.fzj.unicore.uas.impl.tss.rp.ApplicationsResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.AvailableResourcesRP;
import de.fzj.unicore.uas.impl.tss.rp.CPUCountResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.CPUTimeResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.ExecutionEnvironmentRP;
import de.fzj.unicore.uas.impl.tss.rp.ExtendedSecurityInfoResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.MemoryPerNodeResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.NodesResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.OperatingSystemResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.PerformanceDataResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.ProcessorResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.SiteSpecificResourcesRP;
import de.fzj.unicore.uas.impl.tss.rp.TSSReferenceResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.TotalCPUsResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.UpSinceResourceProperty;
import de.fzj.unicore.uas.impl.tss.rp.XLoginResourceProperty;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.uas.xnjs.XNJSFacade;
import de.fzj.unicore.wsrflite.messaging.PullPoint;
import de.fzj.unicore.wsrflite.persistence.Persist;
import de.fzj.unicore.wsrflite.utils.WSServerUtilities;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import de.fzj.unicore.wsrflite.xmlbeans.ResourceProperty;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceNotDestroyedFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceUnavailableFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceUnknownFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.TerminationTimeChangeRejectedFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.UnableToSetTerminationTimeFault;
import de.fzj.unicore.wsrflite.xmlbeans.impl.WSResourceImpl;
import de.fzj.unicore.wsrflite.xmlbeans.rp.ImmutableResourceProperty;

/**
 * Implements the {@link TargetSystemFactory} interface.<br/>
 * 
 * <p>This is a fairly simplistic implementation that does not
 * use all the capabilities of the interface. For all the
 * TargetSystems created by this factory, the same XNJS instance
 * is used. The TargetSystemDescription supplied to CreateTSR() 
 * is ignored.</p>
 * 
 * @author schuller
 */
public class TargetSystemFactoryImpl extends UASWSResourceImpl implements TargetSystemFactory {

	private static final Logger logger=LogUtil.getLogger(LogUtil.SERVICES, TargetSystemFactoryImpl.class);

	/**
	 * initialisation parameter giving the name of this TSF
	 * if not given, the default name (defined in config file) is used
	 */
	public static final String INIT_TSF_NAME="TSF_NAME";
	
	@Override
	protected void addWSResourceInterfaces(BPSupportImpl baseProfile) {
		super.addWSResourceInterfaces(baseProfile);
		baseProfile.addWSResourceInterface(TSF_PORT);
	}
	
	
	/**
	 * maps TSS to TSS owner DNs 
	 */
	@Persist
	private Map<String, String>tssOwners=new HashMap<String, String>();
	
	/**
	 * get the owner for a given TSS
	 * @param tssID
	 * @return owner DN
	 */
	public String getOwnerForTSS(String tssID){
		return tssOwners.get(tssID);
	}
	
	/**
	 * Create a new TargetSystem resource
	 */
	public CreateTSRResponseDocument CreateTSR(CreateTSRDocument in)
			throws BaseFault {
		try {
			Map<String,Object>map=new HashMap<String,Object>();
			map.put(TargetSystemImpl.INIT_TSF_ID,getUniqueID());
			Calendar initialTT=null;
			try{
				initialTT=in.getCreateTSR().getTerminationTime().getCalendarValue();
			}catch(Exception e){
				
			}
			if(initialTT!=null){
				map.put(WSResourceImpl.INIT_INITIAL_TERMINATION_TIME, initialTT);	
			}
			map.put(INIT_PARENT_NODE, nodeHelper);
			
			
			EndpointReferenceType epr=createTargetSystem(map);
			String tssID=WSServerUtilities.extractResourceID(epr);
			String clientName=(getClient()!=null?getClient().getDistinguishedName():"<no client>");
			logger.info("Created new TargetSystem resource <"+tssID+"> for "+clientName);
			tssOwners.put(tssID, clientName);
			
			//add job ref to the relevant resource property
			TSSReferenceResourceProperty ref=getTSSReferenceProperty();
			ref.add(epr);

			CreateTSRResponseDocument response=CreateTSRResponseDocument.Factory.newInstance();
			response.addNewCreateTSRResponse();
			response.getCreateTSRResponse().setTsrReference(epr);
			
			setDirty();
			return response;
		} catch (Exception e) {
			LogUtil.logException("Did not create new TargetSystem resource.",e,logger);
			throw BaseFault.createFault(e.getMessage());
		}
	}

	@Override
	public void initialise(String serviceName, Map<String, Object> initArgs)throws Exception{
		super.initialise(serviceName, initArgs);
		
		//add rps
		properties.put(RPApplicationResource, createApplicationResourceProperty());
		properties.put(RPTSSReferences,  new TSSReferenceResourceProperty(new EndpointReferenceType[0]));
		properties.put(RPAccessibleTSSReferences, new AccessibleTSSReferenceRP(this));
		properties.put(RPCPUTime, new CPUTimeResourceProperty(this,xnjsReference));
		properties.put(RPNodeCount, new NodesResourceProperty(this,xnjsReference));
		properties.put(RPMemoryPerNode, new MemoryPerNodeResourceProperty(this,xnjsReference));
		properties.put(RPCPUCount, new CPUCountResourceProperty(this,xnjsReference));
		properties.put(RPTotalCPUCount, new TotalCPUsResourceProperty(this,xnjsReference));
		properties.put(RPProcessor, new ProcessorResourceProperty(this,xnjsReference));
		properties.put(RPUpSince, new UpSinceResourceProperty());
		properties.put(RPXlogin, new XLoginResourceProperty(this));
		
		String tsfName=(String)initArgs.get(INIT_TSF_NAME);
		NameDocument nd=NameDocument.Factory.newInstance();
		nd.setName(tsfName);
		properties.put(RPName, new ImmutableResourceProperty(nd));
		
		//reservation support?
		SupportsReservationDocument supportsReservation=SupportsReservationDocument.Factory.newInstance();
		supportsReservation.setSupportsReservation(XNJSFacade.get(xnjsReference,kernel).supportsReservation());
		properties.put(ResourceReservation.RP_SUPPORTS_RESERVATION,new ImmutableResourceProperty(supportsReservation));
	
		properties.put(RPSiteSpecificResources, new SiteSpecificResourcesRP(this,xnjsReference));
		properties.put(RPAvailableResources, new AvailableResourcesRP(this,xnjsReference));
		properties.put(RPExecutionEnvironments, new ExecutionEnvironmentRP(this,xnjsReference));
		
		properties.put(RPOperatingSystem, createOSResourceProperty());
		properties.put(RPPerformanceData, new PerformanceDataResourceProperty(this,xnjsReference));
		publish();
	}
	
	@Override
	protected ResourceProperty<?> getSecurityInfoResourceProperty(boolean addServerCert) {
		return new ExtendedSecurityInfoResourceProperty(this, addServerCert);
	}

	@Override
	public DestroyResponseDocument Destroy(DestroyDocument in) throws ResourceNotDestroyedFault, ResourceUnknownFault, ResourceUnavailableFault {
		throw ResourceNotDestroyedFault.createFault("Not destroyed."); 
	}

	@Override
	public SetTerminationTimeResponseDocument SetTerminationTime(
			SetTerminationTimeDocument in)
			throws UnableToSetTerminationTimeFault,
			TerminationTimeChangeRejectedFault, ResourceUnknownFault,
			ResourceUnavailableFault {
		throw TerminationTimeChangeRejectedFault.createFault("Not changed.");
	}
	
	/**
	 * create new Target System and return its epr
	 * Uses the default XNJS
	 */
	protected EndpointReferenceType createTargetSystem(Map<String,Object> initParam)throws Exception{
		String id=kernel.getHome(UAS.TSS).createWSRFServiceInstance(initParam);
		return WSServerUtilities.makeEPR(UAS.TSS,id,TargetSystem.PORTTYPE,kernel);
	}

	/**
	 * create new Target System and return its epr
	 * Uses a new XNJS configured from the supplied InputStream
	 */
	protected EndpointReferenceType createTargetSystem(Map<String,Object> initParam, InputStream xnjsConfig)throws Exception{
		
		//create a new XNJS and configure it
		String xnjsRef=getUniqueID();
		XNJSFacade f=XNJSFacade.get(xnjsRef, kernel);
		f.configure(xnjsConfig);
		//new TSS will pick up the XNJS reference and thus use the "right" XNJS
		initParam.put(INITPARAM_XNJS_REFERENCE, xnjsRef);
		
		String id=kernel.getHome(UAS.TSS).createWSRFServiceInstance(initParam);
		return WSServerUtilities.makeEPR(UAS.TSS,id,TargetSystem.TSS_PORT,kernel);
	}
	
	public TSSReferenceResourceProperty getTSSReferenceProperty(){
		return (TSSReferenceResourceProperty)properties.get(RPTSSReferences);
	}
	
	@Override
	public QName getResourcePropertyDocumentQName() {
		return TargetSystemFactoryPropertiesDocument.type.getDocumentElementName();
	}
	
	/**
	 * create the resource property representing the operating system(s) of this TSS
	 */
	protected ResourceProperty<?> createOSResourceProperty(){
		return new OperatingSystemResourceProperty(this, xnjsReference);
	}
	
	protected ApplicationsResourceProperty createApplicationResourceProperty(){
		return new ApplicationsResourceProperty(this,xnjsReference);
	}

	@Override
	public void customPostActivate(){
		//check for deleted TSSs and remove them...
		try{
			PullPoint p=kernel.getMessaging().getPullPoint(getUniqueID());
			TSSReferenceResourceProperty tssRef=getTSSReferenceProperty();
			while(p.hasNext()){
				String m=(String)p.next().getBody();
				if(m.startsWith("deleted:")){
					String id=m.substring(m.indexOf(":")+1);
					logger.debug("Removing TSS with ID "+id+"...");
					tssRef.remove(id);
					tssOwners.remove(id);
					setDirty();
				}
			}
		}catch(Exception e){
			LogUtil.logException(e.getMessage(),e,logger);
		}
	}
	
	@Override
	public QName getPortType() {
		return TSF_PORT;
	}
}
