package de.fzj.unicore.uas.impl.tss;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.security.auth.x500.X500Principal;

import org.apache.log4j.Logger;

import de.fzj.unicore.persist.PersistenceException;
import de.fzj.unicore.uas.TargetSystem;
import de.fzj.unicore.uas.UAS;
import de.fzj.unicore.uas.impl.reservation.ReservationManagementImpl;
import de.fzj.unicore.uas.impl.tss.rp.ReservationReferenceResourceProperty;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.wsrflite.Home;
import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.wsrflite.exceptions.ResourceUnknownException;
import eu.unicore.security.Client;
import eu.unicore.util.Log;

/**
 * Re-creates the list of accessible reservation references in a TSS as it is created.
 * This only happens if it is the only TSS for the user.
 * <p>
 * We must be careful here: as this is invoked from the thread pool, the client 
 * identity must be set upon creation (that's why we don't use the actual thread-local
 * user).
 * 
 * @author schuller
 */
public class RecreateReservationReferenceList implements Runnable{

	private final Logger logger=LogUtil.getLogger(LogUtil.SERVICES, RecreateReservationReferenceList.class);

	private final String tssID;

	private final Client client;

	private final Home tssHome;

	private final Home reservations;

	public RecreateReservationReferenceList(Kernel kernel, String tssID, Client client)throws PersistenceException{
		this.tssID=tssID;
		this.client=client;
		this.tssHome=kernel.getHome(UAS.TSS);
		this.reservations=kernel.getHome(UAS.RESERVATIONS);
	}

	public void run(){
		try{
			if(reservations==null){
				//nothing to do
				return;
			}
			
			String user = client.getDistinguishedName();
			X500Principal userP = new X500Principal(user);
			
			//check if owner has more TSSs
			Set<String>tssIds=tssHome.getStore().getUniqueIDs();
			tssIds.remove(tssID);
			for(String id: tssIds){
				TargetSystemImpl t=(TargetSystemImpl)tssHome.get(id);
				if (userP.equals(t.getOwner())){
					//nothing to do
					return;
				}
			}
			
			logger.info("Re-generating reservation reference list for " + user);
			List<String>oldReservations=new ArrayList<String>();
			for(String reservationID: getExistingReservations()){
				try{
					ReservationManagementImpl reservation=(ReservationManagementImpl)reservations.get(reservationID);
					if(reservation.getOwner() ==null || userP.equals(reservation.getOwner())){
						oldReservations.add(reservationID);
						try{
							reservation=(ReservationManagementImpl)reservations.getForUpdate(reservationID);
							reservation.setTSSID(tssID);
							reservations.persist(reservation);
						}catch(Exception ex){
							Log.logException("Could not change TSS ID of reservation <"+reservationID+">", ex, logger);
						}
					}
				}catch(ResourceUnknownException re){
					logger.debug("Reservation <"+reservationID+"> not found any more.");
				}
			}
			TargetSystemImpl tss=null;
			try{
				tss=(TargetSystemImpl)tssHome.getForUpdate(tssID);
				ReservationReferenceResourceProperty rRP=(ReservationReferenceResourceProperty)tss.getResourceProperty(TargetSystem.RPReservationReference);
				if(rRP!=null){
					rRP.addAll(oldReservations);
					logger.info("Added <"+rRP.getNumberOfElements()+"> existing reservations to target system");
				}
			}
			finally{
				if(tss!=null)tssHome.getStore().persist(tss);
			}
		}catch(Exception ex){
			logger.error("Could not restore reservations for "+client.getDistinguishedName(),ex);
		}
	}

	private Set<String>getExistingReservations()throws PersistenceException{
		return reservations.getStore().getUniqueIDs();
	}

}
