package de.fzj.unicore.uas.impl.tss;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.security.auth.x500.X500Principal;

import org.apache.log4j.Logger;

import de.fzj.unicore.persist.PersistenceException;
import de.fzj.unicore.uas.TargetSystem;
import de.fzj.unicore.uas.UAS;
import de.fzj.unicore.uas.impl.job.JobManagementImpl;
import de.fzj.unicore.uas.impl.tss.rp.JobReferenceResourceProperty;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.wsrflite.Home;
import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.wsrflite.exceptions.ResourceUnknownException;
import eu.unicore.security.Client;
import eu.unicore.util.Log;

/**
 * Re-creates the list of accessible job references in a TSS as it is created.
 * This only happens if it is the only TSS for the user.
 * <p>
 * We must be careful here: as this is invoked from the thread pool, the client 
 * identity must be set upon creation (that's why we don't use the actual thread-local
 * user).
 * 
 * @author schuller
 */
public class RecreateJMSReferenceList implements Runnable{

	private final Logger logger=LogUtil.getLogger(LogUtil.SERVICES, RecreateJMSReferenceList.class);

	private final String tssID;

	private final Client client;

	private final Home tssHome;

	private final Home jms;

	public RecreateJMSReferenceList(Kernel kernel, String tssID, Client client)throws PersistenceException{
		this.tssID=tssID;
		this.client=client;
		this.tssHome=kernel.getHome(UAS.TSS);
		this.jms=kernel.getHome(UAS.JMS);
	}

	public void run(){
		try{
			String user = client.getDistinguishedName();
			X500Principal userP = new X500Principal(user);
			
			//check if owner has more TSSs
			Set<String>tssIds=tssHome.getStore().getUniqueIDs();
			tssIds.remove(tssID);
			for(String id: tssIds){
				TargetSystemImpl t=(TargetSystemImpl)tssHome.get(id);
				if (userP.equals(t.getOwner())){
					//nothing to do
					return;
				}
			}
			
			logger.info("Re-generating job list for " + user);
			List<String>oldJobs=new ArrayList<String>();
			for(String jobID: getExistingJobs()){
				try{
					JobManagementImpl j=(JobManagementImpl)jms.get(jobID);
					if(j.getOwner() ==null || userP.equals(j.getOwner())){
						oldJobs.add(jobID);
						try{
							j=(JobManagementImpl)jms.getForUpdate(jobID);
							j.setTSSID(tssID);
							jms.persist(j);
						}catch(Exception ex){
							Log.logException("Could not change TSS ID of job <"+jobID+">", ex, logger);
						}
					}
				}catch(ResourceUnknownException re){
					logger.debug("Job <"+jobID+"> not found any more.");
				}
			}
			TargetSystemImpl tss=null;
			try{
				try{
					tss=(TargetSystemImpl)tssHome.getForUpdate(tssID);	
				}catch(ResourceUnknownException rue){
					logger.error(rue);
				}
				
				JobReferenceResourceProperty jRP=(JobReferenceResourceProperty)tss.getResourceProperty(TargetSystem.RPJobReference);
				jRP.addAll(oldJobs);
				logger.info("Added <"+jRP.getNumberOfElements()+"> existing jobs to target system");
			}
			finally{
				tssHome.getStore().persist(tss);
			}
		}catch(Exception ex){
			logger.error("Could not restore jobs for "+client.getDistinguishedName(),ex);
		}
	}

	private Set<String>getExistingJobs()throws PersistenceException{
		return jms.getStore().getUniqueIDs();
	}

}
