/*********************************************************************************
 * Copyright (c) 2010 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
 

package de.fzj.unicore.uas.impl.task;

import java.util.Calendar;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.xml.namespace.QName;

import org.apache.xmlbeans.XmlObject;
import org.unigrids.services.atomic.types.StatusType;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.Task;
import de.fzj.unicore.uas.impl.UASWSResourceImpl;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import de.fzj.unicore.wsrflite.xmlbeans.WSUtilities;
import de.fzj.unicore.wsrflite.xmlbeans.rp.AddressResourceProperty;
import de.fzj.unicore.wsrflite.xmlbeans.rp.ImmutableResourceProperty;
import eu.unicore.unicore6.task.CancelRequestDocument;
import eu.unicore.unicore6.task.CancelResponseDocument;
import eu.unicore.unicore6.task.SubmissionServiceReferenceDocument;
import eu.unicore.unicore6.task.SubmissionTimeDocument;
import eu.unicore.unicore6.task.TaskPropertiesDocument;

/**
 * implementation of the {@link Task} service.<br/>
 * 
 * The actual process is running in the background and can
 * use the {@link #putResult(String, XmlObject)} method to make the result
 * available to the client once it is available.<br/>
 *  
 * @author schuller
 */
public class TaskImpl extends UASWSResourceImpl implements Task {

	//stores results keyed by the unique Task id
	private final static Map<String,XmlObject>results=new ConcurrentHashMap<String,XmlObject>();
	
	//stores Task stati keyed by the unique Task id
	private final static Map<String,TaskStatus>status=new ConcurrentHashMap<String,TaskStatus>();
	
	//init parameter keys
	public static final String INIT_PARENT_SERVICE="parentServiceEPR";
	
	public CancelResponseDocument Cancel(CancelRequestDocument in)throws BaseFault {
		TaskStatus s=getStatus(getUniqueID());
		if(StatusType.RUNNING.equals(s.status)){
			s.status=StatusType.FAILED;
			putResult(getUniqueID(), getCancelledResult());
		}
		CancelResponseDocument crd=CancelResponseDocument.Factory.newInstance();
		crd.addNewCancelResponse();
		return crd;
	}

	@Override
	public void initialise(String serviceName, Map<String, Object> initParams)
			throws Exception {
		super.initialise(serviceName, initParams);
		EndpointReferenceType parent=(EndpointReferenceType)initParams.get(INIT_PARENT_SERVICE);
		QName q=SubmissionServiceReferenceDocument.type.getDocumentElementName();
		String parentService=WSUtilities.extractServiceName(parent.getAddress().getStringValue());
		String parentID=WSUtilities.extractResourceID(parent);
		String serviceSpec=parentID!=null?parentService+"?res="+parentID:parentService;
		AddressResourceProperty p1=new AddressResourceProperty(this,q,serviceSpec,true);
		
		properties.put(RP_SUBMISSION_SERVICE_REFERENCE,p1);
		properties.put(RP_RESULT,new ResultRP(this));
		SubmissionTimeDocument time=SubmissionTimeDocument.Factory.newInstance();
		time.setSubmissionTime(Calendar.getInstance());
		ImmutableResourceProperty p2=new ImmutableResourceProperty(time);
		properties.put(RP_SUBMISSION_TIME,p2);
		TaskStatus s=new TaskStatus();
		s.status=StatusType.RUNNING;
		status.put(getUniqueID(),s);
		properties.put(RP_STATUS,new StatusRP(this));
	}

	@Override
	public QName getResourcePropertyDocumentQName() {
		return TaskPropertiesDocument.type.getDocumentElementName();
	}

	/**
	 * get the result, if available. If available the result will be removed from the underlying storage.
	 *  
	 * @param uuid
	 * @return the result, or <code>null</code>
	 */
	public static XmlObject getAndRemoveResult(String uuid){
		return results.remove(uuid);
	}

	/**
	 * put a result<br/>
	 * 
	 * TODO notify once we support notifications
	 * 
	 * @param uuid
	 * @param result
	 * 
	 * @return the status of the task
	 */
	public static TaskStatus putResult(String uuid, XmlObject result){
		results.put(uuid,result);
		status.get(uuid).status=StatusType.SUCCESSFUL;
		return status.get(uuid);
	}
	
	public static TaskStatus getStatus(String uuid){
		return status.get(uuid);
	}

	public static void updateStatus(String uuid, TaskStatus taskStatus){
		status.put(uuid,taskStatus);
	}
	
	@Override
	public void destroy() {
		results.remove(getUniqueID());
		status.remove(getUniqueID());
		super.destroy();
	}

	public static class TaskStatus{
		public volatile String message;
		public volatile Float progress;
		public volatile Integer exitCode;
		public volatile StatusType.Enum status=StatusType.UNDEFINED;
	}
	
	public static XmlObject getDefaultResult(){
		String x="<Completed xmlns=\""+NAMESPACE+"\"/>";
		try{
			return XmlObject.Factory.parse(x);
		}catch(Exception ex){
			//can't happen
			throw new RuntimeException(ex);
		}
	}
	public static XmlObject getCancelledResult(){
		String x="<Cancelled xmlns=\""+NAMESPACE+"\"/>";
		try{
			return XmlObject.Factory.parse(x);
		}catch(Exception ex){
			//can't happen
			throw new RuntimeException(ex);
		}
	}
	
}
