package de.fzj.unicore.uas.impl.sms;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;

import org.apache.log4j.Logger;
import org.oasisOpen.docs.wsrf.rl2.DestroyDocument;
import org.oasisOpen.docs.wsrf.rl2.DestroyResponseDocument;
import org.oasisOpen.docs.wsrf.rl2.SetTerminationTimeDocument;
import org.oasisOpen.docs.wsrf.rl2.SetTerminationTimeResponseDocument;
import org.oasisOpen.docs.wsrf.rl2.TerminationTimeDocument.TerminationTime;
import org.unigrids.services.atomic.types.PropertyType;
import org.unigrids.x2006.x04.services.smf.AccessibleStorageEnumerationDocument;
import org.unigrids.x2006.x04.services.smf.CreateSMSDocument;
import org.unigrids.x2006.x04.services.smf.CreateSMSResponseDocument;
import org.unigrids.x2006.x04.services.smf.StorageBackendParametersDocument.StorageBackendParameters;
import org.unigrids.x2006.x04.services.smf.StorageDescriptionType;
import org.unigrids.x2006.x04.services.smf.StorageEnumerationDocument;
import org.unigrids.x2006.x04.services.smf.StorageFactoryPropertiesDocument;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.Enumeration;
import de.fzj.unicore.uas.StorageFactory;
import de.fzj.unicore.uas.StorageManagement;
import de.fzj.unicore.uas.UAS;
import de.fzj.unicore.uas.impl.UASWSResourceImpl;
import de.fzj.unicore.uas.impl.enumeration.EnumerationImpl;
import de.fzj.unicore.uas.impl.sms.StorageManagementHomeImpl.StorageTypes;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.wsrflite.Home;
import de.fzj.unicore.wsrflite.exceptions.ResourceUnknownException;
import de.fzj.unicore.wsrflite.messaging.PullPoint;
import de.fzj.unicore.wsrflite.persistence.Persist;
import de.fzj.unicore.wsrflite.utils.WSServerUtilities;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceNotDestroyedFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceUnavailableFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceUnknownFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.TerminationTimeChangeRejectedFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.UnableToSetTerminationTimeFault;
import de.fzj.unicore.wsrflite.xmlbeans.rp.ImmutableResourceProperty;

/**
 * Implements the storage factory
 * 
 * @author schuller
 * @author daivandy
 * @since 6.3
 */
public class StorageFactoryImpl extends UASWSResourceImpl implements StorageFactory{

	private static final Logger logger=LogUtil.getLogger(LogUtil.SERVICES, StorageFactoryImpl.class);

	/**
	 * maps SMS to SMS owner DNs 
	 */
	@Persist
	private Map<String, String>smsOwners=new HashMap<String, String>();

	/**
	 * get the owner for a given sms
	 * @param smsID
	 * @return owner DN
	 */
	public String getOwnerForSMS(String smsID){
		return smsOwners.get(smsID);
	}

	public SMSReferenceResourceProperty getSMSReferenceProperty(){
		return (SMSReferenceResourceProperty)properties.get(RPSMSReferences);
	}

	@Override
	public void initialise(String serviceName, Map<String, Object> initParams)
	throws Exception {
		super.initialise(serviceName, initParams);
		properties.put(RPSMSReferences, new SMSReferenceResourceProperty(new EndpointReferenceType[0]));
		properties.put(RPAccessibleSMSReferences, new AccessibleSMSReferenceRP(this));
		try{
			StorageEnumerationDocument sred=StorageEnumerationDocument.Factory.newInstance();
			sred.setStorageEnumeration(createEnumeration(RPSMSReferences));
			properties.put(RPSMSEnumeration, new ImmutableResourceProperty(sred));
		}
		catch(Exception ex){
			LogUtil.logException("Error creating SMS reference enumeration",ex,logger);
		}
		try{
			AccessibleStorageEnumerationDocument sred=AccessibleStorageEnumerationDocument.Factory.newInstance();
			sred.setAccessibleStorageEnumeration(createEnumeration(RPAccessibleSMSReferences));
			properties.put(RPAccessibleSMSEnumeration, new ImmutableResourceProperty(sred));
		}
		catch(Exception ex){
			LogUtil.logException("Error creating SMS reference enumeration",ex,logger);
		}

		String xnjsRef=null; //TODO: get XNJS reference from Kernel
		properties.put(RPStorageDescription,new StorageDescriptionRP(this,xnjsRef));
		logger.info("Storage factory <"+getUniqueID()+"> created");
		publish();
	}

	public CreateSMSResponseDocument CreateSMS(CreateSMSDocument in) throws BaseFault {
		if(logger.isTraceEnabled())logger.trace("CreateSMS: "+in);
		
		String clientName=(getClient()!=null?getClient().getDistinguishedName():"<no client>");
		try{
			Map<String, StorageDescription> factories = uasProperties.getStorageFactories();
			//choose default backend type. If there is only one, or if there is one
			//named "DEFAULT", it is used. Otherwise, an exception is thrown.
			String storageBackendType = null;
			List<String>types=new ArrayList<String>();
			types.addAll(factories.keySet());
			if(types.contains("DEFAULT")){
				storageBackendType="DEFAULT";
			}
			else if(types.size()==1){
				storageBackendType=types.get(0);
			}
			else{
				throw new IllegalArgumentException("Please specify the storage backend type. Available are "+types);
			}
			
			//extra settings for the to-be-created SMS
			Map<String,String>extraProperties=new HashMap<String,String>();
			
			// accommodate for parametrisable storage backend type
			StorageDescriptionType sdType = in.getCreateSMS().getStorageDescription();
			if(sdType!=null) {
				if(sdType.getStorageBackendType()!=null){
					storageBackendType = sdType.getStorageBackendType();
				}
				if(sdType.getStorageBackendParameters()!=null){
					extraProperties.putAll(getAdditionClientConfigurationItems(
							sdType.getStorageBackendParameters()));
				}
			}
			
			StorageDescription factoryDesc = factories.get(storageBackendType);
			if (factoryDesc == null)
				throw new IllegalArgumentException("Unknown type of storage factory: " + storageBackendType);
			//we don't want to mess the global configuration of SMSFies
			factoryDesc = factoryDesc.clone();
			
			//we allow clients to overwrite the name 
			if (sdType != null && sdType.getFileSystem() != null && sdType.getFileSystem().getName() != null) {
				factoryDesc.setName(sdType.getFileSystem().getName());
			}
			
			//this is to override all user-provided settings with the ones configured
			//FIXME not very flexible: it is impossible to define defaults which can be changed by the user
			//also no way to change the standard settings.
			extraProperties.putAll(factoryDesc.getAdditionalProperties());
			factoryDesc.getAdditionalProperties().putAll(extraProperties);
			
			Map<String,Object>initMap=new HashMap<String,Object>();
			initMap.put(SMSBaseImpl.INIT_STORAGE_DESCRIPTION, factoryDesc);
			initMap.put(SMSBaseImpl.INIT_DIRECTORY_APPEND_UNIQUE_ID,Boolean.TRUE);
			initMap.put(SMSBaseImpl.INIT_FACTORYID, getUniqueID());
			initMap.put(INIT_PARENT_NODE, getNode());
			initMap.put(SMSBaseImpl.INIT_SKIP_EXISTENCECHECK,Boolean.TRUE);
			TerminationTime tt=in.getCreateSMS().getTerminationTime();
			if(tt!=null){
				initMap.put(SMSBaseImpl.INIT_INITIAL_TERMINATION_TIME,tt.getCalendarValue());
			}
			

			EndpointReferenceType epr=createStorageResource(initMap);

			CreateSMSResponseDocument resD=CreateSMSResponseDocument.Factory.newInstance();
			resD.addNewCreateSMSResponse().setSmsReference(epr);
			
			String smsID=WSServerUtilities.extractResourceID(epr);
			initStorage(smsID);
			logger.info("Created new StorageManagement resource <"+smsID+"> for "+clientName);
			smsOwners.put(smsID, clientName);
			getSMSReferenceProperty().add(epr);
			setDirty();
			return resD;
		}catch(Exception ex){
			LogUtil.logException("Could not create Storage instance for <"+clientName+">", ex, logger);
			throw BaseFault.createFault("Could not create Storage instance. Please consult the site administrator.", ex);
		}
	}

	/**
	 * replaces prefix of the storage factory into a normal one of SMS
	 * @param params - parameters provided by the client
	 * @return
	 */
	private Map<String, String> getAdditionClientConfigurationItems(StorageBackendParameters params) {
		Map<String, String> ret = new HashMap<String, String>();
		for(PropertyType p: params.getPropertyArray()){
			ret.put(p.getName(), p.getValue());
		}
		return ret;
	}
	
	protected EndpointReferenceType createEnumeration(QName rp)throws Exception{
		Map<String,Object>init=new HashMap<String, Object>();
		init.put(EnumerationImpl.INIT_TARGETSERVICE_EPR, this.getEPR());
		init.put(EnumerationImpl.INIT_TARGETSERVICE_RP, rp);
		Calendar c=Calendar.getInstance();
		c.add(Calendar.MONTH, 24);
		init.put(EnumerationImpl.INIT_INITIAL_TERMINATION_TIME, c);
		init.put(INIT_PARENT_NODE, getNode());
		Home h=kernel.getHome(UAS.ENUMERATION);
		if(h==null)throw new Exception("Enumeration service is not deployed!");
		String id=h.createWSRFServiceInstance(init);
		return WSServerUtilities.makeEPR(UAS.ENUMERATION,id,Enumeration.ENUM_PORT, kernel);
	}


	/**
	 * create new storage management service and return its epr
	 * @param initialisation params
	 * @return EPR or null if SMS could not be created
	 */
	protected EndpointReferenceType createStorageResource(Map<String,Object> initParam)
	throws Exception{
		Home home=kernel.getHome(UAS.SMS);
		if(home==null){
			throw new ResourceUnknownException("Storage management service is not deployed.");
		}
		String id=home.createWSRFServiceInstance(initParam);
		return WSServerUtilities.makeEPR(UAS.SMS,id,StorageManagement.SMS_PORT, kernel);
	}

	protected void initStorage(String uniqueID)throws Exception{
		StorageManagementHomeImpl smsHome=(StorageManagementHomeImpl)kernel.getHome(UAS.SMS);
		SMSBaseImpl sms=(SMSBaseImpl)smsHome.get(uniqueID);
		
		//FIXME here there was a test whether it is DSMS. If this is not applicable for DSMS then, possibly, also not for other CUSTOMs.
		//and it should be possible to configure DSMS as a CUSTOM SMS. 
		if (sms.getStorageDescription().getStorageType() != StorageTypes.CUSTOM) {
			sms.getStorageAdapter().mkdir("/");
		}
	}

	@Override
	public QName getPortType() {
		return SMF_PORT;
	}

	@Override
	public QName getResourcePropertyDocumentQName() {
		return StorageFactoryPropertiesDocument.type.getDocumentElementName();
	}

	@Override
	public DestroyResponseDocument Destroy(DestroyDocument in) throws ResourceNotDestroyedFault, ResourceUnknownFault, ResourceUnavailableFault {
		throw ResourceNotDestroyedFault.createFault("Not destroyed."); 
	}

	@Override
	public SetTerminationTimeResponseDocument SetTerminationTime(
			SetTerminationTimeDocument in)
	throws UnableToSetTerminationTimeFault,
	TerminationTimeChangeRejectedFault, ResourceUnknownFault,
	ResourceUnavailableFault {
		throw TerminationTimeChangeRejectedFault.createFault("Not changed.");
	}


	@Override
	public void customPostActivate(){
		//check for deleted SMSs and remove them...
		try{
			PullPoint p=kernel.getMessaging().getPullPoint(getUniqueID());
			SMSReferenceResourceProperty smsRef=getSMSReferenceProperty();
			while(p.hasNext()){
				String m=(String)p.next().getBody();
				if(m.startsWith("deleted:")){
					String id=m.substring(m.indexOf(":")+1);
					logger.debug("Removing Storage with ID "+id+"...");
					smsRef.remove(id);
					smsOwners.remove(id);
					setDirty();
				}
			}
		}catch(Exception e){
			LogUtil.logException(e.getMessage(),e,logger);
		}
	}

}
