/*********************************************************************************
 * Copyright (c) 2010 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/


package de.fzj.unicore.uas.impl.sms;

import java.util.Map;

import org.apache.log4j.Logger;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.FileSystemDocument;
import org.unigrids.x2006.x04.services.smf.StorageDescriptionDocument;
import org.unigrids.x2006.x04.services.smf.StorageDescriptionType;

import de.fzj.unicore.uas.UASProperties;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.wsrflite.Resource;
import de.fzj.unicore.wsrflite.security.util.AuthZAttributeStore;
import de.fzj.unicore.wsrflite.xmlbeans.ResourceProperty;
import eu.unicore.security.Client;

/**
 * publishes descriptions of the available storage factory types to clients
 *
 * @author schuller
 */
public class StorageDescriptionRP extends  ResourceProperty<Object>{

	private static final long serialVersionUID=1L;

	private static final Logger logger=LogUtil.getLogger(LogUtil.SERVICES, StorageDescriptionRP.class);

	StorageDescriptionDocument[] xml;

	//FIXME - this is buggy, the variable is never modified
	private long lastPropertyUpdate=-1;

	private final String xnjsReference;
	
	public StorageDescriptionRP(Resource parent, String xnjsReference){
		super(parent);
		xml=new StorageDescriptionDocument[0];
		this.xnjsReference=xnjsReference;
	}

	@Override
	public StorageDescriptionDocument[] getXml() {
		return xml;
	}

	@Override
	public ResourceProperty<Object> update(){
		if(mustUpdate()){
			UASProperties props = parentWSResource.getKernel().getAttribute(UASProperties.class);
			Map<String, StorageDescription> factoriesDesc = props.getStorageFactories();
			xml=new StorageDescriptionDocument[factoriesDesc.size()];
			int i=0;
			for(String type: factoriesDesc.keySet()){
				xml[i]=getStorageDescription(factoriesDesc.get(type));
				i++;
			}
		}
		return this;
	}

	protected StorageDescriptionDocument getStorageDescription(StorageDescription storageDesc){
		StorageDescriptionDocument sdd=StorageDescriptionDocument.Factory.newInstance();
		StorageDescriptionType sType=sdd.addNewStorageDescription();
		sType.setStorageBackendType(storageDesc.getName());
		FileSystemDocument fs=getFileSystemDescription(storageDesc);
		if(fs!=null){
			sType.setFileSystem(fs.getFileSystem());
		}
		return sdd;
	}

	protected FileSystemDocument getFileSystemDescription(StorageDescription storageDesc){
		Kernel kernel=parentWSResource.getKernel();
		Class<? extends StorageInfoProvider> infoP=storageDesc.getInfoProviderClass();
		if(infoP!=null){
			try{
				StorageInfoProvider infoProvider=(StorageInfoProvider)kernel.load(infoP);
				Client client=AuthZAttributeStore.getClient();
				FileSystemDocument fs=infoProvider.getInformation(storageDesc, client, xnjsReference);
				return fs;
			}
			catch(Exception ex){
				LogUtil.logException("Can't get storage information for <"+storageDesc.getName()+">", 
						ex, logger);
			}
		}
		return null;
	}

	protected boolean mustUpdate(){
		return lastPropertyUpdate<=parentWSResource.getKernel().getLastConfigFileChangeTime();
	}

}
