/*
 * Copyright (c) 2012 ICM Uniwersytet Warszawski All rights reserved.
 * See LICENCE.txt file for licensing information.
 */
package de.fzj.unicore.uas.impl.sms;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import de.fzj.unicore.uas.SMSProperties;
import de.fzj.unicore.uas.StorageManagement;
import de.fzj.unicore.uas.impl.sms.StorageManagementHomeImpl.StorageTypes;
import de.fzj.unicore.xnjs.io.IStorageAdapter;
import eu.unicore.util.configuration.ConfigurationException;

/**
 * Parameters used to configure storages which are based on {@link SMSBaseImpl}. Some of the options 
 * are not used by all actual storage implementations. See constructor documentation for details.
 * 
 * @author K. Benedyczak
 */
public class StorageDescription implements Serializable, Cloneable {
	private static final long serialVersionUID = 1L;
	
	private final String id;
	private String name;
	private final Class<? extends StorageManagement> clazz;
	private final StorageTypes type;
	private Class<? extends StorageInfoProvider> infoProviderClass;
	private final String pathSpec;
	private final boolean disableMetadata;
	
	//updateable
	private String description;
	private String protocols;
	private Map<String, String> additionalProperties;
	private boolean filterListing;
	private boolean cleanup;
	private String defaultUmask;

	/**
	 * For all optional parameters null can be passed. Use this one for SMS description.
	 * @param id identifier
	 * @param name mandatory name of the storage
	 * @param pathSpec optional base path used by the storage 
	 * @param type mandatory the storage type
	 * @param clazz optional class of the storage - significant only for custom type
	 * @param protocols optional enabled transport protocols
	 * @param filterListing optional, whether to filter listings to include only accessible files 
	 * @param cleanup optional, whether to cleanup the storage when destroyed
	 * @param disableMetadata optional, whether to disable metadata for this storage
	 * @param defaultUmask optional, the default umask of files
	 * @param description optional storage description
	 * @param additionalProperties optional - additional properties
	 */
	public StorageDescription(String id, String name, String pathSpec, StorageTypes type, Class<? extends StorageManagement> clazz, 
			String protocols, Boolean filterListing, Boolean cleanup, Boolean disableMetadata, String defaultUmask, 
			String description, Map<String, String> additionalProperties){
		this.id = id;
		if (id == null)
			throw new IllegalArgumentException("Storage identifier can not be null");
		this.name = name == null ? id : name;
		this.type=type;
		this.pathSpec=pathSpec;
		this.clazz=clazz;
		this.disableMetadata= disableMetadata==null ? false : disableMetadata.booleanValue();
		update(protocols, filterListing, cleanup, defaultUmask, description, additionalProperties);
	}

	public void update(String protocols, Boolean filterListing, Boolean cleanup, String defaultUmask, 
			String description, Map<String, String> additionalProperties) {
		this.description = description == null ? "Filesystem" : description;
		this.protocols = protocols;
		this.additionalProperties = additionalProperties;
		if (additionalProperties == null)
			additionalProperties = Collections.emptyMap();
		this.filterListing = filterListing == null ? false : filterListing;
		this.defaultUmask = defaultUmask == null ? Integer.toOctalString(IStorageAdapter.DEFAULT_UMASK) : 
			defaultUmask;
		this.cleanup = cleanup == null ? true : cleanup;
		if (!SMSProperties.umaskPattern.matcher(this.defaultUmask).matches())
			throw new ConfigurationException("Specified umask must be an octal number from 0 to 777.");
		if (type.equals(StorageTypes.CUSTOM) && clazz == null)
			throw new ConfigurationException("No class defined for the CUSTOM storage");
	}
	
	public String getId() {
		return id;
	}
	
	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public String getDescription() {
		return description;
	}

	public String getProtocols() {
		return protocols;
	}
	
	public StorageTypes getStorageType() {
		return type;
	}

	public String getStorageTypeAsString() {
		return type.toString();
	}

	public Map<String, String> getAdditionalProperties() {
		return additionalProperties;
	}
	
	public String getPathSpec() {
		return pathSpec;
	}

	public Class<? extends StorageManagement> getStorageClass() {
		return clazz;
	}
	
	public boolean isFilterListing() {
		return filterListing;
	}

	public boolean isCleanupOnDestroy() {
		return cleanup;
	}
	
	public String getDefaultUmask() {
		return defaultUmask;
	}
	
	public Class<? extends StorageInfoProvider> getInfoProviderClass() {
		return infoProviderClass;
	}

	public void setInfoProviderClass(Class<? extends StorageInfoProvider> clazz) {
		this.infoProviderClass = clazz;
		if (infoProviderClass == null) {
			switch (getStorageType()) {
			case FIXEDPATH:
			case VARIABLE:
				this.infoProviderClass = DefaultStorageInfoProvider.class;
				break;
			}
		}
	}

	public String toString(){
		return "Storage description: " + "name="+name+", type="+type+ ", protocols="+protocols 
				+ ", pathSpec=" + pathSpec + ", defaultUmask="+defaultUmask + ", filterListing="+filterListing;
	}
	
	public StorageDescription clone() {
		Map<String, String> clonedAdd = new HashMap<String, String>();
		clonedAdd.putAll(additionalProperties);
		StorageDescription ret = new StorageDescription(id, name, pathSpec, type, clazz, protocols, 
				filterListing, cleanup, disableMetadata, defaultUmask, 
				description, clonedAdd);
		if (infoProviderClass != null)
			ret.setInfoProviderClass(this.infoProviderClass);
		return ret;
	}
}