/*********************************************************************************
 * Copyright (c) 2006 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/


package de.fzj.unicore.uas.impl.sms;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.math.BigInteger;
import java.net.URI;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;

import javax.xml.namespace.QName;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Element;
import net.sf.ehcache.config.CacheConfiguration;
import net.sf.ehcache.config.DiskStoreConfiguration;
import net.sf.ehcache.store.MemoryStoreEvictionPolicy;

import org.apache.log4j.Logger;
import org.unigrids.services.atomic.types.ACLEntryType;
import org.unigrids.services.atomic.types.ACLEntryTypeType;
import org.unigrids.services.atomic.types.ACLType;
import org.unigrids.services.atomic.types.GridFileType;
import org.unigrids.services.atomic.types.MetadataType;
import org.unigrids.services.atomic.types.PermissionsType;
import org.unigrids.services.atomic.types.PropertyType;
import org.unigrids.services.atomic.types.ProtocolType;
import org.unigrids.services.atomic.types.TextInfoType;
import org.unigrids.x2006.x04.services.sms.ACLChangeModeType;
import org.unigrids.x2006.x04.services.sms.ChangeACLEntryType;
import org.unigrids.x2006.x04.services.sms.ChangeACLType;
import org.unigrids.x2006.x04.services.sms.ChangePermissionsDocument;
import org.unigrids.x2006.x04.services.sms.ChangePermissionsEntryType;
import org.unigrids.x2006.x04.services.sms.ChangePermissionsResponseDocument;
import org.unigrids.x2006.x04.services.sms.CopyDocument;
import org.unigrids.x2006.x04.services.sms.CopyResponseDocument;
import org.unigrids.x2006.x04.services.sms.CreateDirectoryDocument;
import org.unigrids.x2006.x04.services.sms.CreateDirectoryResponseDocument;
import org.unigrids.x2006.x04.services.sms.DeleteDocument;
import org.unigrids.x2006.x04.services.sms.DeleteResponseDocument;
import org.unigrids.x2006.x04.services.sms.ExportFileDocument;
import org.unigrids.x2006.x04.services.sms.ExportFileResponseDocument;
import org.unigrids.x2006.x04.services.sms.ExtendedChangePermissionsType;
import org.unigrids.x2006.x04.services.sms.ExtraParametersDocument.ExtraParameters;
import org.unigrids.x2006.x04.services.sms.FindDocument;
import org.unigrids.x2006.x04.services.sms.FindDocument.Find;
import org.unigrids.x2006.x04.services.sms.FindResponseDocument;
import org.unigrids.x2006.x04.services.sms.FindResponseDocument.FindResponse;
import org.unigrids.x2006.x04.services.sms.ImportFileDocument;
import org.unigrids.x2006.x04.services.sms.ImportFileResponseDocument;
import org.unigrids.x2006.x04.services.sms.ListDirectoryDocument;
import org.unigrids.x2006.x04.services.sms.ListDirectoryResponseDocument;
import org.unigrids.x2006.x04.services.sms.ListDirectoryResponseDocument.ListDirectoryResponse;
import org.unigrids.x2006.x04.services.sms.ListPropertiesDocument;
import org.unigrids.x2006.x04.services.sms.ListPropertiesResponseDocument;
import org.unigrids.x2006.x04.services.sms.MetadataServiceReferenceDocument;
import org.unigrids.x2006.x04.services.sms.PermissionsChangeModeType;
import org.unigrids.x2006.x04.services.sms.PermissionsClassType;
import org.unigrids.x2006.x04.services.sms.ReceiveFileDocument;
import org.unigrids.x2006.x04.services.sms.ReceiveFileResponseDocument;
import org.unigrids.x2006.x04.services.sms.RenameDocument;
import org.unigrids.x2006.x04.services.sms.RenameResponseDocument;
import org.unigrids.x2006.x04.services.sms.SendFileDocument;
import org.unigrids.x2006.x04.services.sms.SendFileResponseDocument;
import org.unigrids.x2006.x04.services.sms.StoragePropertiesDocument;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.MetadataManagement;
import de.fzj.unicore.uas.StorageManagement;
import de.fzj.unicore.uas.UAS;
import de.fzj.unicore.uas.UASProperties;
import de.fzj.unicore.uas.client.BaseUASClient;
import de.fzj.unicore.uas.client.StorageClient;
import de.fzj.unicore.uas.fts.FileTransferImpl;
import de.fzj.unicore.uas.fts.ProtocolRP;
import de.fzj.unicore.uas.fts.ServerToServerFileTransferImpl;
import de.fzj.unicore.uas.fts.UpdatingProtocolRP;
import de.fzj.unicore.uas.impl.PersistingPreferencesResource;
import de.fzj.unicore.uas.impl.UmaskResourceProperty;
import de.fzj.unicore.uas.impl.UmaskResourceProperty.UmaskChangedListener;
import de.fzj.unicore.uas.impl.bp.BPSupportImpl;
import de.fzj.unicore.uas.impl.enumeration.EnumerationImpl;
import de.fzj.unicore.uas.metadata.BaseMetadataManagementImpl;
import de.fzj.unicore.uas.metadata.MetadataManagementImpl;
import de.fzj.unicore.uas.metadata.MetadataManager;
import de.fzj.unicore.uas.metadata.MetadataSupport;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.uas.xnjs.StorageAdapterFactory;
import de.fzj.unicore.uas.xnjs.TSIStorageAdapterFactory;
import de.fzj.unicore.uas.xnjs.XNJSFacade;
import de.fzj.unicore.wsrflite.ContainerProperties;
import de.fzj.unicore.wsrflite.Home;
import de.fzj.unicore.wsrflite.exceptions.InvalidModificationException;
import de.fzj.unicore.wsrflite.messaging.Message;
import de.fzj.unicore.wsrflite.messaging.PullPoint;
import de.fzj.unicore.wsrflite.messaging.ResourceAddedMessage;
import de.fzj.unicore.wsrflite.messaging.ResourceDeletedMessage;
import de.fzj.unicore.wsrflite.persistence.Persist;
import de.fzj.unicore.wsrflite.security.VODescription;
import de.fzj.unicore.wsrflite.security.util.AuthZAttributeStore;
import de.fzj.unicore.wsrflite.utils.Utilities;
import de.fzj.unicore.wsrflite.utils.WSServerUtilities;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import de.fzj.unicore.wsrflite.xmlbeans.ResourceProperty;
import de.fzj.unicore.wsrflite.xmlbeans.rp.AddressResourceProperty;
import de.fzj.unicore.xnjs.ems.ExecutionException;
import de.fzj.unicore.xnjs.io.ACLEntry;
import de.fzj.unicore.xnjs.io.ACLEntry.Type;
import de.fzj.unicore.xnjs.io.ChangeACL;
import de.fzj.unicore.xnjs.io.ChangeACL.ACLChangeMode;
import de.fzj.unicore.xnjs.io.ChangePermissions.Mode;
import de.fzj.unicore.xnjs.io.ChangePermissions.PermissionsClass;
import de.fzj.unicore.xnjs.io.CompositeFindOptions;
import de.fzj.unicore.xnjs.io.IStorageAdapter;
import de.fzj.unicore.xnjs.io.Permissions;
import de.fzj.unicore.xnjs.io.XnjsFile;
import de.fzj.unicore.xnjs.io.XnjsFileWithACL;
import de.fzj.unicore.xnjs.simple.LocalTS;
import de.fzj.unicore.xnjs.tsi.TSI;
import de.fzj.unicore.xnjs.util.URIUtils;
import eu.unicore.security.Client;
import eu.unicore.security.OperationType;
import eu.unicore.util.ConcurrentAccess;
import eu.unicore.util.Log;

/**
 * Basic Storage Management service implementation
 * 
 * @author schuller
 */
public abstract class SMSBaseImpl extends PersistingPreferencesResource implements StorageManagement{

	private static final Logger logger=LogUtil.getLogger(LogUtil.SERVICES,SMSBaseImpl.class);

	/**
	 * the maximum number of results to return in a single ListDirectory() operation
	 */
	public static final int MAX_LS_RESULTS=50000;
	
	//initparam keys
	public static final String INIT_STORAGE_DESCRIPTION=SMSBaseImpl.class.getName()+"_storageDescription";
	public static final String INIT_DIRECTORY_APPEND_UNIQUE_ID=SMSBaseImpl.class.getName()+"_appenduniqueid";
	public static final String INIT_FACTORYID=SMSBaseImpl.class.getName()+"_factoryid";
	public static final String INIT_DISABLE_METADATA=SMSBaseImpl.class.getName()+"_disable_metadata";
	public static final String INIT_INHERIT_SHARING=SMSBaseImpl.class.getName()+"_inherit_sharing";
	public static final String INIT_SKIP_EXISTENCECHECK=SMSBaseImpl.class.getName()+"skip_initial_exist_check";

	private static final Set<String> AUTO_NEGOTIATE_SCHEMES = new HashSet<String>(
			Arrays.asList(UAS.AUTO_NEGOTIATE_FT_PROTOCOL));
	private static final Cache protocolCache = initProtocolCache();

	public static final QName RPInternalFiletransferReference=new QName(SMS_NS,"internal_list_of_filetransfers");

	@Persist 
	protected StorageDescription storageDescription;
	
	/**
	 * the directory in which the storage service operates 
	 */
	@Persist
	protected String workdir;

	@Persist
	protected Boolean enableDirectFiletransfer=Boolean.FALSE;

	@Persist
	protected Boolean disableMetadata=Boolean.FALSE;

	@Persist
	protected Boolean inheritSharing=Boolean.FALSE;

	//enable/disable concurrent method execution
	private static final boolean allowConcurrency=true;

	/**
	 * the unique ID of the factory service that created this instance (can be null)
	 */
	@Persist
	protected String storageFactoryID=null;

	@Override
	protected void addWSResourceInterfaces(BPSupportImpl baseProfile) {
		super.addWSResourceInterfaces(baseProfile);
		baseProfile.addWSResourceInterface(SMS_PORT);
	}

	@Override
	public QName getPortType() {
		return SMS_PORT;
	}

	@Override
	public QName getResourcePropertyDocumentQName() {
		return StoragePropertiesDocument.type.getDocumentElementName();
	}

	//TODO - how this should be ported to USE 2.2 API where WSRFRepresentation is used?
	/*
	@Override
	protected void appendAdditionalRPs(XmlObject propertyDoc,
			Set<QName> remainingQNames) throws Exception {
		remainingQNames.remove(RPInternalFiletransferReference);
		super.appendAdditionalRPs(propertyDoc, remainingQNames);
	}
	*/

	@ConcurrentAccess(allow=allowConcurrency)
	public ChangePermissionsResponseDocument ChangePermissions(ChangePermissionsDocument inDoc) 
			throws BaseFault {
		try{
			org.unigrids.x2006.x04.services.sms.ChangePermissionsDocument.ChangePermissions in = 
					inDoc.getChangePermissions();
			String file=makeSMSLocal(in.getPath());
			IStorageAdapter tsi=getStorageAdapter();

			boolean recursive = in.isSetRecursive() ? in.getRecursive() : false;

			//chmod
			ExtendedChangePermissionsType extendedCh = in.getExtendedPermissions();
			if (extendedCh == null || extendedCh.isNil()) {
				if (in.getPermissions() != null)
					legacyChangePermissions(file, tsi, in);
			} else
				extendedChangePermissions(file, tsi, extendedCh, recursive);

			//chgrp
			String newGroup = in.getChangeOwningGroup();
			if (in.isSetChangeOwningGroup() && newGroup != null)
				tsi.chgrp(file, newGroup, recursive);

			//setfacl
			ChangeACLType aclChange = in.getACL();
			if (in.isSetACL() && aclChange != null)
				setACL(file, tsi, aclChange, recursive);

			ChangePermissionsResponseDocument res=ChangePermissionsResponseDocument.Factory.newInstance();
			res.addNewChangePermissionsResponse();
			return res;
		}
		catch(Exception e){
			LogUtil.logException("Could not change permissions.",e,logger);
			throw BaseFault.createFault("Could not change permissions. Reason: " + e.toString());
		}
	}

	public static void extendedChangePermissions(String file, IStorageAdapter tsi, 
			ExtendedChangePermissionsType extendedCh, boolean recursive) throws ExecutionException {
		ChangePermissionsEntryType request[] = extendedCh.getChangePermissionsEntryArray();
		if (request == null)
			throw new ExecutionException("Invalid request - null change perms array");
		de.fzj.unicore.xnjs.io.ChangePermissions xnjsRequest[] = 
				new de.fzj.unicore.xnjs.io.ChangePermissions[request.length];
		for (int i=0; i<request.length; i++) {
			xnjsRequest[i] = new de.fzj.unicore.xnjs.io.ChangePermissions();
			if (request[i].getKind() == null || request[i].getMode() == null ||
					request[i].getPermissions() == null)
				throw new ExecutionException("Invalid request - null change perms entries");

			if (request[i].getKind().equals(PermissionsClassType.GROUP))
				xnjsRequest[i].setClazz(PermissionsClass.GROUP);
			else if (request[i].getKind().equals(PermissionsClassType.USER))
				xnjsRequest[i].setClazz(PermissionsClass.OWNER);
			else
				xnjsRequest[i].setClazz(PermissionsClass.OTHER);

			if (request[i].getMode().equals(PermissionsChangeModeType.ADD))
				xnjsRequest[i].setMode(Mode.ADD);
			else if (request[i].getMode().equals(PermissionsChangeModeType.SUBTRACT))
				xnjsRequest[i].setMode(Mode.SUBTRACT);
			else
				xnjsRequest[i].setMode(Mode.SET);

			xnjsRequest[i].setPermissions(request[i].getPermissions());
		}		
		tsi.chmod2(file, xnjsRequest, recursive);
	}

	public static void setACL(String file, IStorageAdapter tsi, ChangeACLType aclChange, boolean recursive) 
			throws ExecutionException {
		ChangeACLEntryType request[] = aclChange.getChangeACLEntryArray();
		if (request == null)
			throw new ExecutionException("Invalid request - null change ACL array");
		ChangeACL[] xnjsRequest = new ChangeACL[request.length];
		for (int i=0; i<request.length; i++) {
			if (request[i].getKind() == null || request[i].getMode() == null ||
					request[i].getPermissions() == null)
				throw new ExecutionException("Invalid request - null change ACL entries");
			xnjsRequest[i] = new ChangeACL();
			if (request[i].getMode().equals(ACLChangeModeType.MODIFY)) {
				xnjsRequest[i].setChangeMode(ACLChangeMode.MODIFY);
				xnjsRequest[i].setPermissions(request[i].getPermissions());
			} else
				xnjsRequest[i].setChangeMode(ACLChangeMode.REMOVE);

			xnjsRequest[i].setSubject(request[i].getSubject());

			if (request[i].getKind().equals(ACLEntryTypeType.GROUP))
				xnjsRequest[i].setType(Type.GROUP);
			else
				xnjsRequest[i].setType(Type.USER);

			if (request[i].isSetDefaultACL() && request[i].getDefaultACL())
				xnjsRequest[i].setDefaultACL(true);
			else
				xnjsRequest[i].setDefaultACL(false);
		}

		boolean clearAll = aclChange.isSetClearACL() && aclChange.getClearACL();
		tsi.setfacl(file, clearAll, xnjsRequest, recursive);
	}

	public static void legacyChangePermissions(String file, IStorageAdapter tsi, org.unigrids.x2006.x04.services.sms.ChangePermissionsDocument.ChangePermissions in) 
			throws ExecutionException {
		PermissionsType p=in.getPermissions();
		Permissions permissions=new Permissions(p.getReadable(), p.getWritable(), p.getExecutable());
		tsi.chmod(file, permissions);
		if(logger.isDebugEnabled()){
			logger.debug("Changed user permissions for '"+file+"' to <"+permissions.toString()+">");
		}
	}

	@ConcurrentAccess(allow=allowConcurrency)
	public CopyResponseDocument Copy(CopyDocument in) throws BaseFault {
		try{
			String source=makeSMSLocal(in.getCopy().getSource());
			String target=makeSMSLocal(in.getCopy().getDestination());
			IStorageAdapter tsi=getStorageAdapter();
			tsi.cp(source, target);
			if(logger.isDebugEnabled())logger.debug("Copied '"+source+"' -> '"+target+"'");
			CopyResponseDocument res=CopyResponseDocument.Factory.newInstance();
			res.addNewCopyResponse();
			return res;
		}
		catch(Exception e){
			LogUtil.logException("Could not copy.",e,logger);
			throw BaseFault.createFault("Could not copy. Reason: "+e.getClass().getName());
		}
	}

	/**
	 * retrieve the base dir of this storage</br>
	 * It is assumed to end with the file separator
	 * 
	 * @return
	 */
	protected abstract String getStorageRoot() throws ExecutionException;

	/**
	 * create a directory: assumes path relative to storage root
	 */
	@ConcurrentAccess(allow=allowConcurrency)
	public CreateDirectoryResponseDocument CreateDirectory(CreateDirectoryDocument in) throws BaseFault {
		try{
			String path=makeSMSLocal(in.getCreateDirectory().getPath());
			IStorageAdapter tsi=getStorageAdapter();
			tsi.mkdir(path);
			logger.debug("Created directory "+path);
		}catch(Exception e){
			LogUtil.logException("Could not create directory",e,logger);
			throw new BaseFault("Could not create directory.");
		}
		CreateDirectoryResponseDocument res=CreateDirectoryResponseDocument.Factory.newInstance();
		res.addNewCreateDirectoryResponse();
		return res;
	}

	@ConcurrentAccess(allow=allowConcurrency)
	public DeleteResponseDocument Delete(DeleteDocument in) throws BaseFault {
		try{
			String path=makeSMSLocal(in.getDelete().getPath());
			IStorageAdapter tsi=getStorageAdapter();
			tsi.rmdir(path);
			if(logger.isDebugEnabled())logger.debug("Deleted '"+path+"'");
			DeleteResponseDocument res=DeleteResponseDocument.Factory.newInstance();
			res.addNewDeleteResponse();
			return res;
		}
		catch(Exception e){
			LogUtil.logException("Could not perform delete.",e,logger);
			throw BaseFault.createFault("Could not perform delete. Reason: "+e.getClass().getName());
		}

	}

	/**
	 * export a file<br/>
	 * 
	 * the path will always be interpreted as relative to storage root 
	 */
	@ConcurrentAccess(allow=allowConcurrency)
	public ExportFileResponseDocument ExportFile(ExportFileDocument in) throws BaseFault {
		try {
			//check if file exists
			String source=makeSMSLocal(in.getExportFile().getSource());
			XnjsFileWithACL file=getProperties(source);
			if(file==null){
				throw new FileNotFoundException("File <"+source+"> not found on storage");
			}
			ExportFileResponseDocument res=ExportFileResponseDocument.Factory.newInstance();
			//fill in some init params...
			Map<String,Object> map=new HashMap<String,Object>();
			Boolean isPipe=Boolean.valueOf(in.getExportFile().getIsPipe());
			ProtocolType.Enum protocol=in.getExportFile().getProtocol();
			map.put(FileTransferImpl.INIT_SOURCE, source);
			map.put(FileTransferImpl.INIT_IS_PIPE, isPipe);
			map.put(FileTransferImpl.INIT_PROTOCOL,protocol);
			map.put(FileTransferImpl.INIT_IS_EXPORT, Boolean.TRUE);
			map.put(FileTransferImpl.INIT_UMASK, getUmask());
			ExtraParameters param=in.getExportFile().getExtraParameters();
			if(param!=null){
				map.put(FileTransferImpl.INIT_EXTRA_PARAMETERS, parseExtraParameters(param));
			}
			if(logger.isDebugEnabled()){
				logger.debug("Export: '"+source+"' "+protocol+" from "+getStorageRoot());
			}
			//create export file resource...
			EndpointReferenceType epr=createFileTransfer(map,protocol);
			res.addNewExportFileResponse().setExportEPR(epr);
			return res;
		} catch (Exception e) {
			LogUtil.logException("Could not create file export.",e,logger);
			String errMsg=LogUtil.createFaultMessage("Could not create file export", e);
			throw BaseFault.createFault(errMsg);
		}
	}

	
	@ConcurrentAccess(allow=allowConcurrency)
	public ImportFileResponseDocument ImportFile(ImportFileDocument in) throws BaseFault {
		try {
			ImportFileResponseDocument res=ImportFileResponseDocument.Factory.newInstance();
			//fill in some init params...
			Map<String,Object> map=new HashMap<String,Object>();
			String target=makeSMSLocal(in.getImportFile().getDestination());
			checkImportTarget(target);
			Boolean isPipe=Boolean.valueOf(in.getImportFile().getIsPipe());
			ProtocolType.Enum protocol=in.getImportFile().getProtocol();

			map.put(FileTransferImpl.INIT_TARGET, target);
			map.put(FileTransferImpl.INIT_IS_PIPE, isPipe);
			map.put(FileTransferImpl.INIT_PROTOCOL,protocol);
			map.put(FileTransferImpl.INIT_SMS_WORKDIR, getStorageRoot());
			map.put(FileTransferImpl.INIT_IS_EXPORT, Boolean.FALSE);
			map.put(FileTransferImpl.INIT_UMASK, getUmask());
			Boolean overwrite=Boolean.TRUE;
			if(in.getImportFile().isSetOverwrite()){
				overwrite=Boolean.valueOf(in.getImportFile().getOverwrite());
			}
			map.put(FileTransferImpl.INIT_OVERWRITE, overwrite);
			ExtraParameters param=in.getImportFile().getExtraParameters();
			if(param!=null){
				map.put(FileTransferImpl.INIT_EXTRA_PARAMETERS, parseExtraParameters(param));
			}
			if(logger.isDebugEnabled()){
				logger.debug("Import: '"+target+"' "+protocol+" to "+getStorageRoot()+" overwrite="+overwrite);
			}
			//create any missing directories
			createParentDirectories(target);
			//create import file resource...
			EndpointReferenceType epr=createFileTransfer(map, protocol);
			res.addNewImportFileResponse().setImportEPR(epr);
			return res;
		} catch (Exception e) {
			String msg=LogUtil.createFaultMessage("Could not create file import",e);
			LogUtil.logException("Could not create file import",e,logger);
			throw BaseFault.createFault(msg);
		}
	}

	/**
	 * perform sanity checks on the target of an import<br/>
	 * <ul>
	 *   <li>it must not be a directory
	 * </ul>
	 * 
	 * @param target - the target name for an import
	 * @throws ExecutionException if problems with XNJS/TSI occur
	 * @throws IllegalArgumentException - if target is invalid
	 */
	protected void checkImportTarget(String target)throws Exception{
		XnjsFile f=getProperties(target);
		if(f==null)return;
		if(f.isDirectory())throw new IllegalArgumentException("Invalid target filename "+target+": is a directory");
	}

	/**
	 * create a filetransfer that imports data from a remote SMS
	 * 
	 * @param in
	 * @return
	 * @throws BaseFault
	 */
	@ConcurrentAccess(allow=true)
	public ReceiveFileResponseDocument ReceiveFile(ReceiveFileDocument in)
			throws BaseFault{
		String source=in.getReceiveFile().getSource();
		String target=in.getReceiveFile().getDestination();
		try{
			URI sourceURI = new URI(urlEncode(source));

			Map<String,Object> map=new HashMap<String,Object>();
			Boolean isPipe=Boolean.FALSE;
			if(!target.startsWith(getSeparator()))target=getSeparator()+target;

			if(isAutoNegotiateScheme(sourceURI.getScheme()))
			{
				ProtocolType.Enum protocol = autoNegotiateProtocol(sourceURI);
				source = new URI(protocol.toString(),sourceURI.getSchemeSpecificPart(),sourceURI.getFragment()).toString();
			}
			map.put(FileTransferImpl.INIT_TARGET, target);
			map.put(FileTransferImpl.INIT_IS_PIPE, isPipe);
			map.put(FileTransferImpl.INIT_SOURCE, source);
			map.put(FileTransferImpl.INIT_IS_EXPORT, Boolean.FALSE);
			map.put(FileTransferImpl.INIT_UMASK, getUmask());
			if(in.getReceiveFile().isSetReliableMode()){
				map.put(ServerToServerFileTransferImpl.INIT_RELIABLE, Boolean.valueOf(in.getReceiveFile().getReliableMode()));
			}
			else {
				map.put(ServerToServerFileTransferImpl.INIT_RELIABLE,Boolean.FALSE);
			}
			ExtraParameters param=in.getReceiveFile().getExtraParameters();
			if(param!=null){
				map.put(FileTransferImpl.INIT_EXTRA_PARAMETERS, parseExtraParameters(param));
			}
			//create transfer resource...
			EndpointReferenceType epr=createTransferResource(map);

			ReceiveFileResponseDocument res=ReceiveFileResponseDocument.Factory.newInstance();
			res.addNewReceiveFileResponse().setReceiveFileEPR(epr);
			logger.debug("Receiving data from <"+source+"> target is <"+target+">");
			return res;
		}catch(Exception e){
			String msg=LogUtil.createFaultMessage("Could not initiate receive file.",e);
			LogUtil.logException("Could not initiate receive file.",e,logger);
			throw BaseFault.createFault(msg);
		}

	}

	/**
	 * create a filetransfer that pushes data to a remote SMS
	 * 
	 * @param in
	 * @return
	 * @throws BaseFault
	 */
	@ConcurrentAccess(allow=true)
	public SendFileResponseDocument SendFile(SendFileDocument in)
			throws BaseFault {
		String source = in.getSendFile().getSource();
		String target = in.getSendFile().getDestination();
		try {
			XnjsFileWithACL xnjsFile = getProperties(source);
			if(xnjsFile == null)
			{
				throw new Exception("File "+source+" not found on this storage.");
			}

			URI targetURI = new URI(urlEncode(target));
			if(isAutoNegotiateScheme(targetURI.getScheme()))
			{
				ProtocolType.Enum protocol = null;
				if(xnjsFile.getSize() > 1024*1024)
				{
					// file smaller than 1 MB ? => fallback to BFT
					protocol = ProtocolType.BFT;
				}
				else
				{
					protocol = autoNegotiateProtocol(targetURI);
				}
				target = new URI(protocol.toString(),targetURI.getSchemeSpecificPart(),targetURI.getFragment()).toString();
			}
			Map<String, Object> map = new HashMap<String, Object>();
			Boolean isPipe = Boolean.FALSE;
			if (!source.startsWith(getSeparator()))
				source = getSeparator() + source;
			map.put(FileTransferImpl.INIT_TARGET, target);
			map.put(FileTransferImpl.INIT_IS_PIPE, isPipe);
			map.put(FileTransferImpl.INIT_SOURCE, source);
			map.put(FileTransferImpl.INIT_IS_EXPORT, Boolean.TRUE);
			map.put(FileTransferImpl.INIT_UMASK, getUmask());
			ExtraParameters param=in.getSendFile().getExtraParameters();
			if(param!=null){
				map.put(FileTransferImpl.INIT_EXTRA_PARAMETERS, parseExtraParameters(param));
			}
			// create transfer resource...
			EndpointReferenceType epr = createTransferResource(map);

			SendFileResponseDocument res = SendFileResponseDocument.Factory.newInstance();
			res.addNewSendFileResponse().setSendFileEPR(epr);
			logger.debug("Sending data to <"+target+"> source is <"+source+">");
			return res;
		} catch (Exception e) {
			String msg=LogUtil.createFaultMessage("Could not initiate send file", e);
			LogUtil.logException("Could not initiate send file",e,logger);
			throw BaseFault.createFault(msg);
		}

	}

	/**
	 * creates server-server transfer. Common things (like security info) will be added to 
	 * the initParameter map.<br/>
	 * Side effect: the EPR is added to the relevant resource property
	 * @param initParam - basic parameters 
	 * @return {@link EndpointReferenceType} of the new ws resource
	 * @throws Exception
	 */
	protected EndpointReferenceType createTransferResource(Map<String,Object> initParam)
			throws Exception{
		//add common things to the init map
		initParam.put(FileTransferImpl.INIT_SMS_EPR,WSServerUtilities.makeEPR(getServiceName(),getUniqueID(),kernel));
		initParam.put(FileTransferImpl.INIT_SMS_WORKDIR, getStorageRoot());
		initParam.put(FileTransferImpl.INIT_STORAGE_ADAPTER_FACTORY, getStorageAdapterFactory());
		initParam.put(INITPARAM_XNJS_REFERENCE, xnjsReference);
		initParam.put(INIT_PARENT_NODE, nodeHelper);

		String id=kernel.getHome(UAS.FTS_BASE).createWSRFServiceInstance(initParam);
		ResourceAddedMessage m=new ResourceAddedMessage(UAS.FTS_BASE,id);
		getKernel().getMessaging().getChannel(getUniqueID()).publish(m);
		
		EndpointReferenceType epr = WSServerUtilities.makeEPR(UAS.FTS_BASE,id,kernel);
		WSServerUtilities.addUGSRefparamToEpr(epr,id);

		return epr;
	}

	public ListDirectoryResponseDocument ListDirectory(ListDirectoryDocument in) throws BaseFault {
		ListDirectoryResponseDocument resd=ListDirectoryResponseDocument.Factory.newInstance();
		ListDirectoryResponse res=resd.addNewListDirectoryResponse();
		BigInteger offsetP=in.getListDirectory().getOffset();
		int offset=( offsetP!=null?offsetP.intValue():0) ;
		BigInteger limitP=in.getListDirectory().getLimit();
		int max=uasProperties.getIntValue(UASProperties.SMS_LS_LIMIT);
		int limit=limitP!=null?limitP.intValue():max;
		if(limit>max){
			String msg="Could not list directory: the requested number of results " +
					"exceeds the internal limit of <"+max+">. " +
					"Please use the limit and offset parameters!";
			logger.warn(msg);
			throw BaseFault.createFault(msg);
		}
		//get list from tsi
		try{
			String p=makeSMSLocal(in.getListDirectory().getPath());
			if(logger.isDebugEnabled()){
				logger.debug("Listing '"+p+"' workdir="+getStorageRoot());
			}
			XnjsFile[] tsifiles=getListing(p, offset, limit, storageDescription.isFilterListing());
			for(XnjsFile f: tsifiles){
				GridFileType gf=res.addNewGridFile();
				convert(f,gf);
				if(logger.isTraceEnabled())logger.trace("XNJS filepath: "+f.getPath()+" -> "+gf.getPath());
			}
		}catch(Exception e){
			String msg="Could not list directory: "+e.getMessage();
			LogUtil.logException("Could not list directory",e,logger);
			throw BaseFault.createFault(msg);
		}
		return resd;
	}

	/**
	 * gets the contents for directory 'path'
	 * @param path - directory relative to storage root
	 * @return an array of {@link XnjsFile}
	 * @throws ExecutionException
	 */
	protected XnjsFile[] getListing(String path, int offset, int limit, boolean filter)throws Exception{
		IStorageAdapter tsi=getStorageAdapter();
		return tsi.ls(path,offset,limit,filter);
	}

	/**
	 * gets the properties for 'path'
	 * @param path - a path relative to storage root
	 * 
	 * @return {@link XnjsFile} or <code>null</code> if path does not exist
	 * @throws Exception in case of problems performing the request
	 */
	protected XnjsFileWithACL getProperties(String path)throws Exception{
		IStorageAdapter tsi=getStorageAdapter();
		return tsi.getProperties(path);
	}

	/**
	 * convert a path to normal form.<br/> 
	 * <ul>
	 * <li>Replace "\" with "/" </li>
	 * <li>make sure the path starts with "/" </li>
	 * <li>decode special characters (spaces, ...) using {@link #urlDecode(String)}</li>
	 * <li>replace duplicate slashes "//" with a single "/" </li>
	 * </ul>
	 */
	protected String makeSMSLocal(String p){
		String res=urlDecode(p).replace('\\','/');
		if(res.length()==0 || '/'!=res.charAt(0))res="/"+res;
		res=res.replace("//", "/");
		if(logger.isTraceEnabled())logger.trace("Got path:"+p+", converted to "+res);
		return res;
	}

	/**
	 * URL-decode a string (e.g. replacing "%20" and "+" by spaces)
	 * 
	 * @param p - the string to decode
	 * @return
	 */
	public static String urlDecode(String p){
		try{
			return URLDecoder.decode(p.replaceAll("\\+","%20"), "UTF-8");
		}catch(Exception ex){
			logger.warn(ex);
			return p;
		}
	}

	/**
	 * fix illegal characters (like spaces) in the parameter,
	 * so a URL can be built from it
	 * @param p
	 * @return
	 */
	public static String urlEncode(String orig){
		try{
			return URIUtils.encodeAll(orig);
		}catch(Exception e){
			logger.error(e);
			return orig;
		}
	}

	/**
	 * converts from XNJSfile to GridFile
	 * 
	 * @param f - the XNJS file info
	 * @param gf - the Grid file info
	 * @param addMetadata - whether to attach metadata
	 */
	protected void convert(XnjsFile f, GridFileType gf, boolean addMetadata)throws Exception{
		gf.setPath(f.getPath());
		gf.setSize(f.getSize());
		gf.setIsDirectory(f.isDirectory());
		gf.setLastModified(f.getLastModified());
		Permissions p=f.getPermissions();
		gf.addNewPermissions().setReadable(p.isReadable());
		gf.getPermissions().setWritable(p.isWritable());
		gf.getPermissions().setExecutable(p.isExecutable());
		if(addMetadata){
			MetadataManager mm=getMetadataManager();
			try{
				if (mm!= null)
					attachMetadata(f,gf,mm);
			}catch(Exception ex){
				LogUtil.logException("Error attaching metadata for file "+f.getPath(), ex);
			}
		}
		if (f.getUNIXPermissions() != null)
			gf.setFilePermissions(f.getUNIXPermissions());
		if (f.getGroup() != null)
			gf.setGroup(f.getGroup());
		if (f.getOwner() != null)
			gf.setOwner(f.getOwner());

		if (f instanceof XnjsFileWithACL) {
			XnjsFileWithACL fWithACL = (XnjsFileWithACL) f;
			ACLEntry[] acl = fWithACL.getACL();
			if (acl != null) {
				ACLType xmlACL = gf.addNewACL();
				for (ACLEntry aclEntry: acl) {
					ACLEntryType xmlACLEntry = xmlACL.addNewEntry();
					xmlACLEntry.setPermissions(aclEntry.getPermissions());
					xmlACLEntry.setSubject(aclEntry.getSubject());
					if (aclEntry.getType() == ACLEntry.Type.GROUP)
						xmlACLEntry.setType(ACLEntryTypeType.GROUP);
					else
						xmlACLEntry.setType(ACLEntryTypeType.USER);
					xmlACLEntry.setDefaultACL(aclEntry.isDefaultACL());
				}
			}
		}
	}

	protected void attachMetadata(XnjsFile f, GridFileType gridFile, MetadataManager metaManager)throws Exception{
		if(gridFile.getIsDirectory())return;
		String resourceName=gridFile.getPath();
		Map<String,String> metadata=metaManager.getMetadataByName(resourceName);
		MetadataType md=gridFile.addNewMetadata();
		for(Map.Entry<String, String> item: metadata.entrySet()){
			String key=item.getKey();
			String value=item.getValue();
			if("Content-MD5".equalsIgnoreCase(key)){
				md.setContentMD5(value);
			}
			else if("Content-Type".equalsIgnoreCase(key)){
				md.setContentType(value);
			}
			else{
				TextInfoType t=md.addNewProperty();
				t.setName(key);
				t.setValue(value);
			}
		}
	}

	/**
	 * converts from XNJSfile to GridFile
	 * 
	 * @param f, gf
	 */
	protected void convert(XnjsFile f, GridFileType gf)throws Exception{
		convert(f,gf,false);
	}


	public ListPropertiesResponseDocument ListProperties(ListPropertiesDocument in) throws BaseFault {
		String request=in.getListProperties().getPath();
		if(request==null)throw BaseFault.createFault("Could not list properties: target path is null.");
		ListPropertiesResponseDocument res=ListPropertiesResponseDocument.Factory.newInstance();
		res.addNewListPropertiesResponse();
		try{
			String path=makeSMSLocal(request);
			XnjsFileWithACL file=getProperties(path);
			if(file!=null){
				GridFileType gf=res.getListPropertiesResponse().addNewGridFile();
				convert(file, gf, true);
			}
		}catch(Exception e){
			LogUtil.logException("Could not list properties of <"+request+">",e,logger);
			throw BaseFault.createFault("Could not list properties of <"+request+">. Reason: "+e.getClass().getName());
		}
		return res;
	}

	public RenameResponseDocument Rename(RenameDocument in) throws BaseFault {
		try{
			String source=makeSMSLocal(in.getRename().getSource());
			String target=makeSMSLocal(in.getRename().getDestination());
			IStorageAdapter tsi=getStorageAdapter();
			tsi.rename(source, target);
			if(logger.isDebugEnabled())logger.debug("Renamed '"+source+"' -> '"+target+"'");
			RenameResponseDocument res=RenameResponseDocument.Factory.newInstance();
			res.addNewRenameResponse();
			return res;
		}
		catch(Exception e){
			LogUtil.logException("Could not rename.",e,logger);
			throw BaseFault.createFault("Could not rename. Reason: "+e.getClass().getName());
		}
	}

	public FindResponseDocument Find(FindDocument in) throws BaseFault {
		FindResponseDocument resd=FindResponseDocument.Factory.newInstance();
		FindResponse res=resd.addNewFindResponse();
		try{
			Find find=in.getFind();
			String base=find.getBase();
			CompositeFindOptions opts=XNJSFacade.get(xnjsReference,kernel).getFindOptions(find.getFilter());
			if(find.isSetRecurse()){
				opts.setRecurse(find.getRecurse());
			}
			//perform find on tsi
			String p=makeSMSLocal(base);
			if(logger.isTraceEnabled()){
				logger.trace("Listing <"+p+"> workdir="+getStorageRoot());
			}
			IStorageAdapter tsi=getStorageAdapter();
			XnjsFile[] tsifiles=tsi.find(base, opts, -1, -1);
			for(XnjsFile f: tsifiles){
				if(storageDescription.isFilterListing() && !f.isOwnedByCaller()){
					if(logger.isTraceEnabled())logger.trace("Skipping "+f.getPath()+", not owned by caller.");
					continue;
				}
				GridFileType gf=res.addNewGridFile();
				convert(f,gf);
				if(logger.isTraceEnabled())logger.trace("XNJS filepath: "+f.getPath()+" -> "+gf.getPath());
			}

			return resd;
		}
		catch(Exception e){
			LogUtil.logException("Could not perform find operation.",e,logger);
			throw BaseFault.createFault("Could not perform find operation. Reason: "+e.getClass().getName());
		}
	}

	@Override
	public void initialise(String name, Map<String,Object>initobjs)throws Exception{
		super.initialise(name, initobjs);

		storageDescription = (StorageDescription) 
				initobjs.get(INIT_STORAGE_DESCRIPTION);
		if (storageDescription == null)
			throw new IllegalStateException("No storage configuration found");
		
		properties.put(RPProtocol,new UpdatingProtocolRP(this, storageDescription));

		String fsname = storageDescription.getName();
		properties.put(RPFileSystem, makeFileSystemRP(fsname));

		properties.put(ACLSupported, new ACLSupportedRP(this, fsname));

		addUmaskRP(storageDescription.getDefaultUmask());

		//factory ID
		storageFactoryID=(String)initobjs.get(INIT_FACTORYID);

		//check if metadata support should be disabled
		disableMetadata=(Boolean)initobjs.get(INIT_DISABLE_METADATA);
		if(disableMetadata==null)disableMetadata=Boolean.FALSE;

		Boolean inheritSharing = (Boolean)initobjs.get(INIT_INHERIT_SHARING);
		if (inheritSharing != null)
			this.inheritSharing = inheritSharing;
		
		//list of server-server transfers this is NOT published
		//only used by the enumeration
		properties.put(RPInternalFiletransferReference, new FiletransferReferenceRP(this));

		//enumeration for server-server transfers
		String enumerationID=createFTListEnumeration();
		AddressResourceProperty ftListAddress=new AddressResourceProperty(this, RPFiletransferEnumerationReference, 
				UAS.ENUMERATION+"?res="+enumerationID, true);
		properties.put(RPFiletransferEnumerationReference, ftListAddress);

		//publish this in registry?
		try{
			Boolean publishFlag=(Boolean)initobjs.get(INITPARAM_PUBLISH_TO_REGISTRY);
			if(Boolean.TRUE.equals(publishFlag)){
				logger.info("Publishing StorageManagement <"+getUniqueID()+"> to registry.");
				publish();
			}
		}catch(Exception e){
			LogUtil.logException("Problem publishing to registry",e,logger);
		}

	}

	/**
	 * create an instance of the Enumeration service for publishing
	 * the list of filetransfers
	 * @return
	 * @throws Exception
	 */
	protected String createFTListEnumeration()throws Exception{
		Map<String,Object>init=new HashMap<String, Object>();
		init.put(EnumerationImpl.INIT_TARGETSERVICE_EPR, this.getEPR());
		init.put(EnumerationImpl.INIT_TARGETSERVICE_RP, RPInternalFiletransferReference);
		Calendar c=Calendar.getInstance();
		c.add(Calendar.MONTH, 24);
		init.put(EnumerationImpl.INIT_INITIAL_TERMINATION_TIME, c);
		init.put(INIT_PARENT_NODE, getNode());
		Home h=kernel.getHome(UAS.ENUMERATION);
		if(h==null)throw new Exception("Enumeration service is not deployed!");
		return h.createWSRFServiceInstance(init);
	}

	//create an umask rp and add it to the properties map
	protected UmaskResourceProperty addUmaskRP(String umask){
		UmaskResourceProperty umaskRP = new UmaskResourceProperty(this, umask);
		umaskRP.setListener(new UmaskListenerImpl());
		properties.put(StorageManagement.RPUmask, umaskRP);
		return umaskRP;
	}

	/**
	 * Final to enforce invocation. smsPostActivate() is available for overwriting.
	 */
	@Override
	protected final void customPostActivate() {
		enableDirectFiletransfer = uasProperties.getBooleanValue(UASProperties.SMS_DIRECT_FILETRANSFER);
		UmaskResourceProperty umaskRP = (UmaskResourceProperty) properties.get(RPUmask);
		if(umaskRP==null){//can happen after server upgrade
			umaskRP=addUmaskRP(Integer.toOctalString(IStorageAdapter.DEFAULT_UMASK));
		}
		umaskRP.setListener(new UmaskListenerImpl());
		//check for deleted file transfers
		try{
			logger.trace("Getting messages from queue "+getUniqueID());
			PullPoint p=kernel.getMessaging().getPullPoint(getUniqueID());
			FiletransferReferenceRP rp=(FiletransferReferenceRP)properties.get(RPInternalFiletransferReference);

			while(p.hasNext()){
				Message message=p.next();
				if(message instanceof ResourceDeletedMessage){
					ResourceDeletedMessage rdm=(ResourceDeletedMessage)message;
					String id=rdm.getDeletedResource();
					String service=rdm.getServiceName();
					if(UAS.FTS_BASE.equals(service)){
						rp.remove(id);
						setDirty();	
					}
				}
				else if(message instanceof ResourceAddedMessage){
					ResourceAddedMessage ram=(ResourceAddedMessage)message;
					String id=ram.getAddedResource();
					String service=ram.getServiceName();
					if(UAS.FTS_BASE.equals(service)){
						rp.add(id);
						setDirty();	
					}	
				}
			}
		}catch(Exception e){
			LogUtil.logException(e.getMessage(),e,logger);
		}
		smsPostActivate();
	}

	protected void smsPostActivate() {

	}

	/**
	 * resource-specific destruction
	 */
	@Override
	public void destroy() {
		if(storageFactoryID!=null){
			try{
				ResourceDeletedMessage m=new ResourceDeletedMessage("deleted:"+getUniqueID());
				m.setDeletedResource(getUniqueID());
				m.setServiceName(getServiceName());
				kernel.getMessaging().getChannel(storageFactoryID).publish(m);
			}
			catch(Exception e){
				LogUtil.logException("Could not send internal message.",e,logger);
			}
		}

		if (storageDescription.isCleanupOnDestroy()) {
			try{
				//remove the storage root directory itself
				getStorageAdapter().rmdir("/");
			}catch(Exception ex){

			}
		}

		//destroy the metadata resource
		AddressResourceProperty mrp=(AddressResourceProperty)properties.get(RPMetadataServiceReference);
		if(mrp!=null){
			try{
				EndpointReferenceType uspace=((MetadataServiceReferenceDocument)mrp.update().getXml()[0]).getMetadataServiceReference();
				BaseUASClient c=new BaseUASClient(uspace, kernel.getClientConfiguration());
				c.destroy();
			}catch(Exception e){
				LogUtil.logException("Could not destroy metadata service instance.",e,logger);
			}
		}

		super.destroy();
	}

	/**
	 * get the list of protocols
	 * @param protocolList - the space separated list of protocols. If this is <code>null</code>, BFT and RBYTEIO will be enabled
	 */
	public static List<ProtocolType.Enum> parseStorageProtocols(String protocolList){
		ArrayList<ProtocolType.Enum>protocols=new ArrayList<ProtocolType.Enum>();
		if(protocolList!=null){
			logger.debug("Adding protocols <"+protocolList+">");
			try{
				String[] ps=protocolList.split(" +");
				for(String entry: ps){
					ProtocolType.Enum p=ProtocolType.Enum.forString(entry.toUpperCase());
					if(p!=null)protocols.add(p);
				}
			}catch(Exception e){
				logger.error("Can't add protocols", e);
			}
		}
		if(protocols.size()==0){
			logger.debug("No protocols configured for SMS. Falling back to 'RBYTEIO' and 'BFT'");
			protocols.add(ProtocolType.RBYTEIO);
			protocols.add(ProtocolType.BFT);
		}

		return protocols;
	}

	/**
	 * create new client-server FileTransfer resource and return its EPR
	 * 
	 * @param initParam - the initialisation parameter map
	 * @param protocol - the protocol to use
	 * @return EPR
	 */
	protected EndpointReferenceType createFileTransfer(Map<String,Object> initParam, ProtocolType.Enum protocol)
			throws Exception{
		//add common things to the map
		initParam.put(FileTransferImpl.INIT_SMS_EPR,WSServerUtilities.makeEPR(getServiceName(),getUniqueID(),kernel));
		initParam.put(FileTransferImpl.INIT_SMS_WORKDIR, getStorageRoot());
		initParam.put(FileTransferImpl.INIT_STORAGE_ADAPTER_FACTORY, getStorageAdapterFactory());
		initParam.put(INITPARAM_XNJS_REFERENCE, xnjsReference);
		initParam.put(INIT_PARENT_NODE, getNode());

		Home home=kernel.getHome(UAS.FTS_BASE+protocol.toString());
		if(home==null)throw new Exception("Requested service <"+UAS.FTS_BASE+protocol.toString()+"> is not available.");
		String id=home.createWSRFServiceInstance(initParam);
		EndpointReferenceType epr = WSServerUtilities.newEPR(kernel.getContainerSecurityConfiguration());
		if(!enableDirectFiletransfer){
			epr.addNewAddress().setStringValue(
					WSServerUtilities.makeAddress(UAS.FTS_BASE+protocol, id, 
							kernel.getContainerProperties()));
		}
		else{
			String serv=kernel.getContainerProperties().getValue(ContainerProperties.WSRF_SERVLETPATH);
			String baseAddr = Utilities.getPhysicalServerAddress(kernel.getContainerProperties(),
					kernel.getContainerSecurityConfiguration().isSslEnabled()); 
			String add=baseAddr+serv+"/"+UAS.FTS_BASE+protocol+"?res="+id;
			epr.addNewAddress().setStringValue(add);
			logger.debug("Direct filetransfer enabled, address= "+add);
		}
		WSServerUtilities.addUGSRefparamToEpr(epr,id);
		return epr;
	}


	private String sep=null;
	/**
	 * gets the file separator string. Override this if providing a non-TSI storage.
	 * @return the separator char returned by the {@link TSI#getFileSeparator()}
	 */
	protected String getSeparator(){
		if(sep==null){
			try{
				sep=getTSI().getFileSeparator();
			}catch(ExecutionException ex){
				LogUtil.logException("Could not get file separator", ex, logger);
				sep="/";
			}
		}
		return sep;
	}

	/**
	 * creates any missing directories
	 */
	protected void createParentDirectories(String target)throws Exception{
		while(target.startsWith("/"))target=target.substring(1);
		String[] dirs=target.split("/");
		String dir="";
		if(dirs.length>1 && dirs[0].length()!=0){
			//build directory
			int i=0;
			while(i<dirs.length-1){
				if(i>0)dir+="/";
				dir+=dirs[i];
				i++;
			}
		}
		if(dir.length()>0){
			String path=dir;
			IStorageAdapter tsi=getStorageAdapter();
			XnjsFile xDir=tsi.getProperties(path);
			if(xDir==null){
				logger.debug("Creating directory "+path);
				tsi.mkdir(path);	
			}
			else if(!xDir.isDirectory()){
				throw new IOException("</"+dir+"> already exists on this storage and is not a directory");
			}
		}
	}

	/**
	 * get the {@link IStorageAdapter} to use to access the backend storage for import
	 * and export of data.<br/>
	 */
	public IStorageAdapter getStorageAdapter()throws Exception{
		TSI tsi=getTSI();
		tsi.setStorageRoot(getStorageRoot());
		return tsi;
	}

	private TSI getTSI(){
		Client client=getClient();
		TSI ret = XNJSFacade.get(xnjsReference,kernel).getConfiguration().getTargetSystemInterface(client);
		ret.setUmask(getUmask());
		return ret;
	}

	/**
	 * get the {@link StorageAdapterFactory} to use to create 
	 * an {@link IStorageAdapter} for the the backend storage 
	 * This default implementation returns a {@link TSIStorageAdapterFactory} instance, meaning the 
	 * filesystem on the target system will be used as storage
	 */
	protected StorageAdapterFactory getStorageAdapterFactory(){
		return new TSIStorageAdapterFactory(xnjsReference);
	}

	protected FileSystemRP makeFileSystemRP(String name){
		return new FileSystemRP(this,name);
	}

	/**
	 * get the {@link MetadataManager}<br/>
	 * This default implementation will return a storage metadata manager
	 * @return {@link MetadataManager} or <code>null</code> if disabled
	 */
	protected MetadataManager getMetadataManager()throws Exception{
		if(disableMetadata)return null;
		MetadataManager mm= MetadataSupport.getManager(kernel, getStorageAdapter(), getUniqueID());
		return mm;
	}


	/**
	 * which class to use as {@link MetadataManagement} implementation
	 * @return
	 */
	protected String getMetadataManagementImplClassName(){
		return MetadataManagementImpl.class.getName();
	}

	protected String createMetadataServiceInstance(){
		Home mdHome=kernel.getHome(UAS.META);
		if(mdHome!=null){
			try{
				Map<String,Object>opts=new HashMap<String, Object>();
				opts.put(BaseMetadataManagementImpl.INIT_SMS_ID, getUniqueID());
				Calendar tt=Calendar.getInstance();
				tt.add(Calendar.YEAR, 10);
				opts.put(INIT_INITIAL_TERMINATION_TIME, tt);
				//set a "guessable" UUID for the metadata service 
				opts.put(INIT_UNIQUE_ID, getUniqueID()+"_metadata");
				opts.put(INIT_PARENT_NODE, getNode());
				return mdHome.createWSRFServiceInstance(opts);
			}catch(Exception ex){
				Log.logException("", ex, logger);
			}
		}
		return null;
	}

	/**
	 * creates the metadata service instance and adds a resource property 
	 * holding its address 
	 */
	protected void setupMetadataService()throws Exception{
		if(!disableMetadata){
			String uid=createMetadataServiceInstance();
			if(uid==null){
				disableMetadata=Boolean.TRUE;
			}
			else{
				QName q=MetadataServiceReferenceDocument.type.getDocumentElementName();
				String serviceSpec=UAS.META+"?res="+uid;
				AddressResourceProperty rp=new AddressResourceProperty(this,q,serviceSpec,false);
				properties.put(RPMetadataServiceReference, rp);
			}
		}
	}


	protected EndpointReferenceType createRemoteStorageEPR(URI uri)
	{
		EndpointReferenceType epr=EndpointReferenceType.Factory.newInstance();
		String withoutScheme=uri.getSchemeSpecificPart();
		String upToFragment = withoutScheme.split("#")[0];
		epr.addNewAddress().setStringValue(upToFragment);
		return epr;
	}

	/**
	 * Test whether the given scheme (extracted from a file address) hints to
	 * auto-negotiating the transfer protocol.
	 * @param scheme
	 * @return
	 */
	protected boolean isAutoNegotiateScheme(String scheme)
	{
		return AUTO_NEGOTIATE_SCHEMES.contains(scheme.toLowerCase());
	}

	/**
	 * Iterate over our supported protocols and select the first file transfer
	 * protocol that is supported by the peer storage, too.
	 * This assumes that our protocols are ordered according to priority/QoS
	 * @param fileURI
	 * @return
	 */
	protected ProtocolType.Enum autoNegotiateProtocol(URI fileURI) throws Exception
	{
		EndpointReferenceType storageEpr = createRemoteStorageEPR(fileURI);
		String key = storageEpr.getAddress().getStringValue();
		Element cached = protocolCache.get(key);

		ProtocolType.Enum result = cached == null ? null : (ProtocolType.Enum) cached.getValue();
		if(result == null)
		{
			StorageClient sc = new StorageClient(storageEpr, getKernel().getClientConfiguration());
			ProtocolType.Enum[] supported = sc.getSupportedProtocols();
			Set<ProtocolType.Enum> supportedSet = new HashSet<ProtocolType.Enum>();
			if(supported != null && supported.length > 0)
			{
				supportedSet.addAll(Arrays.asList(supported));
			}

			for(ProtocolType.Enum current : getStorageProtocols())
			{
				if(supportedSet.contains(current))
				{
					return current;
				}
			}
			if(result == null) return ProtocolType.BFT; // fallback to BFT
		}
		if(cached == null) cached = new Element(key, result);
		protocolCache.put(cached);
		return result;
	}

	/* TODO - is it needed? If yes - is it OK that no ETD extension is configured??
	protected IClientConfiguration getSecurityProperties()
	{
		IClientConfiguration sec=kernel.getClientConfiguration().clone();
		try{
			ETDAssertionForwarding.configureETD(getClient(), sec);
		}catch(NullPointerException npe){
			logger.warn("No security info available, running in non-secure mode?");
		}
		return sec;
	}
*/

	protected ProtocolType.Enum[] getStorageProtocols()
	{
		ResourceProperty <?>p=getResourceProperty(RPProtocol);
		if(p==null)return new ProtocolType.Enum[0];
		
		if(p instanceof ProtocolRP){
			return ((ProtocolRP)p).getProperty();
		}
		else{
			return ((UpdatingProtocolRP)p).getProperty();
		}
	}

	protected synchronized FiletransferReferenceRP getFiletransferReferenceRP()
	{
		FiletransferReferenceRP rp=(FiletransferReferenceRP) properties.get(RPInternalFiletransferReference);
		if(rp==null){
			//may happen in case of server version upgrade
			rp=new FiletransferReferenceRP(this);
			properties.put(RPInternalFiletransferReference, rp);
			setDirty();
		}
		return rp;
	}


	/**
	 * Create an in-memory cache for remembering file transfer protocol choices
	 * for remote storages. We use a shared (static) instance of this cache.
	 * 
	 * @return
	 */
	private static Cache initProtocolCache()
	{
		net.sf.ehcache.config.Configuration cmCfg = new net.sf.ehcache.config.Configuration();
		CacheConfiguration def = new CacheConfiguration();
		cmCfg.setUpdateCheck(false);
		cmCfg.addDefaultCache(def);
		//EHCache doesn't accept 2 CMs with same (even null) disk path even when nothing
		//is stored on disk ;-(
		DiskStoreConfiguration dsCfg = new DiskStoreConfiguration();
		dsCfg.setPath(System.getProperty("java.io.tmpdir") + File.separator + 
				"protocolCache" + new Random().nextLong());
		cmCfg.addDiskStore(dsCfg);
		CacheManager cacheMan = new CacheManager(cmCfg);

		Cache result = new Cache(
				"protocolCache", 
				100, MemoryStoreEvictionPolicy.LFU,
				false, null, false, 300, 300, false, 60, null);
		cacheMan.addCache(result);
		return result;
	}

	private Map<String,String> parseExtraParameters(ExtraParameters param){
		Map<String,String>result=new HashMap<String, String>();
		PropertyType[]params=param.getParameterArray();
		if(params!=null)
			for(PropertyType p: params){
				result.put(p.getName(),p.getValue());
			}
		return result;
	}

	public String getUmask() {
		UmaskResourceProperty rp = (UmaskResourceProperty) properties.get(RPUmask);
		if (rp == null)
			return Integer.toOctalString(IStorageAdapter.DEFAULT_UMASK);
		return rp.getXml()[0].getUmask();
	}

	public StorageDescription getStorageDescription() {
		return storageDescription;
	}

	/**
	 * Whether we change sharing of SMS when SMS parent resource is changed depends on
	 * case. E.g. for Uspace we inherit, for Home we don't.
	 * {@inheritDoc}
	 */
	@Override
	public boolean isRecursiveVOMembershipChangeHonored() {
		return inheritSharing;
	}
	
	/**
	 * SMS sharing must be reflected in the OS, what is implemented here.
	 * {@inheritDoc}
	 */
	public void updateVoMembership(Map<? extends VODescription, Set<OperationType>> addedVos,
			Map<? extends VODescription, Set<OperationType>> modifiedVos,
			Map<? extends VODescription, Set<OperationType>> removedVos,
			Map<? extends VODescription, Set<OperationType>> newVoMembership) throws Exception {
		
		StorageSharingMode mode = uasProperties.getEnumValue(UASProperties.SMS_GROUP_SHARING_MODE, StorageSharingMode.class);
		if (mode == StorageSharingMode.DISABLED)
			throw new Exception("Sharing of files is disabled on this service by administrators.");
		if (getStorageAdapter() instanceof LocalTS)
			throw new Exception("Sharing of files is not possibled in storage served by embedded Java TSI.");
		
		if (mode == StorageSharingMode.PREFER_ACL) {
			mode = getStorageAdapter().isACLSupported(getStorageRoot()) ? 
					StorageSharingMode.ACL : StorageSharingMode.CHMOD;
		}
			
		if (mode == StorageSharingMode.ACL)
			updateVoMembershipViaAcl(addedVos, modifiedVos, removedVos, newVoMembership);
		else
			updateVoMembershipViaChmod(addedVos, modifiedVos, removedVos, newVoMembership);
	}

	/**
	 * Works as follows:
	 * A) there is a VO removed:
	 *    1) change umask so group's mask == default group's mask.
	 *    2) do recursive chmod g-rwx
	 * B) there is a new VO added or modified:
	 *     1) change umask so group's mask is == owner's mask
	 *     2) do recursive chgrp to pgid of all files.
	 *     3) for read operation type do recursive chmod g+rX
	 *        for write operation type do recursive chmod g+rwX
	 * 
	 * @param addedVos
	 * @param modifiedVos
	 * @param removedVos
	 * @param newVoMembership
	 * @throws Exception
	 */
	protected void updateVoMembershipViaChmod(Map<? extends VODescription, Set<OperationType>> addedVos,
			Map<? extends VODescription, Set<OperationType>> modifiedVos,
			Map<? extends VODescription, Set<OperationType>> removedVos,
			Map<? extends VODescription, Set<OperationType>> newVoMembership) throws Exception {
		if (newVoMembership.size() > 1)
			throw new Exception("This storage can be shared only with in a single group.");
		
		Client client = AuthZAttributeStore.getClient();
		String pgid = client.getXlogin().getGroup();
		IStorageAdapter storage = getStorageAdapter();

		if (removedVos.size() == 1) {
			int curUmask = Integer.parseInt(storage.getUmask(), 8);
			int newMask = (IStorageAdapter.DEFAULT_UMASK & 0070) | 
					(curUmask & ~0070);
			storage.setUmask(Integer.toOctalString(newMask));
			
			de.fzj.unicore.xnjs.io.ChangePermissions chPerm = new de.fzj.unicore.xnjs.io.ChangePermissions(
					Mode.SUBTRACT, PermissionsClass.GROUP, "rwx");
			storage.chmod2(getStorageRoot(), new de.fzj.unicore.xnjs.io.ChangePermissions[] {chPerm}, true);
		}

		Map.Entry<? extends VODescription, Set<OperationType>> added = null;
		if (modifiedVos.size() == 1)
			added = modifiedVos.entrySet().iterator().next();
		else if (addedVos.size() == 1)
			added = addedVos.entrySet().iterator().next();
		
		if (added != null) {
			String permsMod = getPermsMod(added.getValue());
			int curUmask = Integer.parseInt(storage.getUmask(), 8);
			int ownerUmaskShifted = (curUmask & 0700) >> 8;
			int newMask = (curUmask & 0707) | 
					(ownerUmaskShifted);
			storage.setUmask(Integer.toOctalString(newMask));

			storage.chgrp(getStorageRoot(), pgid, true);
			
			de.fzj.unicore.xnjs.io.ChangePermissions chPerm = new de.fzj.unicore.xnjs.io.ChangePermissions(
					Mode.SET, PermissionsClass.GROUP, permsMod);
			storage.chmod2(getStorageRoot(), new de.fzj.unicore.xnjs.io.ChangePermissions[] {chPerm}, true);
		}
	}
	
	/**
	 * for VO removed
	 *     set normal and default group pgid ACL to ---
	 * for VO new or modified
	 *     set normal and default group pgid ACL to rX or rwX depending on OperationType
	 * @param addedVos
	 * @param modifiedVos
	 * @param removedVos
	 * @param newVoMembership
	 * @throws Exception
	 */
	protected void updateVoMembershipViaAcl(Map<? extends VODescription, Set<OperationType>> addedVos,
			Map<? extends VODescription, Set<OperationType>> modifiedVos,
			Map<? extends VODescription, Set<OperationType>> removedVos,
			Map<? extends VODescription, Set<OperationType>> newVoMembership) throws Exception {
		Client client = AuthZAttributeStore.getClient();
		String pgid = client.getXlogin().getGroup();

		if (removedVos.size() > 0)
			setNormalAndDefACL(pgid, "---");
		
		for (VODescription addedKey: addedVos.keySet()) {
			String permsMod = getPermsMod(addedVos.get(addedKey));
			setNormalAndDefACL(pgid, permsMod);			
		}
		
		for (VODescription modifiedKey: modifiedVos.keySet()) {
			String permsMod = getPermsMod(addedVos.get(modifiedKey));
			setNormalAndDefACL(pgid, permsMod);			
		}
	}
	
	protected void setNormalAndDefACL(String gid, String aclSpec) throws Exception {
		IStorageAdapter storage = getStorageAdapter();
		
		ChangeACL change = new ChangeACL(Type.GROUP, gid, aclSpec, false, ACLChangeMode.MODIFY);
		ChangeACL change2 = new ChangeACL(Type.GROUP, gid, aclSpec, true, ACLChangeMode.MODIFY);
		
		storage.setfacl(getStorageRoot(), false, new ChangeACL[] {change, change2}, true);
	}
	
	private String getPermsMod(Set<OperationType> ops) throws Exception {
		if (ops.contains(OperationType.modify))
			return "rwX";
		else if (ops.contains(OperationType.read))
			return "rX";
		else
			throw new Exception("Unsupported operation type for files sharing");
	}
	
	private class UmaskListenerImpl implements UmaskChangedListener {
		@Override
		public void umaskChanged(String newUmask) throws InvalidModificationException {
			try {
				getTSI().setUmask(newUmask);
			} catch (IllegalArgumentException e) {
				throw new InvalidModificationException(e.getMessage());
			}
		}

	}
}
