package de.fzj.unicore.uas.impl.rns;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;

import org.apache.commons.io.FilenameUtils;
import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlObject;
import org.ggf.rns.RNSMetadataType;
import org.ggf.rns.RNSSupportType;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.rns.RNSEntryDoesNotExistFault;
import de.fzj.unicore.uas.rns.RNSEntryExistsFault;
import de.fzj.unicore.uas.rns.WriteNotPermittedFault;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.uas.util.Pair;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import de.fzj.unicore.wsrflite.xmlbeans.WSUtilities;

public abstract class BasicRNSNode implements RNSNode {


	private static final long serialVersionUID = 1L;


	protected static final Logger logger=LogUtil.getLogger(LogUtil.SERVICES,RNSNode.class);

	
	private String name;
	EndpointReferenceType epr;

	public BasicRNSNode(String name)
	{
		this(name, null);
	}

	public BasicRNSNode(String name, EndpointReferenceType epr) {
		this(name, epr, new ArrayList<RNSNode>());
	}

	public BasicRNSNode(String name, EndpointReferenceType epr, RNSNode... children) {
		this(name, epr, Arrays.asList(children));
	}

	public BasicRNSNode(String name, EndpointReferenceType epr, Collection<RNSNode> kids) {
		super();
		this.name = name;
		this.epr = epr;
		
		for(RNSNode child : kids)
		{
			try {
				addChild(child);
			} catch (RNSEntryExistsFault e) {
				// this REALLY shouldn't happen
				logger.error("Tried to initialize an RNS node with multiple children with the same name.. the conflicting name is "+child.getName(),e);
			}
			catch (WriteNotPermittedFault e) {
				// this REALLY shouldn't happen
				logger.error("Thrown WriteNotPermittedFault while initializing BasicRNSNode...",e);
			}
			catch (BaseFault e) {
				logger.error("Thrown BaseFault while initializing BasicRNSNode...",e);
			}
		}
	}



	public EndpointReferenceType getEPR() {
		return epr;
	}

	public void setEPR(EndpointReferenceType epr) {
		this.epr = epr;
	}

	
	@Override
	public RNSOperationResult add(RNSNode... newChildren) {
		RNSOperationResult result = new RNSOperationResult();
		for(RNSNode node : newChildren)
		{
			if(node.getName() == null)
			{
				result.addFault(BaseFault.createFault("Invalid request: asked to add an RNS Entry that's missing a name!"));
			}
			else if(node.getEPR() == null)
			{
				// the BES spec is a bit weird here... this actually means:
				// create a sub directory!
				try {
					node = createDir(node.getName(), node.getSupportsRNS());
					result.addSuccess();
					result.addEPR(node.getEPR());
					result.addMetadata(node.getMetadataType());
					result.addName(node.getName());
				} catch (BaseFault e) {
					result.addFault(e);
				}
				
			}
			else
			{
				try {
					addChild(node);
					result.addSuccess();
					result.addEPR(node.getEPR());
					result.addMetadata(node.getMetadataType());
					result.addName(node.getName());
				} catch (BaseFault e) {
					result.addFault(e);
				}
				
			}
		}
		return result;
	}

	@Override
	public RNSOperationResult lookup(String... names) throws BaseFault {
		RNSOperationResult result = new RNSOperationResult();
		if(names == null || names.length == 0)
		{

			for(RNSNode child : getChildren())
			{
				if(child != null)
				{
					result.addName(child.getName());
					result.addEPR(child.getEPR());
					result.addMetadata(child.getMetadataType());
					result.addSuccess();
				}
			}
		}
		else
		{
			for(String name : names)
			{
				RNSNode child;
				try {
					child = getChild(name);
					result.addName(name);
					result.addEPR(child.getEPR());
					result.addMetadata(child.getMetadataType());
					result.addSuccess();
				} catch (RNSEntryDoesNotExistFault e) {
					result.addFault(e);
				}
			}
		}
		return result;
	}

	@Override
	public RNSOperationResult rename(Pair<String, String>... namePairs) {
		RNSOperationResult result = new RNSOperationResult();
		for(Pair<String, String> pair : namePairs)
		{
			String oldName = pair.getM1();
			String newName = pair.getM2();
			if(oldName == null)
			{
				result.addFault(BaseFault.createFault("Invalid request: asked to rename an RNS Entry without specifying its old name"));
			}
			else if(newName == null)
			{
				result.addFault(BaseFault.createFault("Invalid request: asked to rename an RNS Entry without specifying its new name"));
			}
			else
			{
				try {
					RNSNode node = getChild(oldName);
					node.setName(newName);
					// first add, then remove => make sure noone takes our old 
					// name in the meantime
					addChild(node);
					try {
						remove(oldName);
					} catch (Exception e) {
						// removed/renamed by someone else.. ignore 
					}
					result.addSuccess();
					result.addEPR(node.getEPR());
					result.addMetadata(node.getMetadataType());
					result.addName(node.getName());
				} catch (BaseFault e) {
					result.addFault(e);
				}
				
			}
		}
		return result;
	}
	
	@Override
	public RNSOperationResult setChildrenMetadata(
			Pair<String, RNSMetadataType>... metadata) {
		RNSOperationResult result = new RNSOperationResult();
		for(Pair<String, RNSMetadataType> pair : metadata)
		{
			String name = pair.getM1();
			RNSMetadataType data = pair.getM2();
			if(name == null)
			{
				result.addFault(BaseFault.createFault("Invalid request: asked to set an RNS Entry's metadata without specifying its name"));
			}
			else
			{
				try {
					RNSNode node = getChild(name);
					node.setSupportsRNS(RNSSupportType.TRUE.equals(data.getSupportsRns().getValue()));
					XmlObject[] xml = WSUtilities.extractAllChildren(data);
					node.setMetadata(xml);
					result.addSuccess();
					result.addEPR(node.getEPR());
					result.addMetadata(node.getMetadataType());
					result.addName(node.getName());
				} catch (BaseFault e) {
					result.addFault(e);
				}
				
			}
		}
		return result;
	}

	@Override
	public RNSOperationResult remove(String... names) {
		RNSOperationResult result = new RNSOperationResult();
		for(String name : names)
		{
			
			if(name == null)
			{
				result.addFault(BaseFault.createFault("Invalid request: asked to remove an RNS Entry without specifying its name"));
			}
			else
			{
				try {
					RNSNode node = removeChild(name);
					result.addSuccess();
					result.addEPR(node.getEPR());
					result.addMetadata(node.getMetadataType());
					result.addName(node.getName());
				} catch (BaseFault e) {
					result.addFault(e);
				}
				
			}
		}
		return result;
	}

	public RNSNode getChildForPath(String path) throws BaseFault
	{
		path = normalizePath(path);
		if(path.length() == 0) return this;
		int index = path.indexOf("/");
		if(index < 0) return getChild(path);
		String childName = path.substring(0,index);
		path = path.substring(index+1);
		RNSNode child = getChild(childName);
		return child.getChildForPath(path);
	}

	
	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	@Override
	public int compareTo(RNSNode o) {
		return getName().compareTo(o.getName());
	}


	protected static String normalizePath(String path)
	{
		if(path == null) return "";
		path = path.trim();
		String result = FilenameUtils.normalizeNoEndSeparator(path, true);
		if(result == null) return "";
		if(result.startsWith("/")) result = result.substring(1);
		return result;
	}


}
