package de.fzj.unicore.uas.impl.reservation;

import java.util.Calendar;
import java.util.Map;

import javax.xml.namespace.QName;

import org.apache.log4j.Logger;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.ResourcesDocument;
import org.unigrids.x2006.x04.services.jms.TargetSystemReferenceDocument;
import org.unigrids.x2006.x04.services.reservation.ReservationPropertiesDocument;
import org.unigrids.x2006.x04.services.reservation.ReservationReferenceDocument;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.ReservationManagement;
import de.fzj.unicore.uas.impl.UASWSResourceImpl;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.uas.xnjs.XNJSFacade;
import de.fzj.unicore.wsrflite.messaging.ResourceDeletedMessage;
import de.fzj.unicore.wsrflite.persistence.Persist;
import de.fzj.unicore.wsrflite.utils.WSServerUtilities;
import de.fzj.unicore.wsrflite.xmlbeans.rp.ImmutableResourceProperty;
import de.fzj.unicore.xnjs.ems.ExecutionException;
import de.fzj.unicore.xnjs.tsi.IReservation;
import de.fzj.unicore.xnjs.tsi.ReservationStatus;
import eu.unicore.security.Client;
import eu.unicore.security.Xlogin;

/**
 * WS resource representing a resource reservation
 * 
 * @author schuller
 */
public class ReservationManagementImpl extends UASWSResourceImpl implements ReservationManagement {

	private static final Logger logger=LogUtil.getLogger(LogUtil.SERVICES,ReservationManagementImpl.class);
	
	public static String INITPARAM_RESOURCES="resources";
	public static String INITPARAM_STARTTIME="starttime";
	public static String INITPARAM_TSS_REFERENCE="tss_reference";

	@Persist
	private String reservationReference;
	
	@Persist
	private String tssID;

	//original Xlogin that was used to create the reservation
	@Persist
	private Xlogin xlogin;

	@Override
	public QName getResourcePropertyDocumentQName() {
		return ReservationPropertiesDocument.type.getDocumentElementName();
	}
	
	@Override
	public void initialise(String serviceName, Map<String, Object> initParams) throws Exception {
		Calendar startTime=(Calendar)initParams.get(INITPARAM_STARTTIME);
		ResourcesDocument resources=(ResourcesDocument)initParams.get(INITPARAM_RESOURCES);
		//must do this here, since we want to use the default lifetime
		setServiceName(serviceName);
		//set lifetime to startTime+defaultLifetime
		Calendar lifetime=Calendar.getInstance();
		lifetime.setTime(startTime.getTime());
		lifetime.add(Calendar.SECOND, getDefaultLifetime());
		initParams.put(INIT_INITIAL_TERMINATION_TIME, lifetime);
		super.initialise(serviceName, initParams);

		IReservation reservation=XNJSFacade.get(xnjsReference,kernel).getReservation();
		if(reservation==null)throw new Exception("Reservation not supported.");
		
		reservationReference=reservation.makeReservation(resources, startTime, getClient());
		ReservationReferenceDocument res=ReservationReferenceDocument.Factory.newInstance();
		res.setReservationReference(reservationReference);
		properties.put(RPReservationReference, new ImmutableResourceProperty(res));

		properties.put(RPStartTime, new StartTimeResourceProperty(startTime,this));
		properties.put(RPReservationStatus, new ReservationStatusResourceProperty(this));
		properties.put(RPReservationStatusDescription, new ReservationStatusDescriptionResourceProperty(this));
		properties.put(RPResources, new ImmutableResourceProperty(resources));

		//original Xlogin
		xlogin=getClient()!=null?getClient().getXlogin():null;

		TargetSystemReferenceDocument tssRef=TargetSystemReferenceDocument.Factory.newInstance();
		EndpointReferenceType tssEPR=(EndpointReferenceType)initParams.get(INITPARAM_TSS_REFERENCE);
		tssRef.setTargetSystemReference(tssEPR);
		tssID=WSServerUtilities.extractResourceID(tssEPR);
		properties.put(RPTargetSystemReference, new ImmutableResourceProperty(tssRef));

	}

	/**
	 * on destroy(), cancel the booking on the backend.
	 * Also, send a message to parent TSS
	 */
	@Override
	public void destroy() {
		try{
			ResourceDeletedMessage m=new ResourceDeletedMessage("deleted:"+getUniqueID());
			m.setServiceName(getServiceName());
			m.setDeletedResource(getUniqueID());
			kernel.getMessaging().getChannel(tssID).publish(m);
		}
		catch(Exception e){
			LogUtil.logException("Could not send internal message.",e,logger);
		}
		try{
			Client client=getClient();
			//TODO LOCAL call flag should be always set EXCEPT when call comes from the outside
			if(Client.Type.LOCAL==client.getType() || Client.Type.ANONYMOUS==client.getType()){
				client.setXlogin(xlogin);
				logger.info("Cancelling reservation "+reservationReference+" using xlogin "+xlogin);
			}
			XNJSFacade.get(xnjsReference,kernel).getReservation().cancelReservation(reservationReference, getClient());
		}
		catch(Exception e){
			LogUtil.logException("Could not cancel resource reservation.",e,logger);
		}
		super.destroy();
	}
	
	public void setTSSID(String tssID){
		this.tssID=tssID;
	}

	long lastUpdate=0;
	final long updateInterval=3000;
	transient ReservationStatus reservationStatus;

	public synchronized ReservationStatus getReservationStatus()throws ExecutionException{
		if(reservationStatus==null || System.currentTimeMillis()>lastUpdate+updateInterval){
			reservationStatus=XNJSFacade.get(xnjsReference,kernel).getReservation().queryReservation(reservationReference, getClient());
			lastUpdate=System.currentTimeMillis();
		}
		return reservationStatus;
	}
}
