/*********************************************************************************
 * Copyright (c) 2006 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
 

package de.fzj.unicore.uas.impl.job;

import java.util.Calendar;
import java.util.HashMap;
import java.util.Map;

import javax.xml.namespace.QName;

import org.apache.log4j.Logger;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.JobDefinitionDocument;
import org.unigrids.services.atomic.types.StorageReferenceType;
import org.unigrids.services.atomic.types.StorageTypeEnumeration;
import org.unigrids.x2006.x04.services.jms.AbortDocument;
import org.unigrids.x2006.x04.services.jms.AbortResponseDocument;
import org.unigrids.x2006.x04.services.jms.HoldDocument;
import org.unigrids.x2006.x04.services.jms.HoldResponseDocument;
import org.unigrids.x2006.x04.services.jms.JobPropertiesDocument;
import org.unigrids.x2006.x04.services.jms.OriginalJSDLDocument;
import org.unigrids.x2006.x04.services.jms.ResumeDocument;
import org.unigrids.x2006.x04.services.jms.ResumeResponseDocument;
import org.unigrids.x2006.x04.services.jms.StartDocument;
import org.unigrids.x2006.x04.services.jms.StartResponseDocument;
import org.unigrids.x2006.x04.services.jms.SubmissionTimeDocument;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.JobManagement;
import de.fzj.unicore.uas.StorageManagement;
import de.fzj.unicore.uas.TargetSystem;
import de.fzj.unicore.uas.UAS;
import de.fzj.unicore.uas.client.BaseUASClient;
import de.fzj.unicore.uas.faults.JobNotStartedFault;
import de.fzj.unicore.uas.impl.PersistingPreferencesResource;
import de.fzj.unicore.uas.impl.bp.BPSupportImpl;
import de.fzj.unicore.uas.impl.sms.StorageDescription;
import de.fzj.unicore.uas.impl.sms.SMSBaseImpl;
import de.fzj.unicore.uas.impl.sms.StorageManagementHomeImpl.StorageTypes;
import de.fzj.unicore.uas.impl.tss.rp.StorageReferenceResourceProperty;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.uas.xnjs.XNJSFacade;
import de.fzj.unicore.wsrflite.Home;
import de.fzj.unicore.wsrflite.messaging.ResourceDeletedMessage;
import de.fzj.unicore.wsrflite.persistence.Persist;
import de.fzj.unicore.wsrflite.security.util.AuthZAttributeStore;
import de.fzj.unicore.wsrflite.utils.WSServerUtilities;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import de.fzj.unicore.wsrflite.xmlbeans.rp.ImmutableResourceProperty;
import de.fzj.unicore.xnjs.ems.Action;
import eu.unicore.security.Client;

/**
 * implements a Job resource, and allows job management through WSRF
 * 
 * @author schuller
 */
public class JobManagementImpl extends PersistingPreferencesResource implements JobManagement {
	
	private static final Logger logger=LogUtil.getLogger(LogUtil.SERVICES,JobManagementImpl.class);

	public static final String INITPARAM_ACTION="uas.jobmanagement.action";
	public static final String INITPARAM_ACTION_AUTOSTART="uas.jobmanagement.action.autostart";
	public static final String INITPARAM_ACTION_NO_XNJS_SUBMIT="uas.jobmanagement.action.no_xnjs_submit";
	public static final String INITPARAM_TSS_UNIQUEID="uas.jobmanagement.TSS";
	public static final String INITPARAM_STORAGE_REFERENCE="uas.jobmanagement.SMS";
	
	public static final String JMS_NS = "http://unigrids.org/2006/04/services/jms";
	public static final QName STDOUT = new QName(JMS_NS,"StdOut");
	public static final QName STDERR = new QName(JMS_NS,"StdErr");

	@Persist
	private String tssId;
	
	@Override
	protected void addWSResourceInterfaces(BPSupportImpl baseProfile) {
		super.addWSResourceInterfaces(baseProfile);
		baseProfile.addWSResourceInterface(JMS_PORT);
	}
	
	@Override
	public QName getPortType() {
		return JMS_PORT;
	}
	
	@Override
	public QName getResourcePropertyDocumentQName() {
		return JobPropertiesDocument.type.getDocumentElementName();
	}
	
	public StartResponseDocument Start(StartDocument in) throws JobNotStartedFault {
		StartResponseDocument res=StartResponseDocument.Factory.newInstance();
		try{
			
			XNJSFacade.get(xnjsReference,kernel).getManager().run(getUniqueID(), 
				AuthZAttributeStore.getClient());
			res.addNewStartResponse();
			logger.info("Started "+getUniqueID());
			return res;
		}catch(Exception e){
			LogUtil.logException("Could not start job "+getUniqueID(),e,logger);
			throw JobNotStartedFault.createFault(e.getMessage());
		}
	}

	public AbortResponseDocument Abort(AbortDocument in) throws BaseFault {
		AbortResponseDocument res=AbortResponseDocument.Factory.newInstance();
		try{
			XNJSFacade.get(xnjsReference,kernel).getManager().abort(getUniqueID(), 
				AuthZAttributeStore.getClient());
			res.addNewAbortResponse();
			return res;
		}catch(Exception e){
			LogUtil.logException("Could not abort the job.",e,logger);
			throw BaseFault.createFault("Could not abort the job.",e);
		}
	}

	public HoldResponseDocument Hold(HoldDocument in) throws BaseFault {
		HoldResponseDocument res=HoldResponseDocument.Factory.newInstance();
		try{
			XNJSFacade.get(xnjsReference,kernel).getManager().pause(getUniqueID(), 
				AuthZAttributeStore.getClient());
			res.addNewHoldResponse();
			return res;
		}catch(Exception e){
			LogUtil.logException("Could not hold the job.",e,logger);
			throw BaseFault.createFault("Could not hold the job.",e);
		}
	}

	public ResumeResponseDocument Resume(ResumeDocument in) throws BaseFault {
		ResumeResponseDocument res=ResumeResponseDocument.Factory.newInstance();
		try{
			XNJSFacade.get(xnjsReference,kernel).getManager().resume(getUniqueID(), 
				AuthZAttributeStore.getClient());
			res.addNewResumeResponse();
			return res;
		}catch(Exception e){
			LogUtil.logException("Could not resume the job.",e,logger);
			throw BaseFault.createFault("Could not resume the job",e);
		}
	}

	/**
	 * initialise the new WS Resource<br>
	 * this will submit the job to the XNJS and setup the Job's resource properties
	 */
	@Override
	public void initialise(String serviceName, Map<String, Object> initParams)throws Exception{
		//override uniqueID with action UUID
		Action action=(Action)initParams.get(INITPARAM_ACTION);
		initParams.put(INIT_UNIQUE_ID, action.getUUID());
		super.initialise(serviceName, initParams);
		
		tssId=(String)initParams.get(INITPARAM_TSS_UNIQUEID);

		//autostart?
		if(Boolean.TRUE.equals(initParams.get(INITPARAM_ACTION_AUTOSTART))){
			action.getProcessingContext().put(Action.AUTO_SUBMIT,"true");
		}
		
		//submit to XNJS
		Boolean noXNJSSubmit=(Boolean)initParams.get(INITPARAM_ACTION_NO_XNJS_SUBMIT);
		if(!Boolean.TRUE.equals(noXNJSSubmit)){
			Client client = AuthZAttributeStore.getClient();
			XNJSFacade.get(xnjsReference,kernel).getManager().add(action, client);
			logger.info("Submitted job with id "+action.getUUID()+ " for client "+client);
		}
		
		//submission time
		SubmissionTimeDocument std=SubmissionTimeDocument.Factory.newInstance();
		std.setSubmissionTime(Calendar.getInstance());
		properties.put(RPSubmissionTime, new ImmutableResourceProperty(std));
		
		StatusInfoResourceProperty info=new StatusInfoResourceProperty(
				getUniqueID(),this,xnjsReference);
		properties.put(RPStatusInfo,info);
		
		properties.put(RPTargetSystemReference,createTSSReferenceProperty());
		
		OriginalJSDLDocument ojd=OriginalJSDLDocument.Factory.newInstance();
		ojd.setOriginalJSDL(((JobDefinitionDocument)action.getAjd()).getJobDefinition());
		ImmutableResourceProperty oJsdl=new  ImmutableResourceProperty(ojd);
		properties.put(RPOriginalJSDL,oJsdl);
		
		LogResourceProperty log=new LogResourceProperty(this, getUniqueID(),xnjsReference);
		properties.put(RPLog, log);
		
		ExecutionJSDLResourceProperty eJSDL=new ExecutionJSDLResourceProperty(this,getUniqueID(),xnjsReference);
		properties.put(RPExecutionJSDL, eJSDL);
		
		//Uspace (working directory)
		EndpointReferenceType uspace=createUspace(action);
		WorkingDirResourceProperty workDir=new WorkingDirResourceProperty(uspace);
		properties.put(RPWorkingDir,workDir);
		
		//any additional storage references
		EndpointReferenceType[] smsEprs=(EndpointReferenceType[])initParams.get(INITPARAM_STORAGE_REFERENCE);
		if(smsEprs!=null){
			StorageReferenceResourceProperty sp=new StorageReferenceResourceProperty(smsEprs);
			properties.put(RPStorageReference,sp);
		}
		
		//stdout / error
		properties.put(STDOUT, new StdOutProperty(this,action.getUUID(),xnjsReference));
		properties.put(STDERR, new StdErrProperty(this,action.getUUID(),xnjsReference));
		
		
		setDirty();
		
		//publish this in registry?
		if(Boolean.TRUE.equals(initParams.get(INITPARAM_PUBLISH_TO_REGISTRY)))publish();
	}

	
	/**
	 * resource-specific destruction: send message about our demise
	 */
	@Override
	public void destroy() {
		try{
			ResourceDeletedMessage m=new ResourceDeletedMessage("deleted:"+getUniqueID());
			m.setServiceName(getServiceName());
			m.setDeletedResource(getUniqueID());
			getKernel().getMessaging().getChannel(tssId).publish(m);
		}
		catch(Exception e){
			LogUtil.logException("Could not send internal message.",e,logger);
		}
		//clean up on backend
		try{
			XNJSFacade.get(xnjsReference,kernel).destroyAction(getUniqueID(), getClient());
		}
		catch(Exception e){
			LogUtil.logException("Could not destroy job on XNJS.",e,logger);
		}
		try{
			//destroy the uspace resource
			EndpointReferenceType uspace=((WorkingDirResourceProperty)properties.get(RPWorkingDir)).getXml()[0].getWorkingDirectoryReference();
			BaseUASClient c=new BaseUASClient(uspace, kernel.getClientConfiguration());
			c.destroy();
		}catch(Exception e){
			LogUtil.logException("Could not destroy storages.",e,logger);
		}
		super.destroy();
	}
	
	/**
	 * creates a storage management service for the job
	 * @param initParams
	 * @return
	 */
	protected EndpointReferenceType createStorageManagementService( Map<String, Object> initParams){
		try {
			Home smsHome=kernel.getHome(UAS.SMS);
			String id=smsHome.createWSRFServiceInstance(initParams);
			return WSServerUtilities.makeEPR(UAS.SMS,id,StorageManagement.SMS_PORT,kernel);
		} catch (Exception e) {
			LogUtil.logException("Could not create storage for job",e,logger);
		}
		return null;
	}
	
	/**
	 * creates the Uspace SMS instance
	 * @param a - XNJS Action
	 * @return the EPR of the newly created Uspace
	 */
	protected EndpointReferenceType createUspace(Action a){
		Map<String,Object>initMap=new HashMap<String,Object>();
		
		StorageDescription description = new StorageDescription("UspaceOf-"+getUniqueID(), "Uspace",
				a.getExecutionContext().getWorkingDirectory(), 
				StorageTypes.VARIABLE, null, null, false, false, false, a.getUmask(), 
				"Job's woorkspace", null);
		initMap.put(SMSBaseImpl.INIT_STORAGE_DESCRIPTION, description);
		initMap.put(INIT_PARENT_NODE, nodeHelper);
		initMap.put(SMSBaseImpl.INIT_INHERIT_SHARING, Boolean.TRUE);
		
		//set "infinite" tt, because the SMS will be destroyed together with the JMS
		Calendar tt=Calendar.getInstance();
		tt.add(Calendar.YEAR,10);
		initMap.put(INIT_INITIAL_TERMINATION_TIME,tt);
		initMap.put(INITPARAM_XNJS_REFERENCE, xnjsReference);
		return createStorageManagementService(initMap);
	}

	/**
	 * create storage reference property
	 * @return
	 */
	protected StorageReferenceResourceProperty createStorageProperty(EndpointReferenceType epr){
		StorageReferenceType type=StorageReferenceType.Factory.newInstance();
		type.setStorageEndpointReference(epr);
		type.setType(StorageTypeEnumeration.WORK);
		StorageReferenceResourceProperty srp=new StorageReferenceResourceProperty(new StorageReferenceType[]{type});
		return srp;
	}

	protected TSSReferenceResourceProperty createTSSReferenceProperty(){
		EndpointReferenceType epr=WSServerUtilities.newEPR(kernel.getContainerSecurityConfiguration());
		epr.addNewAddress().setStringValue(WSServerUtilities.makeAddress(UAS.TSS, tssId,
				kernel.getContainerProperties()));
		WSServerUtilities.addPortType(epr, TargetSystem.PORTTYPE);
		TSSReferenceResourceProperty tref=new TSSReferenceResourceProperty(epr);
		return tref;
	}
	
	/**
	 * update the "parent TSS" ID including the associated resource property
	 * @param id
	 */
	public void setTSSID(String id){
		this.tssId=id;
		properties.put(RPTargetSystemReference,createTSSReferenceProperty());
	}
}
