package de.fzj.unicore.uas.impl;

import java.util.List;

import org.apache.log4j.Logger;
import org.unigrids.services.atomic.types.UmaskDocument;

import de.fzj.unicore.uas.SMSProperties;
import de.fzj.unicore.uas.StorageManagement;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.wsrflite.exceptions.InvalidModificationException;
import de.fzj.unicore.wsrflite.xmlbeans.ModifiableResourceProperty;
import de.fzj.unicore.wsrflite.xmlbeans.impl.WSResourceImpl;

/**
 * Provides access to information about actual umask. This property is modifiable.
 * The property is used by SMS and TSS. 
 *
 * @author golbi
 */
public class UmaskResourceProperty extends ModifiableResourceProperty<UmaskDocument, String> {
	private static final Logger logger = LogUtil.getLogger(LogUtil.SERVICES, UmaskResourceProperty.class);

	private static final long serialVersionUID = 1L;

	private UmaskDocument[] initialUmaskDoc;
	
	private transient UmaskChangedListener listener = null;

	/**
	 * Create a new umask property
	 * @param parent - the parent resource
	 * @param umask - the initial umask
	 */
	public UmaskResourceProperty(WSResourceImpl parent, String umask) {
		super(StorageManagement.RPUmask, parent, true, 1, 1);
		this.initialUmaskDoc = new UmaskDocument[1];
		initialUmaskDoc[0] = UmaskDocument.Factory.newInstance();
		initialUmaskDoc[0].setUmask(umask);
		try
		{
			insert(initialUmaskDoc[0]);
		} catch (InvalidModificationException e)
		{
			throw new RuntimeException("Ups! BUG, shouldn't happen", e);
		}
	}

	/**
	 * Sets an listener to be notified when a RP is about to change. Warning: this 
	 * listener is not persisted, therefore must be re-registered in postActivte().
	 * @param listener
	 */
	public void setListener(UmaskChangedListener listener) {
		this.listener = listener;
	}
	
	@Override
	public void update(List<UmaskDocument> o)throws InvalidModificationException{
		if (o.size() != 1)
			throw new InvalidModificationException();
		String newUmask = o.get(0).getUmask();
		if (!SMSProperties.umaskPattern.matcher(newUmask).matches())
			throw new InvalidModificationException("Specified umask must be an octal number from 0 to 777.");
		if (listener != null)
			listener.umaskChanged(newUmask);
		super.update(o);
		logger.debug("Changed umask of " + parentWSResource.getUniqueID() + " to " + newUmask);
	}
	
	public void delete()throws InvalidModificationException{
		super.delete();
	}
	
	public interface UmaskChangedListener {
		public void umaskChanged(String newUmask) throws InvalidModificationException;
	}
}
