/*********************************************************************************
 * Copyright (c) 2006 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
 
package de.fzj.unicore.uas.impl;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.xml.namespace.QName;

import org.ggf.baseprofile.WSResourceInterfacesDocument;
import org.unigrids.services.atomic.types.VersionDocument;

import de.fzj.unicore.uas.UASProperties;
import de.fzj.unicore.uas.VersionResourceProperty;
import de.fzj.unicore.uas.impl.bp.BPSupportImpl;
import de.fzj.unicore.uas.impl.bp.BPWSResource;
import de.fzj.unicore.uas.xnjs.XNJSFacade;
import de.fzj.unicore.wsrflite.persistence.Persist;
import de.fzj.unicore.wsrflite.xfire.WSRFConstants;
import de.fzj.unicore.wsrflite.xmlbeans.impl.WSResourceImpl;
import de.fzj.unicore.wsrflite.xmlbeans.rp.ImmutableResourceProperty;

/**
 * a UNICORE WS Resource<br/>
 * It adds the following features to the basic {@link WSResourceImpl}:
 * <ul>
 *  <li>Support for the OGSA basic profile</li>
 *  <li>Setsup some data commonly needed by UAS services as uasProperties </li>
 * </ul>
 * @author schuller
 */
public abstract class UASWSResourceImpl extends WSResourceImpl implements BPWSResource{
	
	protected BPSupportImpl baseProfile;
	
	protected UASProperties uasProperties;
	
	public UASWSResourceImpl(){
		baseProfile=new BPSupportImpl();
		addWSResourceInterfaces(baseProfile);
	}
	
	protected void addWSResourceInterfaces(BPSupportImpl baseProfile){
		baseProfile.addWSResourceInterface(WSRFConstants.WSRL_DESTROY_PORT);
		baseProfile.addWSResourceInterface(WSRFConstants.WSRL_SET_TERMTIME_PORT);
		baseProfile.addWSResourceInterface(WSRFConstants.WSRP_GET_RP_DOCUMENT_PORT);
		baseProfile.addWSResourceInterface(WSRFConstants.WSRP_GET_RP_PORT);
		baseProfile.addWSResourceInterface(WSRFConstants.WSRP_QUERY_RP_PORT);
	}

	/**
	 * each set of services (TSS, SMS, JMS etc instances) may use its own XNJS instance.
	 * This variable contains an ID to this XNJS instance. If it is null, the
	 * default XNJS is used 
	 */
	@Persist
	protected String xnjsReference;
	
	/**
	 * used for passing the ID of the XNJS instance to be used
	 * @see XNJSFacade#get(String)
	 */
	public static final String INITPARAM_XNJS_REFERENCE=UASWSResourceImpl.class.getName()+".xnjsReference";
	
	
	/**
	 * This method is <code>final</code> to avoid programmer errors in
	 * subclasses. To add custom behaviour, use the hook method
	 * {@link #customPostActivate()}
	 * @see WSRFInstance#postActivate()
	 */
	@Override
	public final void postActivate() {
		super.postActivate();
		try{
			properties.put(BPSupportImpl.RPResourcePropertyNames, BPSupportImpl.getRPNamesProperty(properties.keySet()));
		}catch(Exception e){}
		uasProperties = kernel.getAttribute(UASProperties.class);
		customPostActivate();
	}

	/**
	 * add special post-activation behaviour by overriding this method 
	 */
	protected void customPostActivate(){}

	/**
	 * sets XNJS reference, setups WSRF base profile RPs and the server's version RP
	 */
	@Override
	public void initialise(String serviceName, Map<String, Object> initParams)throws Exception{
		super.initialise(serviceName, initParams);
		uasProperties = kernel.getAttribute(UASProperties.class);
		
		xnjsReference = (String)initParams.get(INITPARAM_XNJS_REFERENCE);
		
		properties.put(BPSupportImpl.RPWsResourceInterfaces,
				new ImmutableResourceProperty(getWSResourceInterfaces()));
		if(getPortType()!=null){
			properties.put(BPSupportImpl.RPFinalWSResourceInterface,
					baseProfile.getFinalResourceInterfaceRP(getPortType()));
			properties.put(BPSupportImpl.RPResourceEndpointReference,BPSupportImpl.getResourceEndpointReferenceRP(getEPR()));
		}
		
		//add version RP
		properties.put(VersionDocument.type.getDocumentElementName(),new VersionResourceProperty(this));
		setDirty();
	}
	
	
	@Override
	public QName getResourcePropertyDocumentQName() {
		return null;
	}

	/**
	 * returns an XML document listing the interfaces (PortTypes) realised
	 * by this class (see OGSA BaseProfile 1.0)
	 *  
	 * @return WSResourceInterfacesDocument
	 */
	public WSResourceInterfacesDocument getWSResourceInterfaces() {
		return baseProfile.getWSResourceInterfaces();
	}

	/**
	 * returns the list of RPs to be published in the registry	
	 * @return
	 */
	protected Set<QName> publishedRPs(){
		Set<QName> s=new HashSet<QName>();
		s.add(BPSupportImpl.RPWsResourceInterfaces);
		return s;
	}

	public String getXNJSReference(){
		return xnjsReference;
	}

}