/*
 * Copyright (c) 2012 ICM Uniwersytet Warszawski All rights reserved.
 * See LICENCE file for licensing information.
 */

package de.fzj.unicore.uas.impl;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.Logger;

import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.wsrflite.persistence.Persist;
import de.fzj.unicore.wsrflite.security.UserAttributeCallback;
import de.fzj.unicore.wsrflite.security.util.AuthZAttributeStore;
import eu.unicore.security.SecurityTokens;

/**
 * Adds persistence of the initial user's preferences to the resource.
 * Such preferences are used as default preferences assuming that the user has not
 * specified other.
 *  
 * @author K. Benedyczak
 */
public abstract class PersistingPreferencesResource extends UASWSResourceImpl {
	
	private static final Logger logger=LogUtil.getLogger(LogUtil.SERVICES,PersistingPreferencesResource.class);

	@Persist
	private Map<String, String[]> storedPreferences;
	
	/**
	 * Saves the initial user's preferences
	 */
	@Override
	public void initialise(String serviceName, Map<String, Object> initParams) throws Exception {
		super.initialise(serviceName, initParams);
		SecurityTokens secTokens = AuthZAttributeStore.getTokens();
		if (secTokens == null)
			return;
		
		@SuppressWarnings("unchecked")
		Map<String, String[]> actualPreferences = (Map<String, String[]>) secTokens.getContext().get(
			UserAttributeCallback.USER_PREFERENCES_KEY);
		if (actualPreferences != null) {
			storedPreferences = new HashMap<String, String[]>();
			storedPreferences.putAll(actualPreferences);
			if (logger.isDebugEnabled())
				logger.debug("Persisted user preferences, used to create the WS-Resource "
						+ getServiceName() + " [" + getUniqueID() + "]: " +
						storedPreferences.keySet());
		}
	}
	
	/**
	 * Applies the saved user's preferences (before the whole 
	 * USE authorization pipeline) whenever the user has not selected a preference 
	 * explicitly.
	 */
	@Override
	public void updateSecurityTokensBeforeAIP(SecurityTokens secTokens) {
		if (storedPreferences == null || storedPreferences.size() == 0)
			return;
		
		@SuppressWarnings("unchecked")
		Map<String, String[]> actualPreferences = (Map<String, String[]>) secTokens.getContext().get(
			UserAttributeCallback.USER_PREFERENCES_KEY);
		if (actualPreferences == null) {
			actualPreferences = new HashMap<String, String[]>();
			actualPreferences.putAll(storedPreferences);
			secTokens.getContext().put(UserAttributeCallback.USER_PREFERENCES_KEY, actualPreferences);
			if (logger.isDebugEnabled())
				logger.debug("Using saved user's preference as a defaults for " 
						+ getServiceName() + " [" + getUniqueID() + "]: " 
						+ storedPreferences.keySet());
			return;
		}
			
		for (Map.Entry<String, String[]> entry: storedPreferences.entrySet()) {
			if (!actualPreferences.containsKey(entry.getKey())) {
				actualPreferences.put(entry.getKey(), entry.getValue());
				if (logger.isDebugEnabled())
					logger.debug("Using saved user's preference as a default for "
						+ getServiceName() + " [" + getUniqueID() + "]: "
						+ entry.getKey() + ": " + Arrays.toString(entry.getValue()));
			}
		}
	}
}