package de.fzj.unicore.uas.fts.http;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URL;

import org.eclipse.jetty.util.resource.Resource;

import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.xnjs.ems.ExecutionException;
import de.fzj.unicore.xnjs.io.IFileTransfer.OverwritePolicy;
import de.fzj.unicore.xnjs.io.IStorageAdapter;
import eu.unicore.security.Client;

/**
 * Implementation of a Jetty {@link Resource} that keeps a reference to
 * the {@link Client} owning the filetransfer, and the path to the actual
 * file<br/>.
 * 
 * The TSI input/output streams are used, so this works with classic UNICORE TSIs
 * as well.  
 * 
 * @author schuller
 * @since 1.0.1
 */
public class UResource extends Resource {
	
	protected final String path;
	protected final String id;
	protected final IStorageAdapter storage;
	protected final Kernel kernel;
	protected long transferred=0;
	protected OverwritePolicy overwrite=OverwritePolicy.OVERWRITE;
	
	/**
	 * creates a Resource object for serving a file
	 * @param id - the unique ID of the resource, can be <code>null</code> if the resource is only temporary. If non-null,
	 * the transferrred bytes will be reported via {@link FileServlet#setTransferredBytes(String, Long)}
	 * @param path - the path of the file relative to storage root
	 * @param storage
	 * @param kernel
	 */
	public UResource(String id, String path, IStorageAdapter storage, Kernel kernel){
		this.id=id;
		this.path=path;
		this.storage=storage;
		this.kernel=kernel;
	}
	
	public void setOverwritePolicy(OverwritePolicy policy){
		this.overwrite=policy;
	}
	
	protected void updateTransferredBytes(){
		if(id!=null){
			FileServlet fs=kernel.getAttribute(FileServlet.class);
			fs.setTransferredBytes(id, transferred);
		}
	}
	
	@Override
	public Resource addPath(String path) throws IOException,
			MalformedURLException {
		return null;
	}

	@Override
	public boolean delete() throws SecurityException {
		return false;
	}

	@Override
	public boolean exists() {
		try{
			return storage.getProperties(path)!=null;
		}
		catch(ExecutionException ee){}
		return true;
	}

	@Override
	public File getFile() throws IOException {
		return null;
	}

	@Override
	public InputStream getInputStream() throws IOException {
		try {
			final InputStream is=storage.getInputStream(path);
			InputStream decoratedStream=new InputStream(){
				@Override
				public int read() throws IOException {
					return is.read();
				}
				@Override
				public int read(byte[] b, int off, int len) throws IOException {
					int r=is.read(b, off, len);
					if(r>0){
						transferred+=r;
						updateTransferredBytes();
					}
					return r;
				}
				
				@Override
				public void close() throws IOException {
					is.close();
				}
				
			};
			return decoratedStream;
			
		}
		catch(Exception e){
			IOException ioe=new IOException();
			ioe.initCause(e);
			throw ioe;
		}
	}

	@Override
	public String getName() {
		return path;
	}

	@Override
	public OutputStream getOutputStream() throws IOException, SecurityException {
		try {
			boolean append=OverwritePolicy.APPEND.equals(overwrite);
			final OutputStream os=storage.getOutputStream(path,append);
			OutputStream decoratedStream=new OutputStream(){
				@Override
				public void write(int b) throws IOException {
					os.write(b);
				}
				@Override
				public void write(byte[] b, int off, int len) throws IOException {
					os.write(b, off, len);
					transferred+=len;
					updateTransferredBytes();
				}
				@Override
				public void close() throws IOException {
					os.close();
				}
				@Override
				public void flush() throws IOException {
					os.flush();
				}
			};
			return decoratedStream;
		}
		catch(Exception e){
			IOException ioe=new IOException();
			ioe.initCause(e);
			throw ioe;
		}
	}

	@Override
	public URL getURL() {
		return null;
	}

	@Override
	public boolean isDirectory() {
		return false;
	}

	@Override
	public long lastModified() {
		return 0;
	}

	@Override
	public long length() {
		try{
			return storage.getProperties(path).getSize();
		}
		catch(Exception ex){
			return 0;
		}
	}

	@Override
	public String[] list() {
		return null;
	}

	@Override
	public void release() {
		
	}

	@Override
	public boolean renameTo(Resource dest) throws SecurityException {
		return false;
	}

	@Override
	public boolean isContainedIn(Resource r) throws MalformedURLException
	{
		return false;
	}

}
