package de.fzj.unicore.uas.fts.http;

import java.io.IOException;
import java.util.Map;

import javax.xml.namespace.QName;

import org.apache.log4j.Logger;

import de.fzj.unicore.uas.UASProperties;
import de.fzj.unicore.uas.ft.http.AccessURLDocument;
import de.fzj.unicore.uas.fts.FileTransferImpl;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.wsrflite.ContainerProperties;
import de.fzj.unicore.wsrflite.persistence.Persist;
import de.fzj.unicore.wsrflite.utils.Utilities;
import de.fzj.unicore.wsrflite.xmlbeans.rp.ImmutableResourceProperty;

/**
 * "Baseline" file transfer, which exposes a file using HTTP(s) via Jetty<br/>
 * 
 * The URI for download is "cryptographically hard to guess" and
 * is exposed using a WSRF resource property, thus protected
 * by the usual U6 access control.<br/>
 * 
 * Based on an idea by Roger Menday. <br/>
 * 
 * @author schuller
 * @since 1.0.1
 */
public class HttpFileTransferImpl extends FileTransferImpl{
	
	private static final Logger logger=LogUtil.getLogger(LogUtil.SERVICES,HttpFileTransferImpl.class);
	
	public final static QName RPAccessURL=AccessURLDocument.type.getDocumentElementName();

	private static final QName portType=new QName("http://unigrids.org/2006/04/services/bfts","BaselineFileTransferService");
		

	@Persist
	protected String unique;

	@Persist
	protected String accessURL;

	@Override
	public void initialise(String name, Map<String, Object> map)throws Exception{
		super.initialise(name, map);
		try{
			FileServlet.initialise(getKernel());
			String url=exposeFile();
			AccessURLDocument urlRP=AccessURLDocument.Factory.newInstance();
			urlRP.setAccessURL(url);
			properties.put(RPAccessURL, new ImmutableResourceProperty(urlRP));
			setOK();
		}catch(Exception e){
			LogUtil.logException("Error initialising filetransfer",e,logger);
			status=STATUS_FAILED;
			description="Error initialising filetransfer";
			throw e;
		}
	}
	
	private String exposeFile()throws IOException{
		if(unique==null){
			unique=Utilities.newUniqueID();
		}
		String path=(isExport? source : target);
		FileServlet fs=kernel.getAttribute(FileServlet.class);
		fs.exposeFile(unique, path, !overWrite,this.getStorageAdapter());
		accessURL=makeAccessURL(unique);
		return accessURL;
	}
	
	protected String makeAccessURL(String uniqueID){
		String add=null;
		
		ContainerProperties mainProps = getKernel().getContainerProperties(); 
		if(!uasProperties.getBooleanValue(UASProperties.SMS_DIRECT_FILETRANSFER)){
			//go via Gateway
			String base=mainProps.getBaseUrl();
			add=base.replace("services", "files")+"/"+uniqueID;
			if(logger.isDebugEnabled())logger.debug("Enabling HTTP access on URL: "+add);
		} else {
			//use UNICORE/X address
			String host=mainProps.getValue(ContainerProperties.WSRF_HOST);
			String port=mainProps.getValue(ContainerProperties.WSRF_PORT);
			String serv="/files";
			String proto="http";
			if (kernel.getContainerSecurityConfiguration().isSslEnabled()) {
				proto="https";
			}
			add=proto+"://"+host+":"+port+serv+"/"+uniqueID;
		}
		if(uasProperties.getBooleanValue(UASProperties.FTS_HTTP_PREFER_POST)){
			add=add+"?method=POST";
		}
		return add;
	}
	
	private void unExposeFile(){	
		FileServlet fs=kernel.getAttribute(FileServlet.class);
		fs.unExposeFile(unique);
	}
	
	@Override
	public void destroy() {
		unExposeFile();
		super.destroy();
	}

	
	@Override
	protected Long getTransferredBytes() {
		FileServlet fs=kernel.getAttribute(FileServlet.class);
		Long t=fs.getTransferredBytes(unique);
		if(t!=null){
			transferredBytes=t;
		}
		else{
			logger.debug("Can't get transferred bytes for transfer <"+unique+">");
		}
		return transferredBytes;
	}

	public QName getPortType()
	{
		return portType;
	}
}
