package de.fzj.unicore.uas.fts.http;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.log4j.Logger;
import org.eclipse.jetty.servlet.DefaultServlet;
import org.eclipse.jetty.servlet.ServletHolder;

import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.xnjs.io.IFileTransfer.OverwritePolicy;
import de.fzj.unicore.xnjs.io.IStorageAdapter;

/**
 * this servlet exposes files under a "hard to guess" URL 
 * 
 * @author schuller
 * @since 1.0.1
 */
public class FileServlet extends DefaultServlet {

	private static final long serialVersionUID = 1L;

	private static final Logger logger=LogUtil.getLogger(LogUtil.SERVICES,FileServlet.class);
	
	private final Map<String,UResource> map=new ConcurrentHashMap<String,UResource>();
	
	private final Map<String,Long> transferredBytes=new ConcurrentHashMap<String,Long>();
	
	private final Kernel kernel;
	
	public FileServlet(Kernel kernel){
		this.kernel=kernel;
	}

	public static synchronized void initialise(Kernel kernel){
		if(kernel.getAttribute(FileServlet.class)!=null)return;
		FileServlet fs=new FileServlet(kernel);
		kernel.setAttribute(FileServlet.class, fs);
		ServletHolder sh=new ServletHolder("fileServlet", fs);
		kernel.getServer().getRootServletContext().addServlet(sh,"/files/*");
	}
	
	public void exposeFile(String id, String remotePath, boolean append, IStorageAdapter storage){
		try{
			UResource fr=new UResource(id,remotePath,storage,kernel);
			if(append){
				fr.setOverwritePolicy(OverwritePolicy.APPEND);
			}
			map.put(id,fr);
		}
		catch(Exception mue){
			LogUtil.logException("",mue);
		}
	}
	
	public void exposeDirectory(String id, String remotePath, boolean append, IStorageAdapter storage){
		try{
			UDirectoryResource fr=new UDirectoryResource(id,remotePath,storage,kernel);
			if(append){
				fr.setOverwritePolicy(OverwritePolicy.APPEND);
			}
			map.put(id,fr);
		}
		catch(Exception mue){
			LogUtil.logException("",mue);
		}
	}
	
	public void unExposeFile(String id){
		UResource fr=map.get(id);
		if(fr!=null){
			fr.release();
		}
		map.remove(id);
		transferredBytes.remove(id);
	}
	
	public Long getTransferredBytes(String id){
		return transferredBytes.get(id);
	}
	
	public void setTransferredBytes(String id, Long transferred){
		transferredBytes.put(id,transferred);
	}
		

	@Override
	public UResource getResource(String pathInContext) {
		return getUResource(new File(pathInContext), null);
	}

	private UResource getUResource(File base, String fileName){
		String id=base.getName();
		UResource fr=map.get(id);
		if(fr==null){
			base=base.getParentFile();
			if(base==null)return null;
			if(fileName!=null){
				fileName=id+"/"+fileName;
			}
			else{
				fileName=id;
			}
			
			return getUResource(base, fileName);
		}
		else{
			if(fileName!=null && fr.isDirectory()){
				UResource newResource=new UResource(null, fr.path+"/"+fileName, fr.storage, kernel);
				return newResource;
			}
		}
		return fr;
	}
	

	private static final int BUFFER_SIZE = 2*8192;

	@Override
	protected void doPut(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		UResource r=getResource(request.getRequestURI());
		OutputStream out=null;
		InputStream in =null;
		try{
			out= r.getOutputStream();
			in= request.getInputStream();
			copy(in,out);
			response.setStatus(HttpURLConnection.HTTP_NO_CONTENT);
		}finally{
			if(out!=null)out.close();
		}
	} 
	
	
	private static final String multipart="multipart/form-data";
	
	@Override
	protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		UResource r=getResource(request.getRequestURI());
		OutputStream out=null;
		InputStream in =null;
		//check that it is the proper type
		String contentType=request.getHeader("Content-Type");
		if(contentType==null || contentType.indexOf(multipart)==-1){
			throw new ServletException("Invalid content type: only '"+multipart+"' is accepted.");
		}
		//OK just read the body
		try{
			out= r.getOutputStream();
			in= request.getInputStream();
			copy(in,out);
			response.setStatus(HttpURLConnection.HTTP_NO_CONTENT);
		}finally{
			if(out!=null)out.close();
		}
	} 
	
	//copy all data from an input stream to an output stream
	private void copy(InputStream in, OutputStream out)throws IOException{
		byte[] buffer = new byte[BUFFER_SIZE];
		int len=0;
		long total=0;
		while (true)
		{
			len=in.read(buffer,0,BUFFER_SIZE);
			if (len<0)
				break;
			if(len>0){
				out.write(buffer,0,len);
				total+=len;
			}
		}
		logger.debug("Total bytes copied : "+total);
		out.flush();
	}


}
