/*********************************************************************************
 * Copyright (c) 2006 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
 
package de.fzj.unicore.uas.fts.byteio;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import javax.xml.namespace.QName;

import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlObject;
import org.ggf.schemas.byteio.x2005.x10.byteIo.TransferInformationType;
import org.ggf.schemas.byteio.x2005.x10.streamableAccess.ReadableDocument;
import org.ggf.schemas.byteio.x2005.x10.streamableAccess.SeekReadDocument;
import org.ggf.schemas.byteio.x2005.x10.streamableAccess.SeekReadDocument.SeekRead;
import org.ggf.schemas.byteio.x2005.x10.streamableAccess.SeekReadResponseDocument;
import org.ggf.schemas.byteio.x2005.x10.streamableAccess.SeekWriteDocument;
import org.ggf.schemas.byteio.x2005.x10.streamableAccess.SeekWriteResponseDocument;
import org.ggf.schemas.byteio.x2005.x10.streamableAccess.SeekableDocument;
import org.ggf.schemas.byteio.x2005.x10.streamableAccess.TransferMechanismDocument;
import org.ggf.schemas.byteio.x2005.x10.streamableAccess.WriteableDocument;

import de.fzj.unicore.uas.fts.FileTransferImpl;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.wsrflite.persistence.Persist;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import de.fzj.unicore.wsrflite.xmlbeans.rp.ImmutableResourceProperty;
import de.fzj.unicore.xnjs.ems.ExecutionException;
import de.fzj.unicore.xnjs.io.IStorageAdapter;
import de.fzj.unicore.xnjs.simple.LocalTS;

/**
 * streamable byte implementation<br/>
 * SByteIO has the concept of a stateful access to a stream, so this class keeps 
 * references to existing Input/Output streams in a static map keyed with the id of the
 * filetransfer.
 
 * @author schuller
 */
public class StreamableByteIOImpl extends FileTransferImpl implements StreamableByteIO {

	private static final Logger logger=LogUtil.getLogger(LogUtil.SERVICES,StreamableByteIOImpl.class);

	private static final Map<String,Object> streams=Collections.synchronizedMap(new HashMap<String,Object>());
	
	@Persist
	protected Long currentPosition=0L;

	public SeekReadResponseDocument seekRead(SeekReadDocument req)
			throws BaseFault {
		try{
			SeekRead r=req.getSeekRead();
			String origin=r.getSeekOrigin();
			long offset=r.getOffset();
			long numBytes=r.getNumBytes();
			byte[] data=doRead(origin,offset,numBytes);
			SeekReadResponseDocument res=SeekReadResponseDocument.Factory.newInstance();
			TransferInformationType ti=res.addNewSeekReadResponse().addNewTransferInformation();
			String mechanism=r.getTransferInformation().getTransferMechanism();
			ti.set(ByteIO.encode(mechanism, data));
			ti.setTransferMechanism(mechanism);
			setDirty();
			setOK();
			return res;
		}catch(Exception e){
			LogUtil.logException("Could not perform seek-read", e, logger);
			status=STATUS_FAILED;
			description="Could not perform seek-read.";
			throw BaseFault.createFault("Could not perform seek-read: "+e.getMessage(),e);
		}
	}

	/**
	 * read bytes
	 * 
	 * @param origin - Only "seek from current position" and "seek from beginning" are supported 
	 * @param offset
	 * @param numBytes
	 * @return
	 * @throws IOException
	 * @throws ExecutionException
	 */
	protected byte[] doRead(String origin,long offset, long numBytes)throws IOException,ExecutionException{
		boolean fromBeginning=false;
		if(SBYTIO_ORIGIN_BEGINNING.equals(origin)){
			fromBeginning=true;
		}
		else if (!SBYTIO_ORIGIN_CURRENT.equals(origin)){
			throw new IOException("Can't seek from <"+origin+">, only 'current position' " +
					"and 'beginning of stream' are supported.");
		}
		if(fromBeginning==true){
			//make sure we get a new stream
			removeStream();
		}
		InputStream is=getInputStream();
		long skipped=is.skip(offset);
		if(skipped<offset){
			throw new IOException("Attempted to offset past end of file.");
		}
		currentPosition+=offset;
		byte[] data = new byte[(int)numBytes];
		ByteArrayOutputStream os=new ByteArrayOutputStream();
		int read=is.read(data);
		if (read > 0) {
			os.write(data,0,read);
			currentPosition+=read;
			setDirty();
		}
		return os.toByteArray();
	}
	
	public SeekWriteResponseDocument seekWrite(SeekWriteDocument req)
	throws BaseFault {
		try{
			String origin=req.getSeekWrite().getSeekOrigin();
			TransferInformationType ti=req.getSeekWrite().getTransferInformation();
			String mechanism=ti.getTransferMechanism();
			byte[] data=ByteIO.decode(mechanism,ti);
			long offset=req.getSeekWrite().getOffset();
			SeekWriteResponseDocument res=SeekWriteResponseDocument.Factory.newInstance();
			TransferInformationType ti2=res.addNewSeekWriteResponse().addNewTransferInformation();
			ti2.setTransferMechanism(mechanism);
			
			//test if we are local -> use RandomAccessFile if yes
			IStorageAdapter tsi=getStorageAdapter();
			if(tsi instanceof LocalTS){
				String file=isExport?source:target;
				RandomAccessFile f=((LocalTS)tsi).getRandomAccessFile(file);
				try{
					if(SBYTIO_ORIGIN_CURRENT.equalsIgnoreCase(origin)){
						currentPosition+=offset;
					}	
					else if(SBYTIO_ORIGIN_BEGINNING.equalsIgnoreCase(origin)){
						currentPosition=offset;
					}
					else if(SBYTIO_ORIGIN_END.equalsIgnoreCase(origin)){
						if(offset>0)throw new BaseFault("Positive offset from end-of-file not allowed.");
						currentPosition=f.length()+offset;
					}
					else{
						throw new BaseFault("Can't fulfil write request. Only seek-origin 'beginning' or 'current' is supported.");
					}
					f.seek(currentPosition);
					f.write(data);
				}finally{
					try{f.close();}catch(Exception e){}
				}
			}
			else{
				if(SBYTIO_ORIGIN_CURRENT.equalsIgnoreCase(origin)){
					//nothing to do
				}	
				else if(SBYTIO_ORIGIN_END.equalsIgnoreCase(origin)){
					//make sure data is appended
					overWrite=false;
				}
				else if(SBYTIO_ORIGIN_BEGINNING.equalsIgnoreCase(origin)){
					//make sure we create a new stream
					streams.remove(getUniqueID()+"_out");
				}
				else throw new BaseFault("Unsupported seek-origin");
				
				doWrite(data, offset);
			}
			currentPosition+=data.length;
			setDirty();
			setOK();
			return res;
		}
		catch(Exception e){
			LogUtil.logException("Could not perform seekWrite", e, logger);
			status=STATUS_FAILED;
			description="Could not perform seekWrite";
			throw BaseFault.createFault("Could not perform seekWrite: "+e.getMessage(),e);
		}
	}
	

	protected void doWrite(byte[] data, long offset)throws Exception{
		getOutputStream().write(data);
		getOutputStream().flush();
	}
	
	protected InputStream getInputStream()throws ExecutionException,IOException{
		InputStream is=(InputStream)streams.get(getUniqueID()+"_in");
		if(is==null){
			is=createNewInputStream();
			streams.put(getUniqueID()+"_in",is);
		}
		return is;
	}
	
	protected OutputStream getOutputStream()throws ExecutionException, IOException{
		OutputStream os=(OutputStream)streams.get(getUniqueID()+"_out");
		if(os==null){
			os=createNewOutputStream(!overWrite);
			streams.put(getUniqueID()+"_out",os);
		}
		return os;
	}
	
	@Override
	public void initialise(String sname, Map<String, Object> map) throws Exception{
		super.initialise(sname, map);
		
		properties.put(StreamableByteIO.RPSize,new ByteIOSizeResourceProperty(this));
	
		//transfermechanism(s)
		TransferMechanismDocument tmd=TransferMechanismDocument.Factory.newInstance();
		tmd.setTransferMechanism(RandomByteIO.TRANSFER_SIMPLE);
		TransferMechanismDocument tmd2=TransferMechanismDocument.Factory.newInstance();
		tmd2.setTransferMechanism(RandomByteIO.TRANSFER_MTOM);
		
		properties.put(RPTransferMechanisms, new ImmutableResourceProperty(
				new XmlObject[]{
						tmd, //tmd2
				}));
		
		ReadableDocument r=ReadableDocument.Factory.newInstance();
		r.setReadable(true);
		WriteableDocument w=WriteableDocument.Factory.newInstance();
		w.setWriteable(true);
		SeekableDocument s=SeekableDocument.Factory.newInstance();
		s.setSeekable(true);

		properties.put(RPReadable, new ImmutableResourceProperty(r));
		properties.put(RPWriteable, new ImmutableResourceProperty(w));
		properties.put(RPSeekable, new ImmutableResourceProperty(s));
		properties.put(RPEndOfStream, new EndOfStreamRP(this));
		properties.put(RPPosition, new PositionRP(this));
	}
	
	@Override
	public void destroy() {
		removeStream();
		super.destroy();
	}
	
	private void removeStream(){
		Object inStr=streams.remove(getUniqueID()+"_in");
		Object outStr=streams.remove(getUniqueID()+"_out");
		try{
			if(inStr!=null){
				((InputStream)inStr).close();
			}
			if(outStr!=null){
				((OutputStream)outStr).close();
			}
		}catch(Exception e){
			LogUtil.logException("Could not remove stream.",e,logger);
		}
	}
	
	protected boolean isEndOfStream(){
		try{
			Object str=streams.get(getUniqueID());
			if(str instanceof InputStream){
				return (!(((InputStream)str).available()>0));
			}
			else if(str instanceof OutputStream){
				return false;
			}
		}catch(Exception e){
			LogUtil.logException("Error accessing stream.",e,logger);
		}
		return true;
	}
	
    protected long getCurrentPosition(){
		return currentPosition;
	}
    
    private static final QName portType=new QName(SBYTIO_NS,"StreamableByteIO");
	
	public QName getPortType()
	{
		return portType;
	}
	
	//for unit testing
	public static boolean streamExists(String id){
		return streams.containsKey(id);
	}
}
