package de.fzj.unicore.uas.fts.byteio;

import java.io.IOException;
import java.math.BigInteger;
import java.util.Map;

import javax.xml.namespace.QName;

import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlObject;
import org.ggf.schemas.byteio.x2005.x10.byteIo.TransferInformationType;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.AppendDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.AppendResponseDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.ReadDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.ReadResponseDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.ReadableDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.TransferMechanismDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.TruncAppendDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.TruncAppendResponseDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.WriteDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.WriteResponseDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.WriteableDocument;

import de.fzj.unicore.uas.fts.DataResource;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import de.fzj.unicore.wsrflite.xmlbeans.ResourceProperty;
import de.fzj.unicore.wsrflite.xmlbeans.rp.ImmutableResourceProperty;

public abstract class RandomByteIOSupport {

	private static final Logger logger=LogUtil.getLogger(LogUtil.SERVICES,RandomByteIOSupport.class);

	private transient Kernel kernel;

	public RandomByteIOSupport() {
		super();

	}

	/**
	 * read some data
	 */
	public ReadResponseDocument read(ReadDocument req) throws BaseFault {
		if(logger.isTraceEnabled())logger.trace(req.toString());
		BigInteger offset=req.getRead().getStartOffset();
		long numBlocks=req.getRead().getNumBlocks();
		long bytesPerBlock=req.getRead().getBytesPerBlock();
		long stride=req.getRead().getStride();
		byte[] data;
		try {
			data = read(offset.longValue(),bytesPerBlock,numBlocks,stride);
		} catch (IOException e) {
			throw BaseFault.createFault("Could not perform read: "+e.getMessage(),e);
		}
		ReadResponseDocument res=ReadResponseDocument.Factory.newInstance();
		TransferInformationType ti=res.addNewReadResponse().addNewTransferInformation();
		String mechanism=req.getRead().getTransferInformation().getTransferMechanism();
		ti.set(ByteIO.encode(mechanism, data));
		ti.setTransferMechanism(mechanism);
		return res;

	}

	/**
	 * write some data
	 */
	public WriteResponseDocument write(WriteDocument req) throws BaseFault {
		if(logger.isTraceEnabled())logger.trace(req.toString());
		//get transfer params
		BigInteger offsetP=req.getWrite().getStartOffset();
		long offset= offsetP==null? 0L: offsetP.longValue();
		long bytesPerBlock=req.getWrite().getBytesPerBlock();
		long stride=req.getWrite().getStride();
		TransferInformationType ti=req.getWrite().getTransferInformation();
		String mechanism=ti.getTransferMechanism();
		byte[] data;
		try {
			data = ByteIO.decode(mechanism,ti);
			write(data,offset,bytesPerBlock,stride);
		} catch (Exception e) {
			throw BaseFault.createFault("Could not perform write: "+e.getMessage(),e);
		} 
		WriteResponseDocument res=WriteResponseDocument.Factory.newInstance();
		TransferInformationType tiRes=res.addNewWriteResponse().addNewTransferInformation();
		tiRes.setTransferMechanism(mechanism);

		return res;

	}

	/**
	 * append data
	 */
	public AppendResponseDocument append(AppendDocument req) throws BaseFault {

		TransferInformationType ti=req.getAppend().getTransferInformation();
		byte[] data;
		try {
			data = ByteIO.decode(ti.getTransferMechanism(),ti);
			append(data);
		} catch (Exception e) {
			throw BaseFault.createFault("Could not perform append: "+e.getMessage(),e);
		} 
		//TODO check the spec: what goes in here?
		AppendResponseDocument res=AppendResponseDocument.Factory.newInstance();
		res.addNewAppendResponse();
		res.getAppendResponse().addNewTransferInformation().setTransferMechanism(
				req.getAppend().getTransferInformation().getTransferMechanism());
		return res;

	}

	public TruncAppendResponseDocument truncAppend(TruncAppendDocument req)
	throws BaseFault {

		TransferInformationType ti=req.getTruncAppend().getTransferInformation();
		byte[] data;
		try {
			data = ByteIO.decode(ti.getTransferMechanism(),ti);
			BigInteger offset=req.getTruncAppend().getOffset();
			truncAppend(data, offset.longValue());
		} catch (Exception e) {
			throw BaseFault.createFault("Could not perform truncAppend: "+e.getMessage(),e);
		} 
		
		//TODO check the spec: what goes in here?
		TruncAppendResponseDocument res=TruncAppendResponseDocument.Factory.newInstance();
		res.addNewTruncAppendResponse();
		res.getTruncAppendResponse().addNewTransferInformation().setTransferMechanism(
				req.getTruncAppend().getTransferInformation().getTransferMechanism());
		return res;

	}


	/**
	 * this adds the byteio specific RPs to an RP doc <br/>
	 *  size, readable, writable, transfermechanism(s)
	 */
	public static void addByteIOProperties(DataResource resource, Map<QName,ResourceProperty> properties, long size, boolean readable, boolean writable) throws Exception {
		properties.put(RandomByteIO.RPSize,new ByteIOSizeResourceProperty(resource));

		ReadableDocument r=ReadableDocument.Factory.newInstance();
		r.setReadable(readable);
		WriteableDocument w=WriteableDocument.Factory.newInstance();
		w.setWriteable(writable);
		properties.put(RandomByteIO.RPReadable, new ImmutableResourceProperty(r));
		properties.put(RandomByteIO.RPWriteable, new ImmutableResourceProperty(w));
		
		
		//transfermechanism(s)
		TransferMechanismDocument tmd=TransferMechanismDocument.Factory.newInstance();
		tmd.setTransferMechanism(RandomByteIO.TRANSFER_SIMPLE);
		TransferMechanismDocument tmd2=TransferMechanismDocument.Factory.newInstance();
		tmd2.setTransferMechanism(RandomByteIO.TRANSFER_MTOM);

		properties.put(RandomByteIO.RPTransferMechanisms, new ImmutableResourceProperty(
				new XmlObject[]{
						tmd, //tmd2
				}));
		
		
	}

	public abstract byte[] read(long offset, long bytesPerBlock, long numBlocks, long stride) throws IOException;

	public abstract void write(byte[] data, long initialOffset, long bytesPerBlock, long stride) throws IOException;

	public abstract void append(byte[] data) throws IOException;

	public abstract void truncAppend(byte[] data, long offset) throws IOException;

	public void setKernel(Kernel kernel) {
		this.kernel = kernel;
	}
}
