/*********************************************************************************
 * Copyright (c) 2006 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
 
package de.fzj.unicore.uas.fts.byteio;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.math.BigInteger;
import java.util.Map;

import javax.xml.namespace.QName;

import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlObject;
import org.ggf.schemas.byteio.x2005.x10.byteIo.TransferInformationType;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.AppendDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.AppendResponseDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.ReadDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.ReadResponseDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.ReadableDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.TransferMechanismDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.TruncAppendDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.TruncAppendResponseDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.WriteDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.WriteResponseDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.WriteableDocument;

import de.fzj.unicore.uas.fts.FileTransferImpl;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.wsrflite.persistence.Persist;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import de.fzj.unicore.wsrflite.xmlbeans.rp.ImmutableResourceProperty;
import de.fzj.unicore.xnjs.ems.ExecutionException;
import de.fzj.unicore.xnjs.io.IStorageAdapter;
import de.fzj.unicore.xnjs.simple.LocalTS;

/**
 * Implementation of the random-access byteio interface
 * 
 * @author schuller
 */
public class RandomByteIOImpl extends FileTransferImpl implements RandomByteIO {

	private static final Logger logger=LogUtil.getLogger(LogUtil.SERVICES,RandomByteIOImpl.class);

	@Persist
	private long transferredBytes=0;
	
	
	@Override
	protected Long getTransferredBytes(){
		return transferredBytes;
	}
	
	/**
	 * read some data
	 */
	public ReadResponseDocument read(ReadDocument req) throws BaseFault {
		try{
			if(logger.isTraceEnabled())logger.trace(req.toString());
			BigInteger offset=req.getRead().getStartOffset();
			long numBlocks=req.getRead().getNumBlocks();
			long bytesPerBlock=req.getRead().getBytesPerBlock();
			long stride=req.getRead().getStride();
			byte[] data=doRead(offset.longValue(),bytesPerBlock,numBlocks,stride);
			ReadResponseDocument res=ReadResponseDocument.Factory.newInstance();
			TransferInformationType ti=res.addNewReadResponse().addNewTransferInformation();
			String mechanism=req.getRead().getTransferInformation().getTransferMechanism();
			ti.set(ByteIO.encode(mechanism, data));
			ti.setTransferMechanism(mechanism);
			setOK();
			return res;
		}catch(Exception e){
			LogUtil.logException("Could not perform read",e,logger);
			status=STATUS_FAILED;
			description="Could not perform read for request "+req;
			setDirty();
			throw BaseFault.createFault("Could not perform read",e);
		}
	}

	/**
	 * write some data
	 */
	public WriteResponseDocument write(WriteDocument req) throws BaseFault {
		try{
			if(logger.isTraceEnabled())logger.trace(req.toString());
			//get transfer params
			BigInteger offsetP=req.getWrite().getStartOffset();
			long offset= offsetP==null? 0L: offsetP.longValue();
			long bytesPerBlock=req.getWrite().getBytesPerBlock();
			long stride=req.getWrite().getStride();
			TransferInformationType ti=req.getWrite().getTransferInformation();
			String mechanism=ti.getTransferMechanism();
			byte[] data=ByteIO.decode(mechanism,ti);
			doWrite(data,offset,bytesPerBlock,stride);
			WriteResponseDocument res=WriteResponseDocument.Factory.newInstance();
			TransferInformationType tiRes=res.addNewWriteResponse().addNewTransferInformation();
			tiRes.setTransferMechanism(mechanism);
			setOK();
			return res;
		}catch(Exception e){
			LogUtil.logException("Could not perform write.",e,logger);
			status=STATUS_FAILED;
			description="Could not perform write for request "+req;
			setDirty();
			throw BaseFault.createFault("Could not perform write.",e);
		}
	}

	/**
	 * append data
	 */
	public AppendResponseDocument append(AppendDocument req) throws BaseFault {
		try{
			TransferInformationType ti=req.getAppend().getTransferInformation();
			byte[] data=ByteIO.decode(ti.getTransferMechanism(),ti);
			doAppend(data);
			//TODO check the spec: what goes in here?
			AppendResponseDocument res=AppendResponseDocument.Factory.newInstance();
			res.addNewAppendResponse();
			res.getAppendResponse().addNewTransferInformation().setTransferMechanism(
					req.getAppend().getTransferInformation().getTransferMechanism());
			setOK();
			return res;
		}catch(Exception e){
			e.printStackTrace();
			LogUtil.logException("Could not perform append().",e,logger);
			status=STATUS_FAILED;
			description="Could not perform append for request "+req;
			setDirty();
			throw BaseFault.createFault("Could not perform append",e);
		}
	}

	public TruncAppendResponseDocument truncAppend(TruncAppendDocument req)
			throws BaseFault {
		try{
			TransferInformationType ti=req.getTruncAppend().getTransferInformation();
			byte[] data=ByteIO.decode(ti.getTransferMechanism(),ti);
			BigInteger offset=req.getTruncAppend().getOffset();
			doTruncAppend(data, offset.longValue());
			//TODO check the spec: what goes in here?
			TruncAppendResponseDocument res=TruncAppendResponseDocument.Factory.newInstance();
			res.addNewTruncAppendResponse();
			res.getTruncAppendResponse().addNewTransferInformation().setTransferMechanism(
					req.getTruncAppend().getTransferInformation().getTransferMechanism());
			setOK();
			return res;
		}catch(Exception e){
			LogUtil.logException("Could not perform truncAppend().",e,logger);
			status=STATUS_FAILED;
			description="Could not perform truncAppend for request "+req;
			setDirty();
			throw BaseFault.createFault("Could not perform truncAppend",e);
		}
	}

	//helpers
	
	protected byte[] doRead(long offset,
				long bytesPerBlock, 
				long numBlocks, 
				long stride)
	throws IOException,ExecutionException{
		ByteArrayOutputStream os = new ByteArrayOutputStream();
		InputStream is=createNewInputStream();
		try{
			byte[] data=new byte[(int)bytesPerBlock];
			long skipped=is.skip(offset);
			if(skipped<offset){
				throw new IOException("Attempted to offset past end of file.");
			}
			for (int block=0; block<numBlocks; block++) {
				int read=is.read(data);
				if (read > 0) {
					transferredBytes+=read;
					os.write(data,0,read);
				}
				if(stride>0){
					if(is!=null)is.close();
					is=createNewInputStream();
					long t=offset+stride*(block+1);
					skipped=is.skip(t);
					if(skipped<t){
						throw new IOException("Attempted to skip past end of file.");
					}
				}
			}
			return os.toByteArray();
		}finally{
			try{is.close();}catch(Exception e){}
		}
	}
	
	protected void doWrite(byte[] data, long initialOffset, long bytesPerBlock, long stride)throws IOException,ExecutionException{
		//do we use a local TS ? if yes use RandomAccessFile
		IStorageAdapter tsi=getStorageAdapter();
		if(tsi instanceof LocalTS){
			String file=isExport?source:target;
			RandomAccessFile f=((LocalTS)tsi).getRandomAccessFile(file);
			try {
				int off = 0;
				while (off < data.length) {
					int toWrite = (data.length - off);
					if (toWrite > bytesPerBlock){
						toWrite = (int)bytesPerBlock;
					}
					if(initialOffset!=0)f.seek(initialOffset);
					f.write(data, off, toWrite);
					transferredBytes+=toWrite;
					initialOffset+= stride;
					off += toWrite;
				}
			} finally{
				if(f!=null)f.close();
			}
		}
		else{
			//fallback to output stream -> not all parameter combinations are supported
			doWriteToOutputStream(data, initialOffset, bytesPerBlock, stride);
		}
	}
	
	protected void doWriteToOutputStream(byte[] data, long initialOffset, long bytesPerBlock, long stride)throws IOException,ExecutionException{
		//else check if we can make sense of the params
		if(initialOffset>0) //we can only write a new file or else append 
		{
			throw new IllegalArgumentException("Can't fulfil write request.");
		}
		OutputStream os=createNewOutputStream(false);
		os.write(data,0,(int)bytesPerBlock);
		transferredBytes+=bytesPerBlock;
		os.close();
	}
	
	protected void doAppend(byte[] data)throws IOException,ExecutionException{
		//boolean append=... TODO check this 
		OutputStream os=createNewOutputStream(true);
		os.write(data);
		transferredBytes+=data.length;
		firstWrite=false;
		os.flush();
		os.close();
	}

	protected void doTruncAppend(byte[] data, long offset)throws IOException,ExecutionException{
		//do we use a local TS ? if yes use RandomAccessFile
		IStorageAdapter tsi=getStorageAdapter();
		if(tsi instanceof LocalTS){
			String file=isExport?source:target;
			RandomAccessFile f=((LocalTS)tsi).getRandomAccessFile(file);
			try{
				f.setLength(offset);
				f.seek(offset);
				f.write(data);
				transferredBytes+=data.length;
			}finally{
				if(f!=null)f.close();
			}
		}
		else{
			doTruncAppendToOutputStream(data, offset);
		}
	
	}
	
	protected void doTruncAppendToOutputStream(byte[] data, long offset)throws IOException,ExecutionException{
		if(!firstWrite){
			//check if file size equals offset
			if(getDataSize()!=offset){
				throw new IOException("Truncate is not supported.");
			}
			//or offset is zero meaning "create new file"
			if(offset==0){
				doWrite(data, 0, data.length, data.length);
			}
		}
		doAppend(data);
	}
	
	@Override
	public void initialise(String sname, Map<String, Object> map) throws Exception{
		super.initialise(sname, map);
	
		properties.put(RandomByteIO.RPSize,new ByteIOSizeResourceProperty(this));
		
		ReadableDocument r=ReadableDocument.Factory.newInstance();
		r.setReadable(true);
		WriteableDocument w=WriteableDocument.Factory.newInstance();
		w.setWriteable(true);
		properties.put(RPReadable, new ImmutableResourceProperty(r));
		properties.put(RPWriteable, new ImmutableResourceProperty(w));

		//transfermechanism(s)
		TransferMechanismDocument tmd=TransferMechanismDocument.Factory.newInstance();
		tmd.setTransferMechanism(RandomByteIO.TRANSFER_SIMPLE);
		TransferMechanismDocument tmd2=TransferMechanismDocument.Factory.newInstance();
		tmd2.setTransferMechanism(RandomByteIO.TRANSFER_MTOM);
		
		properties.put(RPTransferMechanisms, new ImmutableResourceProperty(
				new XmlObject[]{
						tmd, //tmd2
				}));
	}
	
	private static final QName portType=new QName(RBYTIO_NS,"RandomByteIO");
	
	public QName getPortType()
	{
		return portType;
	}
}
