package de.fzj.unicore.uas.fts;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.Logger;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.SourceTargetType;
import org.unigrids.services.atomic.types.ProtocolType;
import org.unigrids.x2006.x04.services.fts.ScheduledStartTimeDocument;
import org.unigrids.x2006.x04.services.fts.TransferRateDocument;

import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.uas.xnjs.U6FileTransferBase;
import de.fzj.unicore.uas.xnjs.XNJSFacade;
import de.fzj.unicore.wsrflite.messaging.ResourceDeletedMessage;
import de.fzj.unicore.wsrflite.persistence.Persist;
import de.fzj.unicore.wsrflite.security.util.AuthZAttributeStore;
import de.fzj.unicore.wsrflite.xmlbeans.WSUtilities;
import de.fzj.unicore.wsrflite.xmlbeans.rp.ImmutableResourceProperty;
import de.fzj.unicore.xnjs.Configuration;
import de.fzj.unicore.xnjs.io.IFileTransfer;
import de.fzj.unicore.xnjs.io.IFileTransfer.OverwritePolicy;
import de.fzj.unicore.xnjs.io.IFileTransfer.Status;
import eu.unicore.security.Client;
import eu.unicore.util.Log;


/**
 * WS-Resource for initiating and monitoring
 * a server-to-server file transfer
 * 
 * the source parameter is a UNICORE6 URI 
 * the target is the local file (relative to storage root)
 * 
 * @author schuller
 */
public class ServerToServerFileTransferImpl extends FileTransferImpl {

	private static final Logger logger=LogUtil.getLogger(LogUtil.SERVICES,ServerToServerFileTransferImpl.class);

	private transient IFileTransfer ft;

	/**
	 * this map contains active file transfers (Runnables) on this server
	 */
	final static Map<String,IFileTransfer>map=new ConcurrentHashMap<String,IFileTransfer>();

	@Persist
	private long scheduledStartTime=0;

	@Persist
	private boolean reliableMode=false;
	
	@Persist
	protected Client client;

	public static final String PARAM_SCHEDULED_START="scheduledStartTime";
	
	public static final String INIT_RELIABLE="reliableMode";

	@Override
	public void initialise(String sname, Map<String, Object> map) throws Exception {
		super.initialise(sname, map);
		properties.put(RPStatus, new StatusResourceProperty(null,this));

		String startTime=extraParameters!=null ? extraParameters.get(PARAM_SCHEDULED_START) : null;
		if(startTime!=null){
			try{
				scheduledStartTime=parseDate(startTime);
				logger.info("Filetransfer scheduled to start at "+startTime);
			}catch(IllegalArgumentException ex){
				throw new Exception("Could not parse supplied scheduledStartTime as a date",ex);
			}
		}
		reliableMode=Boolean.TRUE.equals(map.get(INIT_RELIABLE));
		client = AuthZAttributeStore.getClient();
		overWrite=true;
		createTransfer(OverwritePolicy.OVERWRITE);
		properties.put(TransferRateDocument.type.getDocumentElementName(), new TransferRateResourceProperty(this));
		startFileTransfer();
	}
	

	@Override
	public void customPostActivate() {
		ft=map.get(getUniqueID());
		if(ft!=null){
			Status s=ft.getStatus();
			if(s.equals(Status.FAILED)){
				status=STATUS_FAILED;
				description=ft.getStatusMessage();
			}
			else if(s.equals(Status.DONE)){
				status=STATUS_DONE;
				description="File transfer done.";
			}
			else { //still running
				setOK();
			}
		}
	}

	/**
	 * returns <code>true</code> if the file transfer is not yet finished
	 */
	protected boolean isFinished(){
		return status==STATUS_DONE || status==STATUS_FAILED;
	}

	/**
	 * check if the filetransfer should be restarted, and tries to restart
	 */
	protected void checkRestart(){
		if(!isFinished() && map.get(getUniqueID())==null){
			logger.info("Attempting to recover file transfer "+toString()+
					(client!=null?" for "+client.getDistinguishedName():""));
			try{
				//on restart, must use stored client
				AuthZAttributeStore.setClient(client);
				createTransfer(OverwritePolicy.RESUME_FAILED_TRANSFER);
				startFileTransfer();
			}catch(Exception ex){
				status=STATUS_FAILED;
				description="Failed (during restart, error message: "+ex.getMessage()+")";
				Log.logException("Attempt to restart server-to-server file transfer "+getUniqueID()+" failed", ex);
			}
		}
	}

	@Override
	public void destroy() {
		if(!isFinished()){
			try{
				ft=map.get(getUniqueID());
				if(ft!=null){
					logger.info("Aborting filetransfer "+getUniqueID()+" for client "+getClient().getDistinguishedName());
					ft.abort();
				}
			}
			catch(Exception e){
				LogUtil.logException("Problem cleaning up filetransfer.",e,logger);
			}
			finally{
				map.remove(getUniqueID());
			}
		}
		//notify parent SMS
		try{
			ResourceDeletedMessage m=new ResourceDeletedMessage("deleted:"+getUniqueID());
			m.setServiceName(getServiceName());
			m.setDeletedResource(getUniqueID());
			getKernel().getMessaging().getChannel(WSUtilities.extractResourceID(parentEPR)).publish(m);
		}
		catch(Exception ex){
			LogUtil.logException("Problem notifying parent SMS.",ex,logger);
		}
		super.destroy();
	}

	@Override
	public long getDataSize(){
		return ft!=null ? ft.getDataSize() : -1; 
	}

	/**
	 * transfer rate in bytes per second
	 * @return
	 */
	public long getTransferRate(){
		if(ft!=null && ft instanceof U6FileTransferBase){
			U6FileTransferBase u6ft=(U6FileTransferBase)ft;
			long dataSize = getDataSize();
			long consumedMillis=u6ft.getElapsedTime();
			if(dataSize>0 && consumedMillis>0){
				//need bytes per second
				return 1000*dataSize/consumedMillis;
			}
		}
		
		return -1;
	}
	
	/**
	 * create, but not yet start file transfer
	 * @param policy - {@link OverwritePolicy}
	 * @throws Exception
	 */
	protected void createTransfer(OverwritePolicy policy)throws Exception{
		Configuration config=XNJSFacade.get(xnjsReference, kernel).getConfiguration();
		if(!isExport){
			ft=config.getFileTransferEngine().
					createFileImport(client,workdir,toURI(urlEncode(source)),target,policy,null);
		}
		else{
			ft=config.getFileTransferEngine().
					createFileExport(client,workdir,source,toURI(urlEncode(target)),policy,null);
		}
		
		map.put(getUniqueID(),ft);
		if(ft instanceof U6FileTransferBase){
			((U6FileTransferBase)ft).setStorageAdapter(getStorageAdapter());
			((U6FileTransferBase)ft).setReliableMode(reliableMode);
		}
		//set the actual protocl
		ProtocolType.Enum p=ProtocolType.Enum.forString(ft.getProtocol());
		if(p==null)p=ProtocolType.OTHER;
		properties.put(RPProtocol, new ProtocolRP(new ProtocolType.Enum[]{p}));
	}


	protected void startFileTransfer(){
		Calendar c=Calendar.getInstance();
		long delay=scheduledStartTime-System.currentTimeMillis();
		ScheduledStartTimeDocument sd=ScheduledStartTimeDocument.Factory.newInstance();
		if(delay>20000){
			c.add(Calendar.MILLISECOND, (int)delay);
		}
		sd.setScheduledStartTime(c);
		ImmutableResourceProperty rp=new ImmutableResourceProperty(sd);
		properties.put(ScheduledStartTimeDocument.type.getDocumentElementName(), rp);
		if(delay>20000){
			kernel.getContainerProperties().getThreadingServices().
				getScheduledExecutorService().schedule(ft, delay, TimeUnit.MILLISECONDS);
		}
		else{                     
			kernel.getContainerProperties().getThreadingServices().getExecutorService().execute(ft);
		}
	}

	@Override
	protected Long getTransferredBytes() {
		if(ft!=null){
			transferredBytes=ft.getTransferredBytes();
		}
		return transferredBytes;
	}


	@Override
	protected boolean isExport(){
		return false;
	}


	/**
	 * setup the source and target properties of this transfer<br/>
	 * 
	 * @param rawsource
	 * @param rawtarget
	 */
	@Override
	protected void initialiseSourceAndTarget(String rawsource, String rawtarget){
		SourceTargetType stt=SourceTargetType.Factory.newInstance();
		stt.setURI(rawsource);
		properties.put(RPSource, new SourceResourceProperty(stt));
		source=rawsource;

		SourceTargetType stt2=SourceTargetType.Factory.newInstance();
		stt2.setURI(rawtarget);
		properties.put(RPTarget, new TargetResourceProperty(stt2));
		target=rawtarget;
	}

	/**
	 * parses ISO8601 date/time format and returns the corresponding time in millis
	 */
	private static DateFormat format=null;
	private static synchronized long parseDate(String s)throws ParseException{
		if(format==null){
			format=new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ");
		}
		return format.parse(s).getTime();
	}
}
