/*********************************************************************************
 * Copyright (c) 2006 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/


package de.fzj.unicore.uas.fts;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Map;

import javax.xml.namespace.QName;

import org.apache.log4j.Logger;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.SourceTargetType;
import org.unigrids.services.atomic.types.ProtocolType;
import org.unigrids.services.atomic.types.StorageEndpointReferenceDocument;
import org.unigrids.x2006.x04.services.fts.FileTransferPropertiesDocument;
import org.unigrids.x2006.x04.services.fts.StatusType;
import org.unigrids.x2006.x04.services.fts.SummaryType;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.impl.UASWSResourceImpl;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.uas.xnjs.StorageAdapterFactory;
import de.fzj.unicore.uas.xnjs.XNJSFacade;
import de.fzj.unicore.wsrflite.persistence.Persist;
import de.fzj.unicore.wsrflite.xmlbeans.rp.ImmutableResourceProperty;
import de.fzj.unicore.xnjs.ems.ExecutionException;
import de.fzj.unicore.xnjs.io.IStorageAdapter;
import de.fzj.unicore.xnjs.io.XnjsFile;
import de.fzj.unicore.xnjs.tsi.TSI;
import eu.unicore.security.Client;
import eu.unicore.util.Log;

/**
 * a WS-Resource representing a File transfer<br/>
 * 
 * this class must be extended to support a specific protocol
 *  
 * @author schuller
 */
public abstract class FileTransferImpl extends UASWSResourceImpl implements DataResource, FileTransfer {

	private static final Logger logger=LogUtil.getLogger(LogUtil.SERVICES,FileTransferImpl.class);

	/**
	 * Init parameter: id of the XNJS action this Filetransfer belongs to
	 */
	public static final String INIT_ACTIONID="uas.filetransfer.impl.actionid";

	/**
	 * Init parameter: Filetransfer source
	 */
	public static final String INIT_SOURCE="uas.filetransfer.impl.source";

	/**
	 * Init parameter: Filetransfer target
	 */
	public static final String INIT_TARGET="uas.filetransfer.impl.target";

	/**
	 * Init parameter: pipe yes/no
	 */
	public static final String INIT_IS_PIPE="uas.filetransfer.impl.ispipe";

	/**
	 * Init parameter: protocol
	 */
	public static final String INIT_PROTOCOL="uas.filetransfer.impl.protocol";


	/**
	 * Init parameter: sms reference
	 */
	public static final String INIT_SMS_EPR="uas.filetransfer.impl.sms";

	/**
	 * Init parameter: sms workdir
	 */
	public static final String INIT_SMS_WORKDIR="uas.filetransfer.impl.sms.wd";

	/**
	 * Init parameter: import or export?
	 */
	public static final String INIT_IS_EXPORT="uas.filetransfer.impl.sms.isexport";

	/**
	 * Init parameter: overwrite existing file?
	 */
	public static final String INIT_OVERWRITE="uas.filetransfer.impl.sms.overwrite";

	/**
	 * Init parameter: current umask
	 */
	public static final String INIT_UMASK="uas.filetransfer.impl.sms.umask";

	/**
	 * Init parameter: the {@link StorageAdapterFactory} to use for accessing the backend storage
	 */
	public static final String INIT_STORAGE_ADAPTER_FACTORY="uas.filetransfer.impl.sms.storageadapterfactory";


	/**
	 * Init parameter: extra, protocol dependent parameters as a String/String Map 
	 */
	public static final String INIT_EXTRA_PARAMETERS="uas.filetransfer.impl.extraparameters";

	/**
	 * Configuration key: maps protocol to implementation class.
	 * For example, the config entry 
	 * "uas.filetransfer.protocol.HTTP=my.class.for.http" 
	 * will map the given class to the HTTP protocol.
	 * The class must extend {@link FileTransferImpl}
	 */
	public static final String CONFIG_PROTOCOL_KEY="uas.filetransfer.protocol.";

	@Persist
	protected String source;
	@Persist
	protected String target;
	@Persist
	protected Boolean isPipe;
	@Persist
	protected ProtocolType.Enum protocol;
	@Persist
	protected EndpointReferenceType parentEPR;
	@Persist
	protected String workdir;
	@Persist
	protected Boolean firstWrite=Boolean.TRUE;
	@Persist
	protected Boolean isExport=null;

	@Persist
	protected Boolean overWrite=Boolean.TRUE;

	@Persist
	protected Long transferredBytes=0L;

	@Persist
	protected Map<String,String>extraParameters=null;

	@Persist
	protected StorageAdapterFactory storageAdapterFactory;

	@Persist
	protected String umask;

	protected static final int STATUS_RUNNING=SummaryType.INT_RUNNING;
	protected static final int STATUS_DONE=SummaryType.INT_DONE;
	protected static final int STATUS_FAILED=SummaryType.INT_FAILED;
	protected static final int STATUS_UNDEFINED=SummaryType.INT_UNDEFINED;
	protected static final int STATUS_READY=SummaryType.INT_READY;

	@Persist
	protected int status=1;

	@Persist
	protected String description="";

	@Override
	public QName getResourcePropertyDocumentQName() {
		return FileTransferPropertiesDocument.type.getDocumentElementName();
	}

	/**
	 * initialisation
	 */
	@Override
	@SuppressWarnings("unchecked")
	public void initialise(String sname, Map<String, Object> map) throws Exception{
		super.initialise(sname,map);
		String rawsource=(String)map.get(INIT_SOURCE);
		String rawtarget=(String)map.get(INIT_TARGET);
		isPipe=(Boolean)map.get(INIT_IS_PIPE);
		protocol=(ProtocolType.Enum)map.get(INIT_PROTOCOL);
		parentEPR=(EndpointReferenceType)map.get(INIT_SMS_EPR);
		workdir=(String)map.get(INIT_SMS_WORKDIR);
		isExport=(Boolean)map.get(INIT_IS_EXPORT);
		Boolean o=(Boolean)map.get(INIT_OVERWRITE);
		if(o!=null)overWrite=o;
		umask = (String)map.get(INIT_UMASK);
		extraParameters=(Map<String,String>)map.get(INIT_EXTRA_PARAMETERS);
		//setup resource properties
		properties.put(RPProtocol, new ProtocolRP(new ProtocolType.Enum[]{protocol}));
		properties.put(RPTransferred, new TransferredBytesResourceProperty(Long.valueOf(0),this));
		properties.put(RPStatus, new StatusResourceProperty(null,this));
		StorageEndpointReferenceDocument parent=StorageEndpointReferenceDocument.Factory.newInstance();
		parent.setStorageEndpointReference(parentEPR);
		properties.put(RPParentSMS, new ImmutableResourceProperty(parent));
		initialiseSourceAndTarget(rawsource, rawtarget);
		properties.put(RPSize,new UASSizeResourceProperty(this));
		
		storageAdapterFactory=(StorageAdapterFactory)map.get(INIT_STORAGE_ADAPTER_FACTORY);
		if(isPipe && !isExport()){
			createFIFO();
		}
		logger.info("New file transfer: "+toString());
	}

	/**
	 * setup the source and target of this transfer<br/>
	 * 
	 * @param rawsource
	 * @param rawtarget
	 */
	protected void initialiseSourceAndTarget(String rawsource, String rawtarget){
		if(rawsource!=null){
			SourceTargetType stt=SourceTargetType.Factory.newInstance();
			stt.setURI(makeQualifiedURI(rawsource));
			properties.put(RPSource, new SourceResourceProperty(stt));
			source=rawsource;
		}
		else{
			properties.put(RPSource, new SourceResourceProperty(null));
		}
		if(rawtarget!=null){
			SourceTargetType stt=SourceTargetType.Factory.newInstance();
			stt.setURI(makeQualifiedURI(rawtarget));
			properties.put(RPTarget, new TargetResourceProperty(stt));
			target=rawtarget;
		}
		else{
			properties.put(RPTarget, new TargetResourceProperty(null));
		}
	}

	protected boolean isExport(){
		return isExport;
	}

	protected IStorageAdapter getStorageAdapter()throws IOException{
		if(storageAdapterFactory!=null){
			IStorageAdapter adapter = storageAdapterFactory.createStorageAdapter(this);
			adapter.setUmask(umask);
			adapter.setStorageRoot(workdir);
			return adapter;
		}
		else throw new IOException("No storage adaptor factory.");
	}

	public long getDataSize(){
		String filename=source!=null?source:target;
		try{
			XnjsFile f=getStorageAdapter().getProperties(filename);
			if(f!=null){
				return f.getSize();
			}
		}catch(Exception e){
			LogUtil.logException("Could not determine file size for <"+filename+">",e,logger);
		}

		return 0L;
	}

	/**
	 * create an input stream for reading from the backend storage
	 * @return
	 * @throws ExecutionException
	 */
	protected InputStream createNewInputStream()throws IOException,ExecutionException{
		InputStream is=getStorageAdapter().getInputStream(source);
		return is;
	}

	/**
	 * create an output stream for writing to the backend storage
	 * @return
	 * @throws ExecutionException
	 */
	protected OutputStream createNewOutputStream(boolean append)throws IOException,ExecutionException{
		OutputStream os=getStorageAdapter().getOutputStream(target,append);
		return os;
	}

	protected void createFIFO()throws ExecutionException{
		TSI tsi=XNJSFacade.get(xnjsReference, kernel).getTSI(getClient());
		tsi.setStorageRoot(workdir);
		tsi.mkfifo(target);
	}

	protected void destroyFIFO()throws ExecutionException{
		TSI tsi=XNJSFacade.get(xnjsReference, kernel).getTSI(getClient());
		tsi.setStorageRoot(workdir);
		tsi.rm(target);
	}

	@Override
	public void destroy(){
		if(isPipe && !isExport()){
			try{
				logger.info("Removing fifo...");
				destroyFIFO();
			}catch(Exception e){
				Log.logException("Can't destroy fifo <"+target+">", e, logger);
			}
		}
		super.destroy();
	}

	/*
	 * generates a fully-qualified URI string, including protocol and SMS
	 * reference. The URI is encoded properly.
	 * 
	 */
	protected String makeQualifiedURI(String path){
		String prefix=protocol.toString().toLowerCase()+":"+parentEPR.getAddress().getStringValue()+"#";
		return urlEncode(prefix+path);
	}

	/**
	 * encode some characters that are illegal in URIs
	 * @param orig
	 * @return
	 */
	public static String urlEncode(String orig){
		try{
			return orig.replaceAll(" ", "%20");
		}catch(Exception e){
			logger.error(e);
			return orig;
		}
	}

	protected URI toURI(String path)throws URISyntaxException{
		URI uri= new URI(path);
		return uri;
	}

	protected Long getTransferredBytes(){
		return transferredBytes;
	}

	protected StatusType getStatus(){
		StatusType ret=StatusType.Factory.newInstance();
		ret.setSummary(SummaryType.Enum.forInt(status));
		ret.setDescription(description);
		return ret;
	}

	protected void setReady(){
		status=STATUS_READY;
		description="Ready.";
	}

	protected void setOK(){
		status=STATUS_RUNNING;
		description="OK.";
	}

	@Override
	public String toString(){
		StringBuilder sb=new StringBuilder();
		if(source!=null){
			sb.append(" from '").append(source).append("'");
		}
		if(target!=null){
			sb.append(" to '").append(target).append("'");
		}

		Client cl=getClient();
		
		if(cl!=null && cl.getDistinguishedName()!=null){
			sb.append(" for ").append(cl.getDistinguishedName());
		}
		sb.append(" protocol=").append(protocol);
		sb.append(" isExport=").append(isExport);
		sb.append(" overwrite=").append(overWrite);
		sb.append(" workdir=").append(workdir);
		sb.append(" isPipe=").append(isPipe);
		sb.append(" myAddress=").append(getEPR().getAddress().getStringValue());

		return sb.toString();
	}


	private static final QName portType=new QName("http://unigrids.org/2006/04/services/fts","FileTransfer");

	public QName getPortType()
	{
		return portType;
	}

}
