package de.fzj.unicore.uas.fts;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.ServiceLoader;

import de.fzj.unicore.uas.fts.byteio.RandomByteIOImpl;
import de.fzj.unicore.uas.fts.byteio.StreamableByteIOImpl;
import de.fzj.unicore.uas.fts.http.HttpFileTransferImpl;
import de.fzj.unicore.uas.xnjs.BFTExport;
import de.fzj.unicore.uas.xnjs.BFTImport;
import de.fzj.unicore.uas.xnjs.RByteIOExport;
import de.fzj.unicore.uas.xnjs.RByteIOImport;
import de.fzj.unicore.uas.xnjs.SByteIOExport;
import de.fzj.unicore.uas.xnjs.SByteIOImport;
import de.fzj.unicore.uas.xnjs.U6FileTransferBase;
import de.fzj.unicore.wsrflite.Capabilities;
import de.fzj.unicore.wsrflite.Capability;

/**
 * Describes the available filetransfers in UAS Core<br/>
 * 
 * Additionally there are some utility methods for finding the available filetransfer
 * capabilities
 * 
 * @author schuller
 */
public class FileTransferCapabilities {

	/**
	 * get the appropriate FT implementation for the given protocol 
	 * by checking the available {@link FileTransferCapabilities}
	 * 
	 * @param protocol - the protocol name
	 * @return a file transfer implementation
	 */
	public static FileTransferImpl getFileTransferImpl(String protocol)throws Exception{
		FileTransferCapability cp=getCapability(protocol);
		String clazz=cp.getImplementation().getName();
		
		return (FileTransferImpl)Class.forName(clazz).newInstance();
	}

	private static final Map<String,FileTransferCapability>filetransferCapabilities=new HashMap<String, FileTransferCapability>();

	public static synchronized FileTransferCapability getCapability(String protocol){
		FileTransferCapability result=filetransferCapabilities.get(protocol);
		if(result==null){
			loadCapabilities();
			result=filetransferCapabilities.get(protocol);
		}
		return result;
	}

	private static void loadCapabilities(){
		ServiceLoader<Capabilities> sl=ServiceLoader.load(Capabilities.class);
		Iterator<Capabilities>iter=sl.iterator();
		while(iter.hasNext()){
			Capability[]cs=iter.next().getCapabilities();
			for(int j=0; j<cs.length;j++){
				Capability c=cs[j];
				if(c instanceof FileTransferCapability){
					FileTransferCapability ftc=(FileTransferCapability)c;
					filetransferCapabilities.put(ftc.getProtocol(),ftc);
				}
			}
		}
	}
	
	public static synchronized String[] getProtocols(){
		if(filetransferCapabilities.size()==0)loadCapabilities();
		int l=filetransferCapabilities.size();
		String[] res=new String[l];
		int i=0;
		for(FileTransferCapability fc: filetransferCapabilities.values()){
			res[i]=fc.getProtocol();
			i++;
		}
		return res;
	}
	
	public static FileTransferCapability BFT = new FileTransferCapability(){

		@Override
		public String getProtocol() {
			return "BFT";
		}

		@Override
		public Class<?> getImplementation() {
			return HttpFileTransferImpl.class;
		}

		@Override
		public Class<?> getInterface() {
			return FileTransferImpl.class;
		}

		@Override
		public Class<? extends U6FileTransferBase> getExporter() {
			return BFTExport.class;
		}

		@Override
		public Class<? extends U6FileTransferBase> getImporter() {
			return BFTImport.class;
		}

	};


	public static FileTransferCapability RBYTEIO = new FileTransferCapability(){

		@Override
		public String getProtocol() {
			return "RBYTEIO";
		}

		@Override
		public Class<?> getImplementation() {
			return RandomByteIOImpl.class;
		}

		@Override
		public Class<?> getInterface() {
			return FileTransferImpl.class;
		}
		
		@Override
		public Class<? extends U6FileTransferBase> getExporter() {
			return RByteIOExport.class;
		}

		@Override
		public Class<? extends U6FileTransferBase> getImporter() {
			return RByteIOImport.class;
		}
	};

	public static FileTransferCapability SBYTEIO = new FileTransferCapability(){

		@Override
		public String getProtocol() {
			return "SBYTEIO";
		}

		@Override
		public Class<?> getImplementation() {
			return StreamableByteIOImpl.class;
		}

		@Override
		public Class<?> getInterface() {
			return FileTransferImpl.class;
		}
		@Override
		public Class<? extends U6FileTransferBase> getExporter() {
			return SByteIOExport.class;
		}

		@Override
		public Class<? extends U6FileTransferBase> getImporter() {
			return SByteIOImport.class;
		}
	};
}
