/*
 * Copyright (c) 2012 ICM Uniwersytet Warszawski All rights reserved.
 * See LICENCE.txt file for licensing information.
 */
package de.fzj.unicore.uas;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.log4j.Logger;

import de.fzj.unicore.uas.impl.sms.StorageDescription;
import de.fzj.unicore.uas.impl.sms.SMSBaseImpl;
import de.fzj.unicore.uas.impl.sms.StorageInfoProvider;
import de.fzj.unicore.uas.impl.sms.StorageSharingMode;
import de.fzj.unicore.uas.impl.sms.StorageManagementHomeImpl.StorageTypes;
import de.fzj.unicore.uas.metadata.MetadataManager;

import eu.unicore.util.Log;
import eu.unicore.util.configuration.ConfigurationException;
import eu.unicore.util.configuration.DocumentationReferenceMeta;
import eu.unicore.util.configuration.DocumentationReferencePrefix;
import eu.unicore.util.configuration.PropertiesHelper;
import eu.unicore.util.configuration.PropertyChangeListener;
import eu.unicore.util.configuration.PropertyMD;

/**
 * Configuration of the settings specific to UNICORE core services.
 * Except of defining the typical properties, a methods to retrieve parsed storage and storage factory 
 * descriptions are provided.
 * 
 * @author K. Benedyczak
 * @author B. Schuller
 */
public class UASProperties extends PropertiesHelper {
	private static final Logger log = Log.getLogger(Log.CONFIGURATION, UASProperties.class);
	
	@DocumentationReferencePrefix
	public static final String PREFIX = "coreServices."; 
	
	/**
	 * set the name or the XNJS config file
	 */
	public static final String TSF_XNJS_CONFIGFILE = "targetsystemfactory.xnjs.configfile";

	/**
	 * property defining whether to disable users' home SMSes
	 */
	public static final String TSS_DISABLE_HOME_PROPERTY = "targetsystem.homeDisable";

	/**
	 * if set to <code>true</code>, the storages attached to a TSS will always
	 * have unique IDs. In the default case, storage names will be formed from
	 * the user's xlogin and the storage name, e.g. "a.user-Home".
	 */
	public static final String TSS_FORCE_UNIQUE_STORAGE_IDS = "targetsystem.uniqueStorageIds";

	/**
	 * Property defining the default protocols supported by the SMS<br/> 
	 * This is a list of String-separated protocol names, e.g. "RBYTEIO SBYTEIO"
	 */
	public static final String SMS_PROTOCOLS = "sms.protocols";

	public static final String SMS_LS_LIMIT = "sms.lsLimit";
	
	public static final String SMS_ENABLED_ADDON_STORAGES="targetsystem.enabledStorages";
	
	public static final String SMS_ADDON_STORAGE_PREFIX="targetsystem.storage.";
	
	public static final String SMS_FACTORY_PREFIX = "sms.factory.";
	
	public static final String SMS_ENABLED_FACTORIES = "sms.enabledFactories";

	public static final String SMS_GROUP_SHARING_MODE = "sms.sharingMode";
	
	/**
	 * When doing file transfers, UNICORE tries to detect whether two storage
	 * resources are accessing the same filesystem. If yes, the transfer is done
	 * by direct copying. Set to "true" to disable this feature.
	 */
	public final static String SMS_TRANSFER_FORCEREMOTE = "filetransfer.forceremote";

	/**
	 * publish filetransfer URLs that "bypass" the gateway
	 */
	public static final String SMS_DIRECT_FILETRANSFER = "filetransfer.direct";
	
	/**
	 * How many file transfer threads should be maximally used at the same time. 
	 * Default is "10" 
	 */
	public static final String SMS_STAGING_MAXTHREADS = "filetransfer.maxthreads";
	
	public static final String FTS_HTTP_PREFER_POST = "filetransfer.httpPreferPost";
		
	/**
	 * the base directory (relative to the TSI)
	 */
	public static final String DEFSMS_WORKDIR="defaultsms.workdir";

	/**
	 * the SMS implementation to use
	 */
	public static final String DEFSMS_CLASS="defaultsms.class";

	/**
	 * the name to use for this storage
	 */
	public static final String DEFSMS_NAME="defaultsms.name";
	
	/**
	 * the protocols to configure for this storage
	 */
	public static final String DEFSMS_CONF_PROTOCOLS="defaultsms.protocols";
	
	/**
	 * the property name for configuring the "default" MetadataManager implementation
	 */
	public static final String METADATA_MANAGER_CLASSNAME="metadata.managerClass";
	
	//TODO this is from gridbean module and should be moved there. See PublishGridBean comment
	public static final String GRIDBEAN_DIR = "gridbean.directory";
	
	
	private Collection<StorageDescription> addOnStorages;
	private Map<String, StorageDescription> factories;
	
	@DocumentationReferenceMeta
	public static final Map<String, PropertyMD> META = new HashMap<String, PropertyMD>();
	static {
		META.put(TSS_DISABLE_HOME_PROPERTY, new PropertyMD("false").
				setDescription("Whether to disable a SMS providing access to the user's home directory."));
		META.put(TSS_FORCE_UNIQUE_STORAGE_IDS, new PropertyMD("false").
				setDescription("Whether to use unique identifiers for all storages. If set to true, the storages attached to a TSS will always have unique IDs. In the default case, storage names will be formed from the user's xlogin and the storage name, e.g. 'a.user-Home'. Unique identifiers are long and not possible to be predicted but are needed if many grid users are mapped to the same local account."));
		//TODO - document all possible values
		META.put(SMS_PROTOCOLS, new PropertyMD("BFT RBYTEIO SBYTEIO").setUpdateable().
				setDescription("Space separated list of the default file transport protocols supported by SMSes."));
		META.put(SMS_TRANSFER_FORCEREMOTE, new PropertyMD("false").
				setDescription("When doing file transfers, UNICORE tries to detect whether two storage resources are accessing the same (local) filesystem. If yes, the transfer is done by direct copying. Set to 'true' to disable this feature."));
		
		//TODO - should maybe deprecate this, as it creates too much confusion
		META.put(SMS_DIRECT_FILETRANSFER, new PropertyMD("false").
				setDescription("If set to 'true', UNICORE/X will attempt to bypass the gateway for filetransfers. Note: there are firewall and security implications!"));
		
		META.put(SMS_STAGING_MAXTHREADS, new PropertyMD("10").
				setDescription("The maximum number of file transfer threads."));
		
		//note: if you want to change the maximum of the property, 
		//then also implementation of ListDirectory needs to be changed as it uses the constants.
		//default and min can be freely changed. 
		META.put(SMS_LS_LIMIT, new PropertyMD(SMSBaseImpl.MAX_LS_RESULTS+"").setBounds(1, SMSBaseImpl.MAX_LS_RESULTS).
				setDescription("Controls the default amount of ListDirectory results which are returned in a single query."));
		META.put(SMS_ENABLED_ADDON_STORAGES, new PropertyMD((String)null).
				setDescription("Space separated list of names of enabled additional storages. If this property is left undefined then all defined storages are enabled. If this property value is empty then all are disabled."));
		META.put(SMS_ADDON_STORAGE_PREFIX, new PropertyMD().setCanHaveSubkeys().
				setDescription("Properties with this prefix are used to configure additional storages. See documentation of TargetSystem storages for details."));
		META.put(SMS_FACTORY_PREFIX, new PropertyMD().setCanHaveSubkeys().
				setDescription("Properties with this prefix are used to configure storage factories. See documentation of storage factories for details."));
		META.put(SMS_ENABLED_FACTORIES, new PropertyMD((String)null).
				setDescription("Space separated list of names of enabled storage factories.  If this property is left undefined then all defined factories are enabled. If this property value is empty then all are disabled."));
		META.put(SMS_GROUP_SHARING_MODE, new PropertyMD(StorageSharingMode.DISABLED).
				setDescription("How the group sharing of files is implemented. In the CHMOD files can be shared within one group only, but this is supported by most storages. The ACL implementation allows for sharing between multiple groups, but ACLs must be supported by the storage."));
		META.put(FTS_HTTP_PREFER_POST, new PropertyMD("false").
				setDescription("Controls whether to use HTTP POST for the HTTP filetransfers."));
		
		//TODO shall it be mandatory?
		META.put(TSF_XNJS_CONFIGFILE, new PropertyMD().setPath().
				setDescription("The path or the XNJS config file."));
		//TODO - default? mandatory? without this set metadata won't work
		META.put(METADATA_MANAGER_CLASSNAME, new PropertyMD().setClass(MetadataManager.class).
				setDescription("Metadata manager class name."));
		META.put(DEFSMS_CLASS, new PropertyMD().setClass(StorageManagement.class).
				setDescription("Class name providing implementation of the default SMS. If not set then default SMS is set to 'VARIABLE' SMS type."));
		META.put(DEFSMS_NAME, new PropertyMD("SHARE").
				setDescription("Name of the default SMS."));
		META.put(DEFSMS_WORKDIR, new PropertyMD().
				setDescription("Working directory of the default SMS. If not set, then system's default temp directory is used."));
		META.put(DEFSMS_CONF_PROTOCOLS, new PropertyMD().
				setDescription("Space separated list of the file transport protocols supported by the default SMS."));

		META.put(GRIDBEAN_DIR, new PropertyMD().setPath().
				setDescription("The path where GridBeans for the GridBeanService are stored."));
		
		//register namespaces for other modules
		META.put("metadata.", new PropertyMD().setHidden().setCanHaveSubkeys());
		META.put("uftp.", new PropertyMD().setHidden().setCanHaveSubkeys());
		META.put("dsms.", new PropertyMD().setHidden().setCanHaveSubkeys());
		META.put("bes.", new PropertyMD().setHidden().setCanHaveSubkeys());
		META.put("cip.", new PropertyMD().setHidden().setCanHaveSubkeys());
		META.put("hadoop.", new PropertyMD().setHidden().setCanHaveSubkeys());
		META.put("udt.", new PropertyMD().setHidden().setCanHaveSubkeys());
		META.put("uftp.", new PropertyMD().setHidden().setCanHaveSubkeys());
		META.put("xtreemfs.", new PropertyMD().setHidden().setCanHaveSubkeys());
		META.put("extension.", new PropertyMD().setHidden().setCanHaveSubkeys());
	}

	public UASProperties(Properties properties) throws ConfigurationException, IOException {
		super(PREFIX, properties, META, log);
		factories = parseStorages(SMS_FACTORY_PREFIX, SMS_ENABLED_FACTORIES, true);
		addOnStorages = parseStorages(SMS_ADDON_STORAGE_PREFIX, 
				SMS_ENABLED_ADDON_STORAGES, false).values();

		addPropertyChangeListener(new PropertyChangeListener() {
			private final String[] PROPS = {SMS_ADDON_STORAGE_PREFIX, SMS_FACTORY_PREFIX};
			
			@Override
			public void propertyChanged(String propertyKey) {
				if (propertyKey.equals(SMS_ADDON_STORAGE_PREFIX))
					updateStorages(addOnStorages, SMS_ADDON_STORAGE_PREFIX, false);
				else if (propertyKey.equals(SMS_FACTORY_PREFIX))
					updateStorages(factories.values(), SMS_FACTORY_PREFIX, true);
			}
			
			@Override
			public String[] getInterestingProperties() {
				return PROPS;
			}
		});
	}
	
	@Override
	protected void checkConstraints() throws ConfigurationException {
		super.checkConstraints();
		//this could be optimized little bit as creation of StorageDescriptions is not required for parsing... 
		parseStorages(SMS_FACTORY_PREFIX, SMS_ENABLED_FACTORIES, true);
		parseStorages(SMS_ADDON_STORAGE_PREFIX, SMS_ENABLED_ADDON_STORAGES, false);
	}

	/**
	 * @return a list of TSMSes configurations
	 */
	public Collection<StorageDescription> getAddonStorages() {
		return addOnStorages;
	}

	/**
	 * @return a map of SMSFactories configurations
	 */
	public Map<String, StorageDescription> getStorageFactories() {
		return factories;
	}

	/**
	 * Updates the existing storage descriptions
	 */
	private void updateStorages(Collection<StorageDescription> storages, 
			String prefixComponent, boolean factory) {
		for(StorageDescription storage: storages) {
			String pfx = prefixComponent + storage.getId() + ".";
			SMSProperties smsProps = factory ? new SMSFactoryProperties(pfx, properties) :
					new SMSProperties(pfx, properties);
			storage.update(smsProps.getValue(SMSProperties.PROTOCOLS), 
					smsProps.getBooleanValue(SMSProperties.FILTER_LISTING),
					smsProps.getBooleanValue(SMSProperties.CLEANUP),
					smsProps.getValue(SMSProperties.UMASK_KEY),
					smsProps.getValue(SMSProperties.DESCRIPTION),
					smsProps.getExtraSettings());
		}
	}

	
	/**
	 * @return a map with SMS or factory configurations, by name
	 */
	private Map<String, StorageDescription> parseStorages(String prefixComponent, 
			String enabledKey, boolean factory) {
		String enabledV = getValue(enabledKey); 
		Set<String> storageIds = getStorageIds(enabledV, prefixComponent, properties);
		
		Map<String, StorageDescription> ret = new HashMap<String, StorageDescription>();
		for(String id: storageIds) {
			String pfx = PREFIX+prefixComponent + id + ".";
			
			SMSProperties smsProps = factory ? new SMSFactoryProperties(pfx, properties) :
				new SMSProperties(pfx, properties);
			
			StorageDescription asd = new StorageDescription(id,
					smsProps.getValue(SMSProperties.NAME),
					smsProps.getValue(SMSProperties.PATH), 
					smsProps.getEnumValue(SMSProperties.TYPE, StorageTypes.class),
					smsProps.getClassValue(SMSProperties.CLASS, StorageManagement.class),
					smsProps.getValue(SMSProperties.PROTOCOLS),
					smsProps.getBooleanValue(SMSProperties.FILTER_LISTING),
					smsProps.getBooleanValue(SMSProperties.CLEANUP),
					smsProps.getBooleanValue(SMSProperties.DISABLE_METADATA),
					smsProps.getValue(SMSProperties.UMASK_KEY),
					smsProps.getValue(SMSProperties.DESCRIPTION),
					smsProps.getExtraSettings()); 
			if (factory) {
					asd.setInfoProviderClass(smsProps.getClassValue(
							SMSFactoryProperties.INFO_PROVIDER, StorageInfoProvider.class));
			} 
			ret.put(id, asd);
		}
		return ret;
	}
	
	/**
	 * @return a list of SMSs/factories configuration ids.
	 */
	public static Set<String> getStorageIds(String enabledTypesV, String prefixComponent, Properties properties) {
		boolean useAll = false;
		Set<String> enabledTypes = null;
		if (enabledTypesV == null)
			useAll = true;
		else {
			String[] enabledTypesA = enabledTypesV.trim().split(" +");
			enabledTypes = new HashSet<String>(enabledTypesA.length);
			Collections.addAll(enabledTypes, enabledTypesA);
		}

		
		String pfx = PREFIX+prefixComponent;
		Set<String> ret = new HashSet<String>();
		for(Object k: properties.keySet()){
			String key=((String)k).trim();
			if (!key.startsWith(pfx) || key.length() < pfx.length())
				continue;
			
			String id = key.substring(pfx.length());
			if (!id.contains("."))
				continue;
			id = id.substring(0, id.indexOf('.'));
			if (id.length() == 0)
				continue;
			
			if (!useAll && !enabledTypes.contains(id)) {
				log.debug("Skipping storage definition " + id + " as it is not enabled.");
				continue;
			}
			ret.add(id);
		}
		return ret;
	}
}
