/*********************************************************************************
 * Copyright (c) 2006 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/

package de.fzj.unicore.uas;

import java.util.Properties;

import org.unigrids.services.atomic.types.ProtocolType;

import de.fzj.unicore.uas.admin.UASAdmin;
import de.fzj.unicore.wsrflite.ContainerProperties;
import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.wsrflite.USEContainer;
import de.fzj.unicore.wsrflite.xmlbeans.registry.RegistryHandler;

/**
 * Main class, intended to run the UAS as a set of standalone services,
 * including a target system factory, job management etc. It is based on the
 * USE hosting environment, and uses the XNJS as back-end execution engine.
 * 
 * @author schuller
 */
public class UAS extends USEContainer {
	// service names
	public static final String TSF = "TargetSystemFactoryService";
	public static final String TSS = "TargetSystemService";
	public static final String JMS = "JobManagement";
	public static final String SMS = "StorageManagement";
	public static final String SMF = "StorageFactory";
	public static final String RESERVATIONS = "ReservationManagement";
	public static final String ENUMERATION = "Enumeration";
	public static final String META = "MetadataManagement";
	public static final String TASK = "Task";
	
	/**
	 * you need to add the protocol to this one to get the right service e.g.
	 * FTS_BASE+"RBYTEIO" = "FileTransferRBYTEIO"
	 */
	public static final String FTS_BASE = "FileTransfer";
	
	/**
	 * If one of the following is used as protocol for file transfers instead of
	 * a {@link ProtocolType} constant, UNICORE should automatically try to
	 * negotiate a suitable file transfer protocol.
	 */
	public static final String[] AUTO_NEGOTIATE_FT_PROTOCOL = {"u6","unicore6","unicore"};

	private UASAdmin jmxBean;
	
	private UASProperties uasProperties;

	/**
	 * @param configFile
	 */
	public UAS(String configFile) throws Exception {
		super(configFile, "UNICORE/X");
		initCommon();
	}

	/**
	 * @param configFile
	 */
	public UAS(Properties properties) throws Exception {
		super(properties, "UNICORE/X");
		initCommon();
	}

	private void initCommon() throws Exception {
		this.uasProperties = new UASProperties(kernel.getContainerProperties().getRawProperties());
		kernel.addConfigurationHandler(UASProperties.class, uasProperties);
		this.jmxBean=new UASAdmin(kernel);
		Kernel.addMBean(jmxBean, "UNICORE/X");
	}
	
	
	public String getConnectionStatus(){
		if (!kernel.getContainerProperties().getBooleanValue(ContainerProperties.ON_STARTUP_SELFTEST))
			return "";
		StringBuilder report = new StringBuilder();
		report.append(super.getConnectionStatus());
		String newline = System.getProperty("line.separator");
		report.append(newline);
		report.append("Registry connection: ");
		RegistryHandler handler = kernel.getAttribute(RegistryHandler.class);
		report.append(handler.getRegistryConnectionStatus());
		report.append(newline);
	
		if (kernel.getHome(UAS.TSS) != null) {
			report.append("TSI connection: ");
			report.append(jmxBean.getTSIConnectionStatus());
			report.append(newline);
		}
		return report.toString();
	}

	public static void main(String[] args) throws Exception {
		try{
			System.out.println("Reading config from " + args[0]);
			UAS uas=new UAS(args[0]);
			uas.startSynchronous();
		}catch(Exception ex){
			ex.printStackTrace();
			System.err.println("ERROR during server startup, server NOT started.");
			System.exit(1);
		}
	}
}
