/*
 * Copyright (c) 2012 ICM Uniwersytet Warszawski All rights reserved.
 * See LICENCE.txt file for licensing information.
 */
package de.fzj.unicore.uas;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;

import de.fzj.unicore.uas.impl.sms.StorageManagementHomeImpl.StorageTypes;
import de.fzj.unicore.xnjs.io.IStorageAdapter;

import eu.unicore.util.Log;
import eu.unicore.util.configuration.ConfigurationException;
import eu.unicore.util.configuration.DocumentationReferenceMeta;
import eu.unicore.util.configuration.PropertiesHelper;
import eu.unicore.util.configuration.PropertyGroupHelper;
import eu.unicore.util.configuration.PropertyMD;

/**
 * This class is used to handle configuration of a particular SMS (typically target system SMS).
 * <p>
 * Note for documentation generation: an artificial prefix must be manually set, as this class is used
 * many times with different prefixed, determined at runtime.
 * 
 * @author K. Benedyczak
 */
public class SMSProperties extends PropertiesHelper {
	
	private static final Logger log = Log.getLogger(Log.SERVICES, SMSProperties.class);

	// Subkeys of general-purpose configuration properties of storages and storage factories.
	public static final String NAME="name";
	public static final String TYPE="type";
	public static final String PATH="path";
	public static final String CLASS="class";
	public static final String PROTOCOLS="protocols";
	
	public static final String FILTER_LISTING="filterFiles";
	public static final String CLEANUP="cleanup";
	public static final String DISABLE_METADATA="disableMetadata";
	public static final String UMASK_KEY = "defaultUmask";
	public static final String DESCRIPTION="description";

	public static final String EXTRA_PREFIX="settings.";

	public static final Pattern umaskPattern = Pattern.compile("[0]?[0-7]?[0-7]?[0-7]");
	
	@DocumentationReferenceMeta
	public static final Map<String, PropertyMD> META = new HashMap<String, PropertyMD>();
	static {
		META.put(NAME, new PropertyMD().
				setDescription("Storage name. If not set then the identifier is used."));
		META.put(TYPE, new PropertyMD(StorageTypes.FIXEDPATH).
				setDescription("Storage type. FIXEDPATH: mapped to a fixed directory, VARIABLE: resolved using an environmental variable lookup, CUSTOM: specified class is used."));
		META.put(PATH, new PropertyMD().
				setDescription("Denotes a storage base path or the name of an environment variable in case of the VARIABLE type."));
		META.put(CLASS, new PropertyMD().setClass(StorageManagement.class).
				setDescription("Storage implementation class used (and mandatory) in case of the CUSTOM type."));
		META.put(PROTOCOLS, new PropertyMD().setUpdateable().
				setDescription("Which protocols to enable, default is defined by the global container setting."));
		META.put(FILTER_LISTING, new PropertyMD("false").setUpdateable().
				setDescription("If set to true then this SMS will filter returned files in response of the ListDirectory command: only files owned or accessible by the caller will be returned."));
		META.put(CLEANUP, new PropertyMD("false").setUpdateable().
				setDescription("Whether files of the storage should be removed when the storage is destroyed. This is mostly useful for storage factories."));
		META.put(DISABLE_METADATA, new PropertyMD("false").
				setDescription("Whether the metadata service should be disabled for this storage."));
		META.put(UMASK_KEY, new PropertyMD(Integer.toOctalString(IStorageAdapter.DEFAULT_UMASK)).
				setDescription("Default (initial) umask for files in the storage. Must be an octal number. Note that this property is not updateable at runtime for normal storages as it wouldn't have sense (it is the initial umask by definition). However in case of storage factory it is, i.e. after the property change, the SMSes created by the factory will use the new umask as the initial one. At runtime the SMS umask can be changed by the clients (if are authorized to do so)."));
		META.put(DESCRIPTION, new PropertyMD("Filesystem").setUpdateable().
				setDescription("Description of the storage. It will be presented to the users."));
		META.put(EXTRA_PREFIX, new PropertyMD().setCanHaveSubkeys().setUpdateable().
				setDescription("Useful for CUSTOM storage types: allows to set additional settings (if needed) by such storages. Please refer to documentation of a particular custom storage type for details. Note that while in general updates of the proprties at runtime are propagated to the chosen implementation, it is up to it to use the updated values or ignore changes."));
	}
	
	public SMSProperties(String prefix, Properties properties) throws ConfigurationException {
		super(prefix, properties, META, log);
	}

	protected SMSProperties(String prefix, Properties properties, Map<String, PropertyMD> meta, Logger log) throws ConfigurationException {
		super(prefix, properties, meta, log);
	}

	@Override
	protected void checkConstraints() throws ConfigurationException {
		super.checkConstraints();
		String type = getValue(TYPE);
		String clazz = getValue(CLASS);
		if (type != null && type.equalsIgnoreCase(StorageTypes.CUSTOM.toString()) && clazz == null)
			throw new ConfigurationException("For the CUSTOM storage type the class must be always set.");
	}
	
	@Override
	protected void checkPropertyConstraints(PropertyMD meta, String key) throws ConfigurationException {
		super.checkPropertyConstraints(meta, key);
		if (key.equals(UMASK_KEY)) {
			String newUmask = getValue(key);
			if (newUmask != null && !umaskPattern.matcher(newUmask).matches())
				throw new ConfigurationException("Specified umask must be an octal number from 0 to 777.");
		}
	}
	
	public Map<String, String> getExtraSettings() {
		PropertyGroupHelper helper = new PropertyGroupHelper(properties, prefix+EXTRA_PREFIX);
		Map<String, String> filteredMap = helper.getFilteredMap();
		Map<String, String> cutMap = new HashMap<String, String>();
		int len = prefix.length()+EXTRA_PREFIX.length();
		for (String key: filteredMap.keySet()) {
			cutMap.put(key.substring(len), filteredMap.get(key));
		}
		return cutMap; 
	}
}
