/*********************************************************************************
 * Copyright (c) 2008 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
 
package de.fzj.unicore.uas.security;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.cert.X509Certificate;

import org.apache.commons.io.FileUtils;
import org.apache.log4j.Logger;
import org.bouncycastle.openssl.PEMWriter;
import org.codehaus.xfire.MessageContext;
import org.codehaus.xfire.client.Client;
import org.codehaus.xfire.handler.AbstractHandler;
import org.codehaus.xfire.handler.Phase;
import org.jdom.Element;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

import eu.emi.security.authn.x509.X509Credential;
import eu.emi.security.authn.x509.impl.CertificateUtils;
import eu.emi.security.authn.x509.proxy.ProxyCertificate;
import eu.emi.security.authn.x509.proxy.ProxyCertificateOptions;
import eu.emi.security.authn.x509.proxy.ProxyGenerator;
import eu.unicore.security.xfireutil.client.Configurable;
import eu.unicore.security.xfireutil.client.DSigOutHandler;
import eu.unicore.util.Log;
import eu.unicore.util.httpclient.IClientConfiguration;

/**
 * If needed then a token with a proxy cert in PEM format is inserted into the header.<br/>
 * The properties of the issued proxy can be configured in the security properties passed
 * to the doInit() method: unicore.proxy.{lifetime|keysize} <br/>
 * <p>
 * The handler will regenerate a proxy when half of lifetime of the previous one has passed.
 * <b>This handler is intended for client use</b><br/>
 * 
 * @author schuller
 * @author golbi
 */
public class ProxyCertOutHandler extends AbstractHandler implements Configurable {
	
	private static final Logger logger = Log.getLogger(Log.SECURITY,ProxyCertOutHandler.class);
	
	protected IClientConfiguration sec;
	
	private Element proxyHeaderAsJDOM=null;
	

	/**
	 * property for defining the lifetime in seconds (default: 2*3600, i.e. two hours)
	 */
	public static final String PROXY_LIFETIME="unicore.proxy.lifetime";
	
	/**
	 * property for defining the keysize (default: 1024)
	 */
	public static final String PROXY_KEYSIZE="unicore.proxy.keysize";
	
	/**
	 * property defining an existing proxy file to be used
	 */
	public static final String PROXY_FILE="unicore.proxy.file";
	
	private long expiryInstant;
	
	private String pem;
	
	public ProxyCertOutHandler() {
		super();
		setPhase(Phase.POST_INVOKE);
		before(DSigOutHandler.class.getName());
	}


	public synchronized void configure(IClientConfiguration sec) {
		this.sec = sec;
		pem=null;
		try{
			if(sec.getExtraSettings().getProperty(PROXY_FILE)!=null){
				pem=readProxyFromFile(sec.getExtraSettings().getProperty(PROXY_FILE));
			}
			else{
				pem=generateProxy();
			}

			StringBuilder sb=new StringBuilder();
			sb.append("<proxy:Proxy xmlns:proxy=\"http://www.unicore.eu/unicore6\">");
			sb.append(pem);
			sb.append("</proxy:Proxy>");
			proxyHeaderAsJDOM=new SAXBuilder().build(new ByteArrayInputStream(
					sb.toString().getBytes())).detachRootElement();
		}
		catch(Exception ce){
			logger.error("Can't create Proxy header: " , ce);
			return;
		}
		
		if(logger.isDebugEnabled()){
			logger.debug("(Re-)initialised Proxy Outhandler");
			try{
				ByteArrayOutputStream bos=new ByteArrayOutputStream();
				new XMLOutputter(Format.getPrettyFormat()).output(proxyHeaderAsJDOM, bos);
				logger.debug(bos.toString());
			}catch(Exception e){
				logger.warn("",e);
			}
		}
	}
	
	protected String readProxyFromFile(String fileName) throws IOException{
		InputStream is=new FileInputStream(fileName);
		try{
			KeyStore ks=CertificateUtils.loadPEMKeystore(is,(char[])null,"none".toCharArray());
			X509Certificate x509 =(X509Certificate)ks.getCertificate("default");
			logger.info("Read proxy from '"+fileName+"' valid till "+x509.getNotAfter());
			expiryInstant=x509.getNotAfter().getTime() - (ProxyCertificateOptions.DEFAULT_LIFETIME/2)*1000;
			return FileUtils.readFileToString(new File(fileName));
		}
		catch(KeyStoreException ex){
			throw new IOException(ex);
		}
		finally{
			is.close();
		}
		
	}
	
	/**
	 * @return pem-encoded proxy or <code>null</code> if not created
	 */
	protected String generateProxy()throws Exception{
		X509Credential credential = sec.getCredential();
		ProxyCertificateOptions param = new ProxyCertificateOptions(credential.getCertificateChain());
		String lifetimeP=sec.getExtraSettings().getProperty(PROXY_LIFETIME);
		if(lifetimeP!=null)
			param.setLifetime(Integer.parseInt(lifetimeP));
		String keysizeP=sec.getExtraSettings().getProperty(PROXY_KEYSIZE);
		if(keysizeP!=null)
			param.setKeyLength(Integer.parseInt(keysizeP));	
		
		ProxyCertificate proxy = ProxyGenerator.generate(param, credential.getKey());

		int lifetime = param.getLifetime();
		expiryInstant=proxy.getCertificateChain()[0].getNotAfter().getTime() - (lifetime/2)*1000;
		
		ByteArrayOutputStream os = new ByteArrayOutputStream(10240);
		OutputStreamWriter ow=new OutputStreamWriter(os);
		PEMWriter pw=new PEMWriter(ow);
		pw.writeObject(proxy.getCertificateChain()[0]);
		pw.writeObject(proxy.getPrivateKey());
		pw.writeObject(credential.getCertificate());
		pw.flush();
		ow.close();
		return os.toString("US-ASCII");
	}

	public Element getProxyHeader() {
		return proxyHeaderAsJDOM;
	}
	
	public synchronized void invoke(MessageContext context) throws Exception {
		if (proxyHeaderAsJDOM == null){
			return;
		}
		
		//do nothing if not client call
		Boolean clientMode = (Boolean) context.getProperty(Client.CLIENT_MODE);
		if (clientMode == null || !clientMode.booleanValue())
			return;

		if(System.currentTimeMillis()>expiryInstant){
			configure(sec);
		}
		Element h = context.getOutMessage().getOrCreateHeader();
		h.addContent((Element)proxyHeaderAsJDOM.clone());
	}
	
	// testing use only
	String getPem(){
		return pem;
	}
}


