/*
 * Copyright 2006 University of Virginia
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy
 * of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package de.fzj.unicore.uas.rns;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.Stack;

import org.apache.log4j.Logger;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import eu.unicore.util.Log;

/**
 * The RNSPath class is the main client side interface between developers and
 * the grid directory structure. Nearly all directory related operations
 * (including creating files and destroying instances), should be handled
 * through instances of this class.
 * 
 * @author Mark Morgan
 * @author Bastian Demuth
 */
public class RNSPath implements Serializable, Cloneable
{
	
	private static final Logger logger = Log.getLogger(Log.CLIENT,RNSPath.class);
	
	public static final int DONT_CARE = 0;
	public static final int MUST_EXIST = 1;
	public static final int MUST_NOT_EXIST = 2;
	
	
	private RNSPath parent;
	private String name;
	private EndpointReferenceType cachedEPR;
	
	
	/**
	 * Get the EPR of the current entry if it exists.  If the entry doesn't
	 * exist, this method throws an exception.
	 * 
	 * @return The EPR of this entry if it exists.
	 * 
	 * @throws RNSPathDoesNotExistException
	 */
	public EndpointReferenceType getEndpointReference()
		throws RNSEntryDoesNotExistFault
	{
		return resolveRequired();
	}
	
	
	private EndpointReferenceType resolveRequired()
		throws RNSEntryDoesNotExistFault
	{
		EndpointReferenceType epr = resolveOptional();
		if (epr == null)
			throw new RNSEntryDoesNotExistFault(toString());
		
		return epr;
	}
	
	private EndpointReferenceType resolveOptional()
	{
		return null;
	}
	
	/**
	 * Construct a new RNS path.
	 * 
	 * @param parent The RNSPath for the parent under which this entry exists.
	 * @param nameFromParent The name that this entry has inside of the parent
	 * directory.
	 * @param cachedEPR The EPR of this entry (null if unknown).
	 * entry if we don't have any EPR yet.
	 */
	public RNSPath(RNSPath parent, String nameFromParent, 
		EndpointReferenceType cachedEPR)
	{
		this.parent = parent;
		this.name = nameFromParent;
		this.cachedEPR = cachedEPR;
		
		if ((parent == null && nameFromParent != null) || 
			(parent != null && nameFromParent == null))
		{
			throw new IllegalArgumentException(
				"The parent and the nameFromParent parameters must either " +
				"both be null, or both be non-null.");
		}
		
		if (parent == null && cachedEPR == null)
			throw new IllegalArgumentException("Cannot have a null EPR for the root.");
		
	}
	
	/**
	 * Create a new RNSPath which represents a new rooted RNS namespace
	 * at the given EPR.
	 * 
	 * @param root The EPR which represents the root of this new namespace.
	 */
	public RNSPath(EndpointReferenceType root)
	{
		this(null, null, root);
	}
	
	
	/**
	 * Retrieve the name of this entry as represented by the parent
	 * directory.
	 * 
	 * @return The name of this RNS entry.
	 */
	public String getName()
	{
		if (name == null)
			return "/";
		
		return name;
	}
	
	
	/**
	 * Return the full path to this entry starting at the root of the
	 * namespace.
	 * 
	 * @return A slash separated string representing the full grid path
	 * to this entry.
	 */
	public String toString()
	{
		if (parent == null)
			return "/";
		
		String s = parent.toString();
		if (s == "/")
			return s + name;
		
		return s + "/" + name;
	}
	
	/**
	 * Test to see if two RNS paths are equal.  This comparison is for String
	 * path representation only (no comparison of EPRs or other metadata is
	 * done).
	 * 
	 * @param other The other path to compare against.
	 * @return True if the two paths are equal, false otherwise.
	 */
	public boolean equals(RNSPath other)
	{
		if (other instanceof RNSPath)
		{
			return toString().equals(other.toString());
		}
		else return false;
	}
	
	
	
	/**
	 * Calculate a hashcode for the path represented by this RNS path.
	 */
	@Override
	public int hashCode()
	{
		return toString().hashCode();
	}
	
	/**
	 * Retrieve the root RNSPath entry for this namespace.
	 * 
	 * @return The root RNSPath entry for this namespace.
	 */
	public RNSPath getRoot()
	{
		if (parent == null)
			return this;
		
		return parent.getRoot();
	}
	
	/**
	 * Retrieve the parent RNSPath entry for this entry.
	 * 
	 * @return The parent RNSPath entry for this entry.
	 */
	public RNSPath getParent()
	{
		if (parent == null)
			return this;
		
		return parent;
	}
	
	/**
	 * Determines if this RNSPath entry represents the root of a namespace.
	 * 
	 * @return True if this entry is the root of a namespace, false otherwise.
	 */
	public boolean isRoot()
	{
		return parent == null;
	}
	
	/**
	 * Determine if this entry represents a grid resource that exists (has
	 * an EPR) or doesn't.
	 * 
	 * @return True if this entry has an EPR, false otherwise.
	 */
	public boolean exists()
	{
		return resolveOptional() != null;
	}
	
	/**
	 * Assuming that this entry doesn't exist, this method creates a new
	 * directory at the indicated path.
	 * 
	 * @throws RNSFault
	 * @throws RNSEntryExistsFault
	 * @throws RNSEntryDoesNotExistFault
	 */
	public void mkdir()
		throws RNSEntryDoesNotExistFault, 
			RNSEntryExistsFault,
			RNSFault
	{
		if (exists())
			throw new RNSEntryExistsFault(toString());
		
		if (parent == null)
			throw new RNSFault(
				"Someone tried to create the root directory, " +
				"which can't be done.");
		
		EndpointReferenceType parentEndpoint = parent.resolveRequired();
		
	}
	
	
	
	/**
	 * Similar to mkdir(), but this operation creates all directories that
	 * don't exist in the indicated path, including this one.
	 * 
	 * @throws RNSFault
	 * @throws RNSEntryExistsFault
	 * @throws RNSEntryDoesNotExistFault
	 */
	public void mkdirs()
		throws RNSEntryDoesNotExistFault, 
		RNSEntryExistsFault, RNSFault
	{
		if (exists())
			throw new RNSEntryExistsFault(toString());
		
		if (parent == null)
			throw new RNSFault(
				"Someone tried to create the root directory, " +
				"which can't be done.");
		
		if (!parent.exists())
			parent.mkdirs();
		
		mkdir();
	}
	
	/**
	 * Creates a new ByteIO file at this path.  This operation assumes that
	 * the current path doesn't yet exist.  The type of ByteIO created can
	 * be any valid ByteIO implementation that the parent directory decides
	 * to create.
	 * 
	 * @return The EPR of a newly created ByteIO file.
	 * 
	 * @throws RNSEntryExistsFault
	 * @throws RNSEntryDoesNotExistFault
	 * @throws RNSFault
	 */
	public EndpointReferenceType createNewFile()
		throws RNSEntryExistsFault, RNSEntryDoesNotExistFault,
			RNSFault
	{
		throw new RNSFault("Unable to create new file: NOT IMPLEMENTED");
//		if (exists())
//			throw new RNSEntryExistsFault(pwd());
//		
//		if (_parent == null)
//			throw new RNSFault(
//				"Someone tried to create a file as the root directory.");
//		
//		EndpointReferenceType parentEPR = _parent.resolveRequired();
//		
//		RNSLegacyProxy proxy = new RNSLegacyProxy(createProxy(parentEPR,
//			EnhancedRNSPortType.class));
//		try
//		{
//			_cachedEPR = proxy.createFile(_nameFromParent);
//			_attemptedResolve = true;
//			
//			return _cachedEPR;
//		}
//		catch (RemoteException re)
//		{
//			throw new RNSFault("Unable to create new file.", re);
//		}
	}
	
	private void arrayify(ArrayList<RNSPath> rep)
	{
		if (parent != null)
			parent.arrayify(rep);
		rep.add(this);
	}
	
	/**
	 * Lookup an RNSPath based off of this path.  This path can be relative
	 * or absolute compared to this path.  The path does not have to exist
	 * either.  If the indicated path does not exist, an RNSPath entry with
	 * no EPR will be returned.
	 * 
	 * @param path The relative or absolute path to lookup.
	 * 
	 * @return An RNSPath entry representing the path looked up.
	 */
	public RNSPath lookup(String path)
	{
		try
		{
			return lookup(path, DONT_CARE);
		}
		catch (RNSEntryDoesNotExistFault rpdnee)
		{
			logger.error("This exception shouldn't have happened.", rpdnee);
		}
		catch (RNSEntryExistsFault rpdnee)
		{
			logger.error("This exception shouldn't have happened.", rpdnee);
		}
		
		return null;
	}
	
	
	
	
	
	
	
	static private String formPath(String []pathElements)
	{
		StringBuilder builder = new StringBuilder();
		for (String element : pathElements)
			builder.append(String.format("/%s", element));
		
		if (builder.length() == 0)
			builder.append("/");
		
		return builder.toString();
	}
	
	/**
	 * Similar to lookup above, this operation looks up a new RNSPath entry at
	 * a given query path.  This operation however takes a flag which can
	 * specify whether or not to return directories that don't actually
	 * exist.
	 * 
	 * @param path The RNS path to lookup.
	 * @param queryFlag A flag indicating whether or not to fault if the
	 * entry exists/does not exist.
	 * 
	 * @return The RNSPath entry that was found/indicated.
	 * @throws RNSEntryDoesNotExistFault
	 * @throws RNSEntryExistsFault
	 */
	public RNSPath lookup(String path, int queryFlag)
			throws RNSEntryDoesNotExistFault, RNSEntryExistsFault
	{
		if (path == null)
			throw new IllegalArgumentException(
				"Cannot lookup a path which is null.");
		
		String[] pathElements = normalizePath(toString(), path);
		String fullPath = formPath(pathElements);
		RNSPath ret = null;
		
		ArrayList<RNSPath> arrayRep = new ArrayList<RNSPath>();
		arrayify(arrayRep);
		
		int lcv = 0;
		while (true)
		{
			if (lcv >= pathElements.length)
				break;
			if (lcv + 1 >= arrayRep.size())
				break;
			if (!pathElements[lcv].equals(arrayRep.get(lcv + 1).name))
				break;
			lcv++;
		}
		
		if (lcv >= pathElements.length)
		{
			// We completely matched a portion of the original path
			ret = arrayRep.get(lcv);
			return ret;
		}
		
		RNSPath next = arrayRep.get(lcv);
		for (;lcv < pathElements.length; lcv++)
		{
//			next = new RNSPath(next, pathElements[lcv], null, false);
		}
		
		if (isFlagSet(queryFlag,MUST_EXIST))
		{
			if (!next.exists())
				throw new RNSEntryDoesNotExistFault(next.toString());
		}
		else if (isFlagSet(queryFlag,MUST_NOT_EXIST))
		{
			if (next.exists())
				throw new RNSEntryExistsFault(next.toString());
		}
		
		return next;
	}
	
	public static boolean isFlagSet(int value, int flag)
	{
		return (flag & value) != 0;
	}
	
	
	static public String[] normalizePath(String currentPath, String path)
	{
		String fullPath;
		
		if (path.startsWith("/"))
			fullPath = path;
		else
			fullPath = currentPath + "/" + path;
		
		return normalizePath(fullPath);
		
	}
	
	static public String[] normalizePath(String path)
	{
		Stack<String> nPath = new Stack<String>();
		String []ret = path.split("/");
		for (String s : ret)
		{
			if (s == null || s.length() == 0)
				continue;
			
			if (s.equals(".."))
			{
				if (!nPath.empty())
					nPath.pop();
			} else if (!s.equals("."))
			{
				nPath.push(s);
			}
		}
		
		ret = new String[nPath.size()];
		nPath.toArray(ret);
		return ret;
	}
	
	private Collection<RNSPath> expand(
		RNSPath parent, String []pathElements, int nextElement,
		FilterFactory filterType)
	{
		Collection<RNSPath> ret = new LinkedList<RNSPath>();
//		Collection<RNSPath> tmp;
//			
//		if (nextElement >= pathElements.length)
//			ret.add(parent);
//		else
//		{
//			Filter filter = filterType.createFilter(pathElements[nextElement]);
//			try
//			{
//				TypeInformation typeInfo = new TypeInformation(
//					parent.getEndpoint());
//				if (typeInfo.isRNS())
//				{
//					_logger.debug("Attempting to list contents of \"" + parent + "\".");
//					
//					for (RNSPath candidate : parent.listContents())
//					{
//						if (filter.matches(candidate.getName()))
//						{
//							tmp = expand(candidate, pathElements, 
//								nextElement + 1, filterType);
//							if (tmp != null)
//								ret.addAll(tmp);
//						}
//					}
//				}
//			}
//			catch (RNSFault rne)
//			{
//				_logger.debug(
//					"Skipping a directory in an RSNPath expansion which " +
//					"can't be expanded.", rne);
//			}
//		}
		
		if (ret.size() == 0)
			return null;
		
		return ret;
	}

	/**
	 * A utility operation that looks up a path expression and returns exactly
	 * one matching path entry.  If more then one path entry matching the 
	 * pathExpression, an exception is thrown.
	 * 
	 * @param pathExpression A path expression which is to be looked up.  This
	 * path expression can contain standard file system globbing patterns such
	 * as *.
	 * 
	 * @return The resultant RNSPath entry (if any).
	 * @throws RNSMultiLookupResultException
	 */
//	public RNSPath expandSingleton(String pathExpression)
//		throws RNSMultiLookupResultException
//	{
//		return expandSingleton(pathExpression, null);
//	}
	
	/**
	 * This operation looks up pathExpressions and returns the exact matching
	 * entry that is found if any.
	 * 
	 * @param pathExpression The path expression to lookup.  This expression
	 * will be matched against the filterType indicated.
	 * @param filterType A filter which figures out how to expand the
	 * pathExpression language given.  Two pathExpressio filterTypes are
	 * available by default -- one parses file globbing patterns, the other
	 * parses Regular Expressions.
	 * 
	 * @return The matched RNSPath entry.
	 * @throws RNSMultiLookupResultException
	 */
//	public RNSPath expandSingleton(String pathExpression, 
//		FilterFactory filterType) throws RNSMultiLookupResultException
//	{
//		Collection<RNSPath> ret = expand(pathExpression, filterType);
//		
//		if (ret.size() < 1)
//			return null;
//		else if (ret.size() > 1)
//			throw new RNSMultiLookupResultException(pathExpression);
//		
//		return ret.iterator().next();
//	}
	
	/**
	 * Similar to expandSingleton above, but this version of the operation
	 * matches 0 or more entries.
	 * 
	 * @param pathExpression The file pattern globbing path expression to
	 * lookup.
	 * 
	 * @return A collection of zero or more RNSPath entries that matched
	 * the query.
	 */
	public Collection<RNSPath> expand(String pathExpression)
	{
		return null;
//		return expand(pathExpression, new FilePatternFilterFactory());
	}
	
	/**
	 * Similar to the expandSingleton operation above except that this
	 * version of the operation can return 0 or more entries that match
	 * the path query.
	 * 
	 * @param pathExpression The path expression to lookup.
	 * @param filterType The file pattern matcher to use.
	 * 
	 * @return The RNSPath entries that matched the query.
	 */
//	public Collection<RNSPath> expand(String pathExpression,
//		FilterFactory filterType)
//	{
//		if (pathExpression == null)
//			throw new IllegalArgumentException(
//				"Cannot lookup a path which is null.");
//		
//		if (filterType == null)
//			filterType = new FilePatternFilterFactory();
//		
//		try
//		{
//			String[] pathElements = PathUtils.normalizePath(pwd(), 
//				pathExpression);
//			
//			Collection<RNSPath> ret = expand(getRoot(), pathElements, 
//				0, filterType);
//			if (ret == null)
//			{
//				ret = new ArrayList<RNSPath>(1);
//				ret.add(lookup(pathExpression, 
//					DONT_CARE));
//			}
//			
//			return ret;
//		}
//		catch (RNSFault rne)
//		{
//			throw new RuntimeException(
//				"Unexpected RNS path expansion exception.", rne);
//		}
//	}
	
	/**
	 * List the contents of the current RNS directory that match the given
	 * filter.
	 * 
	 * @param filter A filter that is used to select entries in the current
	 * RNS directory.
	 * 
	 * @return The set of all RNS entries in the current directory that matched
	 * the given pattern.
	 * 
	 * @throws RNSEntryDoesNotExistFault
	 * @throws RNSFault
	 */
//	public Collection<RNSPath> listContents(RNSFilter filter)
//		throws RNSEntryDoesNotExistFault, RNSFault
//	{
//		Collection<RNSPath> ret = new LinkedList<RNSPath>();
//		
//		for (RNSPath path : listContents())
//		{
//			if (filter.matches(path))
//				ret.add(path);
//		}
//		
//		return ret;
//	}
	
	/**
	 * List all of the entries in the given RNS directory.
	 * 
	 * @return The set of all RNSPath entries contained in this directory.
	 * 
	 * @throws RNSEntryDoesNotExistFault
	 * @throws RNSFault
	 */
	public Collection<RNSPath> listContents()
		throws RNSEntryDoesNotExistFault, RNSFault
	{
		throw new RNSFault("NOT IMPLEMENTED");
//		EndpointReferenceType me = resolveRequired();
//		EnhancedRNSPortType rpt = createProxy(
//			me, EnhancedRNSPortType.class);
//		RNSLegacyProxy proxy = new RNSLegacyProxy(rpt);
//		RNSIterable entries = null;
//		
//		try
//		{
//			entries = proxy.iterateList();
//			LinkedList<RNSPath> ret = new LinkedList<RNSPath>();
//			for (RNSEntryResponseType entry : entries)
//			{
//				RNSPath newEntry = new RNSPath(this, entry.getEntryName(),
//					entry.getEndpoint(), true);
//				_lookupCache.put(newEntry.pwd(), newEntry);
//				ret.add(newEntry);
//			}
//			
//			return ret;
//		}
//		catch (GenesisIISecurityException gse)
//		{
//			throw new RNSFault("Unable to list contents -- " +
//				"security exception.", gse);
//		}
//		catch (RemoteException re)
//		{
//			throw new RNSFault("Unable to list contents.", re);
//		}
	}
	
	/**
	 * Assuming that the indicated RNSPath entry does not yet exist (but that
	 * it's parent does), links the given EPR into this named entry.
	 * 
	 * @param epr The EPR to link to this indicated name.
	 * 
	 * @throws RNSEntryExistsFault
	 * @throws RNSEntryDoesNotExistFault
	 * @throws RNSFault
	 */
	public void link(EndpointReferenceType epr)
		throws RNSEntryExistsFault, RNSEntryDoesNotExistFault,
			RNSFault
	{
//		if (exists())
//			throw new RNSEntryExistsFault(pwd());
//		
//		if (_parent == null)
//			throw new RNSFault(
//				"Someone tried to link the root directory, " +
//				"which can't be done.");
//		
//		EndpointReferenceType parentEPR = _parent.resolveRequired();
//		
//		RNSLegacyProxy proxy = new RNSLegacyProxy(
//			createProxy(parentEPR, EnhancedRNSPortType.class));
//		try
//		{
//			_cachedEPR = proxy.add(_nameFromParent, epr);
//			_attemptedResolve = true;
//		}
//		catch (RemoteException re)
//		{
//			throw new RNSFault("Unable to link in new EPR.", re);
//		}
	}
	
	/**
	 * Unlink this RNSPath entry from the namespace (this will never
	 * destroy the target resource -- it merely unlinks it from the
	 * filesystem).
	 * 
	 * @throws RNSEntryDoesNotExistFault
	 * @throws RNSFault
	 */
	public void unlink()
		throws RNSEntryDoesNotExistFault, RNSFault
	{
//		if (!exists())
//			throw new RNSEntryDoesNotExistFault(pwd());
//		
//		if (_parent == null)
//			throw new RNSFault("Attempt to unlink root not allowed.");
//		
//		EndpointReferenceType parentEPR = _parent.resolveRequired();
//		
//		EnhancedRNSPortType rpt = createProxy(parentEPR, EnhancedRNSPortType.class);
//		RNSLegacyProxy proxy = new RNSLegacyProxy(rpt);
//		try
//		{
//			proxy.remove(_nameFromParent);
//			_cachedEPR = null;
//			_attemptedResolve = true;
//		}
//		catch (RemoteException re)
//		{
//			throw new RNSFault("Unable to unlink entry.", re);
//		}
	}
	
	/**
	 * Similar to unlink above, but this operation also destroy the target
	 * resource if at all possible.
	 * 
	 * @throws RNSEntryDoesNotExistFault
	 * @throws RNSFault
	 */
	public void delete()
		throws RNSEntryDoesNotExistFault, RNSFault
	{
//		if (!exists())
//			throw new RNSEntryDoesNotExistFault(pwd());
//		
//		if (_parent == null)
//			throw new RNSFault("Attempt to unlink root not allowed.");
//		
//		EndpointReferenceType parentEPR = _parent.resolveRequired();
//		try
//		{
//			if (EPRUtils.isCommunicable(_cachedEPR))
//			{
//				GeniiCommon common = createProxy(_cachedEPR, GeniiCommon.class);
//				common.destroy(new Destroy());
//			}
//			
//			EnhancedRNSPortType rpt = createProxy(
//				parentEPR, EnhancedRNSPortType.class);
//			RNSLegacyProxy proxy = new RNSLegacyProxy(rpt);
//			proxy.remove(_nameFromParent);
//			_cachedEPR = null;
//			_attemptedResolve = true;
//		}
//		catch (RemoteException re)
//		{
//			throw new RNSFault("Unable to delete entry.", re);
//		}
	}
	

	
	
}