package de.fzj.unicore.uas.client;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.unigrids.services.atomic.types.ProtocolType;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.fts.FiletransferOptions;
import de.fzj.unicore.uas.fts.ProgressListener;
import eu.unicore.uftp.client.UFTPClient;
import eu.unicore.uftp.client.UFTPProgressListener;
import eu.unicore.uftp.dpc.Utils;
import eu.unicore.util.httpclient.IClientConfiguration;

/**
 * UFTP file transfer client. It connects to an UFTP file transfer instance and 
 * is used to download or upload data. 
 *  
 * @author schuller
 */
public class UFTPFileTransferClient extends FileTransferClient 
implements Configurable, UFTPConstants, FiletransferOptions.IMonitorable, UFTPProgressListener{

	private String secret;

	private final InetAddress[] serverHosts;
	private final int serverPort;
	private final int streams;
	private final byte[] key;
	
	private ProgressListener<Long>listener;
	
	public UFTPFileTransferClient(EndpointReferenceType epr,
			IClientConfiguration sec) throws Exception {
		this(epr.getAddress().getStringValue(),epr, sec);
	}

	public UFTPFileTransferClient(String url, EndpointReferenceType epr,
			IClientConfiguration sec) throws Exception {
		super(url, epr, sec);
		Map<String,String>params=getProtocolDependentRPs();
		//server host is given in file transfer RP
		String hostParam=params.get(PARAM_SERVER_HOST);
		//but we allow the client to override it
		hostParam=sec.getExtraSettings().getProperty(PARAM_SERVER_HOST, hostParam);
		serverHosts=asHosts(hostParam);
		if(serverHosts.length==0){
			throw new Exception("No usable UFTP server host could be determined from <"+hostParam+">");
		}
		serverPort=Integer.parseInt(params.get(PARAM_SERVER_PORT));
		streams=Integer.parseInt(params.get(PARAM_STREAMS));
		String keySpec=params.get(PARAM_ENCRYPTION_KEY);
		key= keySpec!=null ? Utils.decodeBase64(keySpec) : null;
	}

	public void configure(Map<String,String>params){
		String secret=params.get(UFTPConstants.PARAM_SECRET);
		if(secret!=null)setSecret(secret);
	}
	
	public void setSecret(String secret){
		this.secret=secret;
	}
	
	@Override
	public void readAllData(OutputStream target) throws Exception {
		UFTPClient c=new UFTPClient(serverHosts,serverPort,target);
		c.setNumConnections(streams);
		c.setSecret(secret);
		c.setKey(key);
		if(listener!=null)c.setProgressListener(this);
		c.run();
	}

	@Override
	public void writeAllData(InputStream source) throws Exception {	
		UFTPClient c=new UFTPClient(serverHosts,serverPort,source);
		c.setNumConnections(streams);
		c.setSecret(secret);
		c.setKey(key);
		if(listener!=null)c.setProgressListener(this);
		c.run();
	}

	
	@Override
	public void setProgressListener(ProgressListener<Long> listener) {
		this.listener=listener;
	}

	/**
	 * convenience method to create a UFTP file import
	 * @param path - the file name
	 * @param sms -  the storage
	 * @param append - whether to append data
	 * @param clientHost - the client host (name or IP)
	 * @param numConnections - the requested number of connctions
	 * @oaram secret - the authz token that the server should use to authorize this transfer
	 * @param encrypt - if <code>true</code> data will be encrypted
	 * @return UFTP client
	 */
	public static UFTPFileTransferClient createImport(String path, StorageClient sms, boolean append,
			String clientHost, int numConnections, String secret, boolean encrypt)throws Exception{
		UFTPFileTransferClient c=(UFTPFileTransferClient)sms.getImport(path, append, 
				makeParams(clientHost, numConnections, secret, encrypt), ProtocolType.UFTP);
		c.secret=secret;
		return c;
	}

	/**
	 * convenience method to create a UFTP file export
	 * @param path - the file name
	 * @param sms -  the storage
	 * @param clientHost - the client host (name or IP)
	 * @param numConnections - the requested number of connctions
	 * @param encrypt - if <code>true</code> data will be encrypted
	 * @return UFTP client
	 */
	public static UFTPFileTransferClient createExport(String path, StorageClient sms, 
			String clientHost, int numConnections, String secret, boolean encrypt)throws Exception{
		UFTPFileTransferClient c=(UFTPFileTransferClient)sms.getExport(path, 
				makeParams(clientHost, numConnections, secret, encrypt), ProtocolType.UFTP);
		c.secret=secret;
		return c;
	}

	
	public InetAddress[] getServerHosts() {
		return serverHosts;
	}

	public int getServerPort() {
		return serverPort;
	}

	public int getStreams() {
		return streams;
	}

	/**
	 * the base64 encoded encryption key, or <code>null</code> if no encryption
	 * @return
	 */
	public String getEncryptionKey() {
		return key!=null? Utils.encodeBase64(key) : null;
	}
	
	private long lastTotal=0;
	
	@Override
	public void notifyTotalBytesTransferred(long totalBytesTransferred) {
		if(listener!=null){
			//need to calculate difference to last call
			listener.notifyProgress(totalBytesTransferred-lastTotal);
			lastTotal=totalBytesTransferred;
		}
	}

	private static Map<String,String>makeParams(String host, int numConnections, String secret, boolean encrypt){
		Map<String,String>p=new HashMap<String, String>();
		p.put(PARAM_CLIENT_HOST,host);
		p.put(PARAM_STREAMS,String.valueOf(numConnections));
		p.put(PARAM_SECRET,secret);
		p.put(PARAM_ENABLE_ENCRYPTION, String.valueOf(encrypt));
		return p;
	}

	public InetAddress[]asHosts(String hostsProperty){
		List<InetAddress>hostList=new ArrayList<InetAddress>();
		String[] hosts=hostsProperty.split("[ ,]+");
		for(String h: hosts){
			try{
				hostList.add(InetAddress.getByName(h));
			}catch(IOException io){
				logger.trace("Un-usable UFTP host address <"+h+"> : "+io);
			}
		}
		return hostList.toArray(new InetAddress[hostList.size()]);
	}
	
	public String asString(InetAddress[] ips){
		StringBuilder sb=new StringBuilder();
		for(InetAddress ip: ips){
			if(sb.length()>0)sb.append(',');
			sb.append(ip.getHostName());
		}
		return sb.toString();
	}
}
