package de.fzj.unicore.uas.client;

import java.util.Arrays;
import java.util.Calendar;
import java.util.List;

import org.apache.log4j.Logger;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.OperatingSystemType;
import org.oasisOpen.docs.wsrf.rl2.TerminationTimeDocument.TerminationTime;
import org.unigrids.services.atomic.types.SecurityType;
import org.unigrids.x2006.x04.services.tsf.CreateTSRDocument;
import org.unigrids.x2006.x04.services.tsf.CreateTSRResponseDocument;
import org.unigrids.x2006.x04.services.tsf.TargetSystemFactoryPropertiesDocument;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.TargetSystemFactory;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import eu.unicore.util.Log;
import eu.unicore.util.httpclient.IClientConfiguration;

/**
 * Client for the TargetSystemFactory service
 * 
 * @author schuller
 */
public class TSFClient extends BaseUASClient {

	private static final Logger logger=Log.getLogger(Log.CLIENT,TSFClient.class);

	private final TargetSystemFactory tsf;

	public TSFClient(String endpointUrl, EndpointReferenceType epr, IClientConfiguration sec)throws Exception {
		super(endpointUrl, epr,sec);
		tsf=makeProxy(TargetSystemFactory.class);
		setRetryDefaults();
	}

	public TSFClient(EndpointReferenceType epr, IClientConfiguration sec)throws Exception {
		this(epr.getAddress().getStringValue(), epr,sec);
	}

	public TSSClient createTSS(CreateTSRDocument in) throws Exception {
		logger.info("Calling target system factory service at: "+getEPR().getAddress().getStringValue());
		CreateTSRResponseDocument res=tsf.CreateTSR(in);
		EndpointReferenceType epr=res.getCreateTSRResponse().getTsrReference();
		return new TSSClient(epr.getAddress().getStringValue(),epr, getSecurityConfiguration());
	}

	/**
	 * create a TSS with the supplied initial termination time
	 * 
	 * @param initialTerminationTime
	 * @return
	 * @throws BaseFault
	 */
	public TSSClient createTSS(Calendar initialTerminationTime) throws Exception {
		CreateTSRDocument in=CreateTSRDocument.Factory.newInstance();
		TerminationTime tt=TerminationTime.Factory.newInstance();
		tt.setCalendarValue(initialTerminationTime);
		in.addNewCreateTSR().setTerminationTime(tt);
		return createTSS(in);
	}

	public TSSClient createTSS() throws Exception {
		CreateTSRDocument in=CreateTSRDocument.Factory.newInstance();
		in.addNewCreateTSR();
		return createTSS(in);
	}

	/**
	 * returns the service's TargetSystemPropertiesDocument
	 * @param endpointUrl
	 * @param epr
	 */
	public TargetSystemFactoryPropertiesDocument getResourcePropertiesDocument()throws Exception{
		return TargetSystemFactoryPropertiesDocument.Factory.parse(GetResourcePropertyDocument().getGetResourcePropertyDocumentResponse().newInputStream());
	}

	public boolean supportsReservation()throws Exception{
		return getResourcePropertiesDocument().getTargetSystemFactoryProperties().getSupportsReservation();
	}

	public boolean supportsVirtualImages()throws Exception{
		return getResourcePropertiesDocument().getTargetSystemFactoryProperties().getSupportsVirtualImages();
	}
	
	public List<EndpointReferenceType> getTargetSystems(){
		try{
			EndpointReferenceType[] eprs=getResourcePropertiesDocument().getTargetSystemFactoryProperties().
			getTargetSystemReferenceArray();
			return Arrays.asList(eprs);
		}catch(Exception e){
			logger.error("Can't get target system list.",e);
		}
		return null;
	}


	public OperatingSystemType[] getOperatingSystems() throws Exception{
		return getResourcePropertiesDocument().getTargetSystemFactoryProperties().getOperatingSystemArray();
	}

	/**
	 * get the user's logins on the remote machine
	 * @return String[] or null if an error occured
	 */
	public String[] getXlogins(){
		try{
			SecurityType secT=getSecurityInfo();
			if(secT==null){
				//pre 6.5.0 server
				return getResourcePropertiesDocument().getTargetSystemFactoryProperties().getXloginArray();
			}
			else{
				return super.getXlogins();
			}
		}catch(Exception e){
			logger.error("Can't get xlogins.",e);
		}
		return null;
	}

	/**
	 * get the user's groups on the remote machine
	 * @return String[] or null if an error occured
	 */
	public String[] getXgroups(){
		try{
			String[] srs=null;
			SecurityType secT=getSecurityInfo();
			if(secT==null){
				//pre 6.5.0 server
				srs=getResourcePropertiesDocument().getTargetSystemFactoryProperties().getXgroupArray();
			}
			else{
				return super.getXgroups();
			}
			return srs;
		}catch(Exception e){
			logger.error("Can't get xgroups.",e);
		}
		return null;
	}
	
	/**
	 * gets the TSSs that are accessible for this client
	 * 
	 * @return accessible TSS addresses
	 * @since 1.1.2
	 */
	public List<EndpointReferenceType> getAccessibleTargetSystems(){
		try{
			EndpointReferenceType[] eprs=getResourcePropertiesDocument().getTargetSystemFactoryProperties().
			getAccessibleTargetSystemReferenceArray();
			return Arrays.asList(eprs);
		}catch(Exception e){
			logger.error("Can't get accessible target system list.",e);
		}
		return null;
	}

}
