package de.fzj.unicore.uas.client;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Iterator;
import java.util.List;

import org.apache.log4j.Logger;
import org.oasisOpen.docs.wsrf.rl2.TerminationTimeDocument.TerminationTime;
import org.unigrids.x2006.x04.services.smf.AccessibleStorageReferenceDocument;
import org.unigrids.x2006.x04.services.smf.CreateSMSDocument;
import org.unigrids.x2006.x04.services.smf.CreateSMSResponseDocument;
import org.unigrids.x2006.x04.services.smf.StorageDescriptionType;
import org.unigrids.x2006.x04.services.smf.StorageFactoryPropertiesDocument;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.StorageFactory;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import eu.unicore.util.Log;
import eu.unicore.util.httpclient.IClientConfiguration;

/**
 * Client for the StorageFactory service
 * 
 * @author schuller
 * @since 6.2.3
 */
public class StorageFactoryClient extends BaseUASClient {

	private static final Logger logger=Log.getLogger(Log.CLIENT,StorageFactoryClient.class);

	private final StorageFactory smf;
	
	/**
	 * @param endpointUrl - the URL to connect to
	 * @param epr - the EPR of the target service
	 * @param sec - the security settings to use
	 * @throws Exception
	 */
	public StorageFactoryClient(String endpointUrl, EndpointReferenceType epr, IClientConfiguration sec)throws Exception {
		super(endpointUrl, epr,sec);
		smf=makeProxy(StorageFactory.class);
		setRetryDefaults();
	}

	/**
	 * @param epr - the EPR of the target service
	 * @param sec - the security settings to use
	 * @throws Exception
	 */
	public StorageFactoryClient(EndpointReferenceType epr, IClientConfiguration sec)throws Exception {
		this(epr.getAddress().getStringValue(), epr,sec);
	}

	/**
	 * create an SMS instance as described by the given {@link CreateSMSDocument}
	 * @param in - describes the instance to be created
	 * @return a StorageClient for accessing the newly created SMS
	 * @throws Exception
	 */
	public StorageClient createSMS(CreateSMSDocument in) throws Exception {
		logger.info("Calling storage factory service at: "+getEPR().getAddress().getStringValue());
		CreateSMSResponseDocument res=smf.CreateSMS(in);
		EndpointReferenceType epr=res.getCreateSMSResponse().getSmsReference();
		return new StorageClient(epr.getAddress().getStringValue(),epr,getSecurityConfiguration());
	}

	/**
	 * create a SMS of the specified type with the supplied initial termination time
	 * 
	 * @param type - the backend storage type (can be <code>null</code> if default should be used
	 * @param name - the name of the storage
	 * @param initialTerminationTime - the termination time
	 * @return
	 * @throws BaseFault
	 */
	public StorageClient createSMS(String type, String name, Calendar initialTerminationTime) throws Exception {
		CreateSMSDocument in=CreateSMSDocument.Factory.newInstance();
		in.addNewCreateSMS();
		if(initialTerminationTime!=null){
			TerminationTime tt=TerminationTime.Factory.newInstance();
			tt.setCalendarValue(initialTerminationTime);
			in.getCreateSMS().setTerminationTime(tt);
		}
		StorageDescriptionType sd=in.getCreateSMS().addNewStorageDescription();
		if(type!=null)sd.setStorageBackendType(type);
		if(name!=null){
			sd.addNewFileSystem().setName(name);
		}
		return createSMS(in);
	}

	public StorageClient createSMS(Calendar initialTerminationTime) throws Exception {
		CreateSMSDocument in=CreateSMSDocument.Factory.newInstance();
		TerminationTime tt=TerminationTime.Factory.newInstance();
		tt.setCalendarValue(initialTerminationTime);
		in.addNewCreateSMS().setTerminationTime(tt);
		return createSMS(in);
	}
	
	public StorageClient createSMS() throws Exception {
		CreateSMSDocument in=CreateSMSDocument.Factory.newInstance();
		in.addNewCreateSMS();
		return createSMS(in);
	}

	/**
	 * returns the service's TargetSystemPropertiesDocument
	 * @param endpointUrl
	 * @param epr
	 */
	public StorageFactoryPropertiesDocument getResourcePropertiesDocument()throws Exception{
		return StorageFactoryPropertiesDocument.Factory.parse(GetResourcePropertyDocument().getGetResourcePropertyDocumentResponse().newInputStream());
	}

	public List<EndpointReferenceType> getStorages(){
		try{
			EndpointReferenceType[] eprs=getResourcePropertiesDocument().getStorageFactoryProperties().getStorageReferenceArray();
			return Arrays.asList(eprs);
		}catch(Exception e){
			logger.error("Can't get storage list.",e);
		}
		return null;
	}

	/**
	 * gets the SMSSs that are accessible for this client
	 * 
	 * @return accessible SMS addresses
	 */
	public List<EndpointReferenceType> getAccessibleStorages()throws Exception{
		EnumerationClient<AccessibleStorageReferenceDocument>c=getAccessibleSMSEnumeration();
		if(c==null)return getAccessibleSMSWithoutEnumeration();
		List<EndpointReferenceType>res=new ArrayList<EndpointReferenceType>();
		Iterator<AccessibleStorageReferenceDocument>iter=c.iterator();
		while(iter.hasNext()){
			res.add(iter.next().getAccessibleStorageReference());
		}
		return res;
	}

	private EnumerationClient<AccessibleStorageReferenceDocument>getAccessibleSMSEnumeration()throws Exception{
		EndpointReferenceType epr=getResourcePropertiesDocument().getStorageFactoryProperties().getAccessibleStorageEnumeration();
		if(epr!=null){
			EnumerationClient<AccessibleStorageReferenceDocument>c=new EnumerationClient<AccessibleStorageReferenceDocument>(epr,
					getSecurityConfiguration(),
					AccessibleStorageReferenceDocument.type.getDocumentElementName());
			return c;
		}
		return null;
	}
	
	private List<EndpointReferenceType> getAccessibleSMSWithoutEnumeration(){
		try{
			EndpointReferenceType[] eprs=getResourcePropertiesDocument().getStorageFactoryProperties().
			getAccessibleStorageReferenceArray();
			return Arrays.asList(eprs);
		}catch(Exception e){
			logger.error("Can't get accessible storage list.",e);
		}
		return null;
	}
	
	public StorageDescriptionType[] getBackendStorageDescription()throws Exception{
		return getResourcePropertiesDocument().getStorageFactoryProperties().getStorageDescriptionArray();
	}
	
}
