/*********************************************************************************
 * Copyright (c) 2006 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
 
package de.fzj.unicore.uas.client;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlException;
import org.ggf.schemas.byteio.x2005.x10.byteIo.TransferInformationType;
import org.ggf.schemas.byteio.x2005.x10.streamableAccess.SeekReadDocument;
import org.ggf.schemas.byteio.x2005.x10.streamableAccess.SeekReadDocument.SeekRead;
import org.ggf.schemas.byteio.x2005.x10.streamableAccess.SeekReadResponseDocument;
import org.ggf.schemas.byteio.x2005.x10.streamableAccess.SeekWriteDocument;
import org.ggf.schemas.byteio.x2005.x10.streamableAccess.SeekWriteDocument.SeekWrite;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.fts.ProgressListener;
import de.fzj.unicore.uas.fts.byteio.ByteIO;
import de.fzj.unicore.uas.fts.byteio.StreamableByteIO;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import eu.unicore.util.Log;
import eu.unicore.util.httpclient.IClientConfiguration;

/**
 * manage data transfer using streamable byteio
 * only the "simple" mechanism (inline Base64) is supported
 * 
 * @author schuller
 */
public class SByteIOClient extends ByteIOBaseClient{
	
	private static final Logger logger=Log.getLogger(Log.CLIENT,SByteIOClient.class);
	
	private final StreamableByteIO fts;
	
	/**
	 * @param url
	 * @param epr
	 * @param sec
	 */
	public SByteIOClient(String url, EndpointReferenceType epr, IClientConfiguration sec)throws Exception {
		super(url, epr, sec);
		fts=makeProxy(StreamableByteIO.class);
	}

	/**
	 * @param epr
	 * @param sec
	 */
	public SByteIOClient(EndpointReferenceType epr, IClientConfiguration sec)throws Exception {
		this(epr.getAddress().getStringValue(), epr, sec);
	}

	/**
	 * read a block from the current position
	 * 
	 * @param offset
	 * @param numBytes
	 * @return
	 */
	public byte[] seekRead(long offset,int numBytes)throws IOException{
		try{
			return doSeekRead(StreamableByteIO.SBYTIO_ORIGIN_CURRENT,offset,numBytes);
		}catch(Exception e){
			IOException ioe=new IOException("Error in seekRead");
			ioe.initCause(e);
			throw ioe;
		}
	}
	/**
	 * read a block from the beginning of the stream
	 * 
	 * @param offset
	 * @param numBytes
	 * @return
	 */
	public byte[] seekReadFromOrigin(long offset,int numBytes)throws IOException{
		try{
			return doSeekRead(StreamableByteIO.SBYTIO_ORIGIN_BEGINNING,offset,numBytes);
		}catch(Exception e){
			IOException ioe=new IOException("Error in seekReadFromOrigin");
			ioe.initCause(e);
			throw ioe;
		}
	}
	
	public void seekWrite(byte[] data,long offset, String origin)throws IOException{
		try{
			doWrite(data, offset, origin);
		}catch(Exception e){
			IOException ioe=new IOException("Error in seekWrite");
			ioe.initCause(e);
			throw ioe;
		}
	}
	
	public void append(byte[] data)throws IOException{
		try{
			doWrite(data,0,StreamableByteIO.SBYTIO_ORIGIN_END);
		}catch(IOException ioe){
			throw ioe;
		}catch(Exception e){
			IOException ioe=new IOException("Error in append");
			ioe.initCause(e);
			throw ioe;
		}
	}
	
	protected byte[] doSeekRead(String seekOrigin, long offset, int numBytes)
	throws IOException,BaseFault,XmlException{
		SeekReadDocument req=SeekReadDocument.Factory.newInstance();
		SeekRead r=req.addNewSeekRead();
		r.addNewTransferInformation().setTransferMechanism(StreamableByteIO.TRANSFER_SIMPLE);
		r.setNumBytes(numBytes);
		r.setOffset(offset);
		r.setSeekOrigin(seekOrigin);
		SeekReadResponseDocument res=fts.seekRead(req);
		byte[] data=ByteIO.decode(
				StreamableByteIO.TRANSFER_SIMPLE,
				res.getSeekReadResponse().getTransferInformation());
		return data;
	}
	
	protected void doWrite(byte[] data, long offset, String origin)throws IOException,BaseFault,XmlException{
		SeekWriteDocument req=SeekWriteDocument.Factory.newInstance();
		SeekWrite r=req.addNewSeekWrite();
		r.setOffset(offset);
		r.setSeekOrigin(origin);
		TransferInformationType ti=r.addNewTransferInformation();
		ti.set(ByteIO.encode(StreamableByteIO.TRANSFER_SIMPLE,data));
		ti.setTransferMechanism(StreamableByteIO.TRANSFER_SIMPLE);
		fts.seekWrite(req);
	}

	public void readAllData(OutputStream target )throws IOException{
		readAllData(target, DEFAULT_CHUNKSIZE);
	}
	
	/**
	 * convenience method that reads all data and writes it to an output stream
	 * @param os
	 * @þaram chunksize
	 */
	public void readAllData(OutputStream os, int chunksize)throws IOException{
		try{
			final long offset=0;
			while(true){
				byte[] data=seekRead(offset, chunksize);
				if(data==null || data.length==0)break;
				if(logger.isDebugEnabled()){
						logger.debug("Read "+data.length+" bytes.");
				}
				os.write(data);
				os.flush();
				if(progressListener!=null){
					try{
						progressListener.notifyProgress(Long.valueOf(data.length));
					}catch(Exception e){
						Log.logException("Error reporting progress.",e,logger);
					}
					if(progressListener.isCancelled())throw new ProgressListener.CancelledException("Cancelled.");
				}
			}
		}catch(ProgressListener.CancelledException ce){
			throw ce;
		}catch(Exception e){
			IOException ioe=new IOException("Error in readAllData");
			ioe.initCause(e);
			throw ioe;
		}
	}
	
	/**
	 * writes all data from <code>source</code> to the remote file,
	 * using the current chunksize
	 * @param source
	 * @throws Exception
	 */
	public void writeAllData(InputStream source)throws Exception{
		writeAllData(source, chunksize);
	}

	/**
	 * writes all data from <code>source</code> to the remote file,
	 * using the specified <code>chunksize</code>
	 * @param source
	 * @throws Exception
	 */
	public void writeAllData(InputStream source, int chunksize)throws Exception{
		byte[] data=new byte[chunksize];
		int l;
		long total=0;
		boolean doAppend=append;
		while(true){
			l=source.read(data);
			if(l==-1){
				//write an empty chunk to make sure the remote file is created
				if(total==0)doWrite(new byte[0],doAppend);
				break;
			}
			if(l<data.length){
				byte[]read=new byte[l];
				System.arraycopy(data, 0, read, 0, l);
				doWrite(read,doAppend);
				
			}else doWrite(data,doAppend);
			total+=l;
			doAppend=true;
			if(progressListener!=null){
				try{
					progressListener.notifyProgress(Long.valueOf(l));
				}catch(Exception e){
					Log.logException("Error reporting progress.",e,logger);
				}
				if(progressListener.isCancelled())throw new ProgressListener.CancelledException("Cancelled.");
			}
		}
	}
	
	private void doWrite(byte[] data, boolean append)throws Exception{
		if(append){
			append(data);
		}
		else {
			doWrite(data, 0, StreamableByteIO.SBYTIO_ORIGIN_BEGINNING);
		}
	}

}
