/*********************************************************************************
 * Copyright (c) 2006 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
 
package de.fzj.unicore.uas.client;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigInteger;

import org.apache.log4j.Logger;
import org.ggf.schemas.byteio.x2005.x10.byteIo.TransferInformationType;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.AppendDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.AppendDocument.Append;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.AppendResponseDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.ReadDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.ReadDocument.Read;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.ReadResponseDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.TruncAppendDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.TruncAppendDocument.TruncAppend;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.WriteDocument;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.WriteDocument.Write;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.WriteResponseDocument;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.fts.ProgressListener;
import de.fzj.unicore.uas.fts.byteio.ByteIO;
import de.fzj.unicore.uas.fts.byteio.RandomByteIO;
import de.fzj.unicore.wsrflite.xfire.MTOMInHandler;
import eu.unicore.util.Log;
import eu.unicore.util.httpclient.IClientConfiguration;

/**
 * manage data transfer using random byteio
 * only the "simple" mechanism (inline Base64) is supported
 * 
 * @author schuller
 */
public class RByteIOClient extends ByteIOBaseClient{
	
	private static final Logger logger=Log.getLogger(Log.CLIENT,RByteIOClient.class);
	
	private final RandomByteIO fts;

	/**
	 * @param url
	 * @param epr
	 * @param sec
	 */
	public RByteIOClient(String url, EndpointReferenceType epr, IClientConfiguration sec)throws Exception {
		super(url, epr, sec);
		fts=makeProxy(RandomByteIO.class);
	}

	/**
	 * @param epr
	 * @param sec
	 */
	public RByteIOClient(EndpointReferenceType epr, IClientConfiguration sec)throws Exception {
		this(epr.getAddress().getStringValue(), epr, sec);
	}

	public byte[] read(long offset, long numBlocks, long bytesPerBlock, long stride) throws Exception{
		ReadDocument req=ReadDocument.Factory.newInstance();
		Read r=req.addNewRead();
		r.setNumBlocks(numBlocks);
		r.setBytesPerBlock(bytesPerBlock);
		r.setStartOffset(BigInteger.valueOf(offset));
		r.setStride(stride);
		String mechanism=isMTOMEnabled()?RandomByteIO.TRANSFER_MTOM:RandomByteIO.TRANSFER_SIMPLE;
		r.addNewTransferInformation().setTransferMechanism(mechanism);
		ReadResponseDocument res=fts.read(req);
		byte[] data=null;
		if(isMTOMEnabled()){
			data=MTOMInHandler.getAttachment();
		}
		else{
			data=ByteIO.decode(
				RandomByteIO.TRANSFER_SIMPLE,
				res.getReadResponse().getTransferInformation());
		}
		return data;
	}
	
	/**
	 * write a single block of data
	 * 
	 * @param data
	 * @throws Exception
	 */
	public void write(byte[] data)throws Exception{
		write(data,0,1,data.length,0);
	}
	
	/**
	 * write a block of data, beginning at the given offset, and truncating
	 * the remainder of the file
	 * 
	 * @param offset - the offset at which to write the data
	 * @param data - the data to write
	 * @throws Exception
	 */
	public void truncAppend(long offset, byte[]data)throws Exception{
		TruncAppendDocument req=TruncAppendDocument.Factory.newInstance();
		TruncAppend ta=req.addNewTruncAppend();
		ta.setOffset(BigInteger.valueOf(offset));
		TransferInformationType ti=ta.addNewTransferInformation();
		String mechanism=isMTOMEnabled()?RandomByteIO.TRANSFER_MTOM:RandomByteIO.TRANSFER_SIMPLE;
		ti.set(ByteIO.encode(mechanism,data));
		ti.setTransferMechanism(mechanism);
		fts.truncAppend(req);
	}
	
	/**
	 * write data
	 * 
	 * @param data - the data to write
	 * @param offset - the initial offset
	 * @param numBlocks - the number of blocks
	 * @param bytesPerBlock - the number of bytes in a block
	 * @param stride - how many bytes to skip after writing a block
	 * @return
	 * @throws Exception
	 */
	public WriteResponseDocument write(byte[] data, long offset, long numBlocks, long bytesPerBlock, long stride) throws Exception{
		WriteDocument req=WriteDocument.Factory.newInstance();
		Write r=req.addNewWrite();
		TransferInformationType ti=r.addNewTransferInformation();
		String mechanism=isMTOMEnabled()?RandomByteIO.TRANSFER_MTOM:RandomByteIO.TRANSFER_SIMPLE;
		ti.set(ByteIO.encode(mechanism,data));
		ti.setTransferMechanism(mechanism);
		r.setBytesPerBlock(bytesPerBlock);
		r.setStartOffset(BigInteger.valueOf(offset));
		r.setStride(stride);
		return fts.write(req);
	}
	
	public AppendResponseDocument append(byte[] data) throws Exception{
		AppendDocument req=AppendDocument.Factory.newInstance();
		Append r=req.addNewAppend();
		TransferInformationType ti=r.addNewTransferInformation();
		String mechanism=isMTOMEnabled()?RandomByteIO.TRANSFER_MTOM:RandomByteIO.TRANSFER_SIMPLE;
		ti.set(ByteIO.encode(mechanism,data));
		ti.setTransferMechanism(mechanism);
		return fts.append(req);
	}
	
	/**
	 * convenience method that reads all data and writes it to an output stream
	 * @param os
	 */
	public void readAllData(OutputStream os)throws IOException{
		readAllData(os, chunksize);
	}
	
	/**
	 * convenience method that reads all data and writes it to an output stream
	 * @param os
	 * @þaram chunksize
	 */
	public void readAllData(OutputStream os, int chunksize)throws IOException{
		try{
			long offset=0;
			while(true){
				byte[] data=read(offset,1,chunksize,0);
				if(data==null || data.length==0)break;
				offset+=data.length;
				if(progressListener!=null){
					try{
						progressListener.notifyProgress(Long.valueOf(data.length));
					}catch(Exception e){
						logger.warn("Error reporting progress.",e);
					}
					if(progressListener.isCancelled())throw new ProgressListener.CancelledException("Cancelled.");
					
				}
				if(logger.isDebugEnabled()){
						logger.debug("Read "+data.length+" bytes.");
				}
				os.write(data);
				os.flush();
			}
		}catch(ProgressListener.CancelledException ce){
			throw ce;
		}catch(Exception e){
			IOException ioe=new IOException("IO error: "+e.getMessage());
			ioe.initCause(e);
			throw ioe;
		}
	}
	
	/**
	 * writes all data from <code>source</code> to the remote file,
	 * using the current chunksize
	 * @param source
	 * @throws Exception
	 */
	public void writeAllData(InputStream source)throws Exception{
		writeAllData(source, chunksize);
	}

	/**
	 * writes all data from <code>source</code> to the remote file,
	 * using the specified <code>chunksize</code>
	 * @param source
	 * @throws Exception
	 */
	public void writeAllData(InputStream source, int chunksize)throws Exception{
		byte[] data=new byte[chunksize];
		int l;
		long total=0;
		//the first block is appended only if we are in append mode...
		boolean doAppend=append;
		while(true){
			l=source.read(data);
			if(l==-1){
				//write an empty chunk to make sure the remote file is created
				if(total==0){
					doWrite(new byte[0],doAppend);
				}
				break;
			}
			if(l<data.length){
				byte[]read=new byte[l];
				System.arraycopy(data, 0, read, 0, l);
				{
					doWrite(read,doAppend);
				}
			}
			else doWrite(data,doAppend);
			total+=l;
			//always append after writing the first block
			doAppend=true;
			if(progressListener!=null){
				try{
					progressListener.notifyProgress(Long.valueOf(l));
				}catch(Exception e){
					Log.logException("Error reporting progress.",e,logger);
				}
				if(progressListener.isCancelled())throw new ProgressListener.CancelledException("Cancelled.");
			}
		}
	}

	private void doWrite(byte[] data, boolean append)throws Exception{
		if(append){
			append(data);
		}
		else truncAppend(0,data);
	}

}
