/*********************************************************************************
 * Copyright (c) 2006 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
 
package de.fzj.unicore.uas.client;

import org.apache.log4j.Logger;
import org.ggf.schemas.byteio.x2005.x10.randomAccess.SizeDocument;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.fts.FiletransferOptions;
import de.fzj.unicore.uas.fts.ProgressListener;
import de.fzj.unicore.uas.fts.byteio.RandomByteIO;
import eu.unicore.util.Log;
import eu.unicore.util.httpclient.IClientConfiguration;

/**
 * base class for the byteio clients
 * 
 * @author schuller
 */
public abstract class ByteIOBaseClient extends FileTransferClient 
implements FiletransferOptions.IChunked, FiletransferOptions.IMonitorable {
	
	private static final Logger logger=Log.getLogger(Log.CLIENT,JobClient.class);
	
	public static final int DEFAULT_CHUNKSIZE=128000;
	
	public static final int DEFAULT_CHUNKSIZE_MTOM=20*DEFAULT_CHUNKSIZE;
	
	protected int chunksize=DEFAULT_CHUNKSIZE;
	
	protected ProgressListener<Long>progressListener;
	
	//whether the MTOM transfer mechanism should be used (if available)
	protected boolean wantMTOM=true;
	
	/**
	 * @param url
	 * @param epr
	 * @param sec
	 */
	public ByteIOBaseClient(String url, EndpointReferenceType epr, IClientConfiguration sec)throws Exception {
		super(url, epr, sec);
	}
	
	/**
	 * @param epr
	 * @param sec
	 */
	public ByteIOBaseClient(EndpointReferenceType epr, IClientConfiguration sec)throws Exception {
		super(epr.getAddress().getStringValue(),epr, sec);
	}

	/**
	 * get the size of the (export) file
	 * 
	 * @return
	 */
	public long getSize(){
		try{
			SizeDocument sd=SizeDocument.Factory.parse(getResourceProperty(RandomByteIO.RPSize));
			return sd.getSize().longValue();
		}catch(Exception e){
			Log.logException("Error getting file size.",e,logger);
			return -1;
		}
	}

	public void setChunksize(int chunksize){
		this.chunksize=chunksize;
	}
	
	public void setProgressListener(ProgressListener<Long> listener) {
		this.progressListener=listener;
	}
	
	private Boolean mtomAvailable=null;
	//checks if the server side supports MTOM
	public boolean supportsMTOM()throws Exception{
		if(mtomAvailable==null){
			mtomAvailable=getResourceProperty(RandomByteIO.RPTransferMechanisms).contains(RandomByteIO.TRANSFER_MTOM);
			if(mtomAvailable && chunksize==DEFAULT_CHUNKSIZE){
				chunksize=DEFAULT_CHUNKSIZE_MTOM;
			}
		}
		return mtomAvailable.booleanValue();
	}
	
	/**
	 * whether MTOM should be used, if available
	 * @param useMTOM
	 */
	public void setWantMTOM(boolean wantMTOM){
		this.wantMTOM=wantMTOM;
	}
	
	/**
	 * whether MTOM should be used
	 * @return <code>true</code> if the server supports MTOM and the user wants to use it
	 * @throws Exception on remoting problems 
	 */
	protected boolean isMTOMEnabled()throws Exception{
		return wantMTOM && supportsMTOM();
	}
}
